﻿///////////////////////////////////////////////////////////////////////////////
// This file is part of SharpWebBrowser.
///////////////////////////////////////////////////////////////////////////////
// Copyright (C) 2009-2011 bis5 <bis5@users.sourceforge.jp>
// All rights reserved.
///////////////////////////////////////////////////////////////////////////////
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the Microsoft Public Licence (Ms-PL) as published by 
// Microsoft Corp.
//
// You should have received a copy of the Microsoft Public License along 
// with this program. 
// If not, see <http://www.microsoft.com/opensource/licenses.mspx>
//
///////////////////////////////////////////////////////////////////////////////
//
// File information
// Name: MainForm.cs, MainForm.Designer.cs
// Author: bis5
//
///////////////////////////////////////////////////////////////////////////////

using System;
using System.Drawing;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Linq;
using Bis5Products.SharpWebBrowser.Dialogs;

namespace Bis5Products.SharpWebBrowser {
    public partial class MainForm : Form, IMainControl {
        //Const
        internal const string strName = "SharpWebBrowser";
        private const string strWU = "http://update.microsoft.com";
        private const string ab = "about:blank";
        private string strWinVer;
        private readonly string strLoc = Application.ExecutablePath;
        private int intPMem;
        private int intVmem;
        internal string query;
        private readonly string strInetCpl = " /d " + Environment.GetEnvironmentVariable("windir") +
            "\\System32\\shell32.dll,Control_RunDLL INETCPL.CPL";

        //Object Refer(s)
        private Microsoft.VisualBasic.Devices.ComputerInfo pcInfo = new Microsoft.VisualBasic.Devices.ComputerInfo();
        private WebBrowserController wc;
        private BookmarkManager bm;
        private ConfigurationManager cm;

        //Forms
        VerInfoDialog frmVer;
        internal PluginInfo frmPlgInfo;

        internal LanguageManager lnsel;

        //Language
        private string strCloseConfirm;
        private string strNoItem;
        private string strError;
        private string strAddTab;
        private string strRemoveTab;
        private string strNoTitlePage;
        private string strOpenFileDialog;
        private string strPropGeneric;
        private string strOpenBookmarkDropDown;
        private string strEditBookmarkDropDown;
        private string strRemoveBookmarkDropDown;
        private string editFolderText;
        private string removeFolderText;
        private string createFolderText;
        private string strAddBookmarkToolStripMenuItemText;
        private string strRemoveConfirm;
        private string strFolderRemoveConfirm;

        string[] cLineArgs;

        /// <summary>
        /// <see cref="Bis5Products.SharpWebBrowser.MainForm" />の新しいインスタンスを開始します。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        public MainForm(string[] args) {
            strWinVer = pcInfo.OSFullName + "  " + pcInfo.OSVersion;
            intPMem = (int)(pcInfo.TotalPhysicalMemory / (1024 ^ 2));
            intVmem = (int)(pcInfo.TotalVirtualMemory / (1024 ^ 2));

            InitializeComponent();
            cLineArgs = args;
        }

        /// <summary>
        /// <see cref="LanguageSelecter"/>から言語定数を読み込みます。
        /// </summary>
        private void SetText() {
            strCloseConfirm = lnsel.GetValue("strCloseConfirm");
            strNoItem = lnsel.GetValue("strNoItem");
            strError = lnsel.GetValue("strError");
            strAddTab = lnsel.GetValue("strAddTab");
            strRemoveTab = lnsel.GetValue("strRemoveTab");
            strNoTitlePage = lnsel.GetValue("strNoTitlePage");
            this.Text = lnsel.GetValue("MainForm.Text");
            this.menuBrowserToolStripMenuItem.Text = lnsel.GetValue("MenuBrowser.Text");
            this.menuBookmarkToolStripMenuItem.Text = lnsel.GetValue("MenuBookmark.Text");
            this.menuToolToolStripMenuItem.Text = lnsel.GetValue("MenuTool.Text");
            this.menuHelpToolStripMenuItem.Text = lnsel.GetValue("MenuHelp.Text");
            this.labelURL.Text = lnsel.GetValue("LabelURL.Text");
            this.butNavi.Text = lnsel.GetValue("butNavi.Text");
            this.menuPrintPrwviewToolStripMenuItem.Text = lnsel.GetValue("MenuPrintPreview.Text");
            this.menuPrintToolStripMenuItem.Text = lnsel.GetValue("MenuPrint.Text");
            this.menuWUToolStripMenuItem.Text = lnsel.GetValue("MenuWU.Text");
            this.menuExit2ToolStripMenuItem.Text = lnsel.GetValue("MenuExit2.Text");
            strAddBookmarkToolStripMenuItemText = lnsel.GetValue("MenuBookmarkAdd.Text");
            this.menuVerInfoToolStripMenuItem.Text = lnsel.GetValue("MenuVerInfo.Text");
            this.menuNewTabToolStripMenuItem.Text = lnsel.GetValue("menuNewTabToolStripMenuItem.Text");
            this.menuOpenToolStripMenuItem.Text = lnsel.GetValue("menuOpenToolStripMenuItem.Text");
            this.menuCloseTabToolStripMenuItem.Text = lnsel.GetValue("menuCloseTabToolStripMenuItem.Text");
            this.menuSaveAsToolStripMenuItem.Text = lnsel.GetValue("menuSaveAsToolStripMenuItem.Text");
            this.menuPageSetupToolStripMenuItem.Text = lnsel.GetValue("menuPageSetupToolStripMenuItem.Text");
            strOpenFileDialog = lnsel.GetValue("strOpenFileDialog");
            this.menuPlgInfoToolStripMenuItem.Text = lnsel.GetValue("Core_PluginInfo_Text");
            strPropGeneric = lnsel.GetValue("strPropGeneric");
            strOpenBookmarkDropDown = lnsel.GetValue("strOpenBookmarkDropDown");
            strEditBookmarkDropDown = lnsel.GetValue("strEditBookmarkDropDown");
            strRemoveBookmarkDropDown = lnsel.GetValue("strRemoveBookmarkDropDown");
            editFolderText = lnsel.GetValue("editFolderText");
            removeFolderText = lnsel.GetValue("removeFolderText");
            createFolderText = lnsel.GetValue("createFolderText");
            strRemoveConfirm = lnsel.GetValue("strRemoveConfirm");
            strFolderRemoveConfirm = lnsel.GetValue("strFolderRemoveConfirm");
            this.btReload.Text = lnsel.GetValue("btReload.Text");
            this.btGoHome.Text = lnsel.GetValue("btGoHome.Text");
            this.btStop.Text = lnsel.GetValue("btStop.Text");
            this.btPrint.Text = lnsel.GetValue("btPrint.Text");
            this.menuPrintPreview.Text = lnsel.GetValue("menuPrintPreview.Text");
            this.menuPageConfig.Text = lnsel.GetValue("menuPageConfig.Text");
            this.btAddBookmark.Text = lnsel.GetValue("btAddBookmark.Text");
            this.btSearch.Text = lnsel.GetValue("btSearch.Text");
            this.btGoBack.Text = lnsel.GetValue("btGoBack.Text");
            this.btGoForward.Text = lnsel.GetValue("btGoForward.Text");
            this.menuInetOptionToolStripMenuItem.Text = lnsel.GetValue("menuInetOption.Text");
            this.menuConfigToolStripMenuItem.Text = lnsel.GetValue("menuConfig.Text");
        }

        /// <summary>
        /// 設定値を作成するために、初回起動時に一回だけ実行する。
        /// </summary>
        private void DoConfigInitialize() {
            cm.Add("WindowState", this.WindowState.ToString());
            cm.Add("WindowWidth", this.Size.Width);
            cm.Add("WindowHeight", this.Size.Height);
            cm.Add("Language", "Japanese");
            cm.Add("LocationX", 0);
            cm.Add("LocationY", 0);
            cm.Add("NewTabOpenMode", TabOpenMode.hp.ToString());
            cm.Add("HomePage", "http://www.google.com");
            cm.Add("EnableSearchProvider", "http://www.google.com/search?hl=ja&q={0}");
            cm.Set("IsFirstRun", false);
        }

        /// <summary>
        /// 設定を保存します。
        /// </summary>
        private void SaveSettings() {
            cm.Set("WindowState", this.WindowState);
            if (this.WindowState == FormWindowState.Normal) {
                cm.Set("WindowWidth", this.Size.Width);
                cm.Set("WindowHeight", this.Size.Height);
                cm.Set("LocationX", this.Location.X);
                cm.Set("LocationY", this.Location.Y);
            }
            cm.Save();

            bm.Save();
        }

        private void MainForm_Load(object sender, EventArgs e) {
            this.Visible = false;
            /*Start Initialize*/
            frmVer = new Dialogs.VerInfoDialog(new DateTime(2010, 6, 15, 21, 30, 00, DateTimeKind.Local));

            cm = new ConfigurationManager(Application.StartupPath);
            Program.OnSaveSettings += new Program.SaveSettings(cm.Save);

            if ((bool)cm.GetBool("IsFirstRun")) {
                DoConfigInitialize();
            } else {
                //Settings Load Section
                this.Size = new Size(cm.GetInt32("WindowWidth"), cm.GetInt32("WindowHeight"));
                this.Location = new Point(cm.GetInt32("LocationX"), cm.GetInt32("LocationY"));
                if ((FormWindowState)cm.GetObj("WindowState", typeof(FormWindowState)) != FormWindowState.Minimized) {
                    this.WindowState = (FormWindowState)cm.GetObj("WindowState", typeof(FormWindowState));
                }
            }
            if (cm.IsContains("EnableSearchProvider"))
                query = cm.GetStr("EnableSearchProvider");
            else//ファイル引き継ぎなどで設定項目が存在しない場合の処理
                query = "http://www.google.com/search?hl=ja&q={0}";

            //Initialize LanguageManager
            lnsel = new LanguageManager(Application.StartupPath, cm);
            lnsel.AppendDelegate(new LmgrAppender(SetText));
            SetText();

            cm.SetLangMgr((ILangMgr)lnsel);

            //Module Initialise
            wc = new WebBrowserController((ILangMgr)lnsel);
            wc.RegisterMainForm(this, tbURL, tabControl1, statusLabel, toolStripProgressBar1);
            bm = new BookmarkManager(Application.StartupPath, lnsel, this);
            Program.OnSaveBookmark += new Program.SaveBookmark(bm.Save);

            bm.BookmarkItemChanged += new BookmarKItemChangedEventHandler(LoadBookmarkItems);

            //Dialogs Initialize
            frmPlgInfo = new PluginInfo(lnsel);

            //Set Version Info
            frmPlgInfo.AddVersionInfo(strName + " Core", Application.ProductVersion, "bis5",
                System.Reflection.Assembly.GetExecutingAssembly().GetName().Name + ".exe");
            string[] modver = cm.GetVersion();
            frmPlgInfo.AddVersionInfo(modver[0], modver[1], modver[2], modver[3]);
            modver = lnsel.GetVersion();
            frmPlgInfo.AddVersionInfo(modver[0], modver[1], modver[2], modver[3]);
            modver = bm.GetVersion();
            frmPlgInfo.AddVersionInfo(modver[0], modver[1], modver[2], modver[3]);
            modver = wc.GetVersion();
            frmPlgInfo.AddVersionInfo(modver[0], modver[1], modver[2], modver[3]);


            LoadBookmarkItems();


            cm.AddPropertySheet((ISWBPropertySheet)new PropertySheet(cm, lnsel, this));
            cm.AddPropertySheet((ISWBPropertySheet)new BehaveProp(cm, lnsel));
            lnsel.RegistProperty2CMgr();
#if(DEBUG)
            //cm.AddPropertySheet((ISWBPropertySheet)new Core_DebugProperty(this));
#endif
            //Create tab page (includes web browser control)
            AddTab(cm.GetStr("HomePage"));

            //customize version info dialog
            frmVer.textBoxDescription.Text += "\r\nFile Location : " + strLoc;
            frmVer.textBoxDescription.Text += "\r\nWindows Version : " + strWinVer;
            frmVer.textBoxDescription.Text += "\r\nTotal Physical Memory : " + intPMem.ToString() + "KB";
            frmVer.textBoxDescription.Text += "\r\nTotal Virtual Memory : " + intVmem.ToString() + "KB";

            try {
                if (cLineArgs[0] != string.Empty)
                    WB_Navigate(cLineArgs[0]);
            } catch { }

            //create tab control's context menu
            MenuItem menuItem1 = new MenuItem(strAddTab, new EventHandler(AddTab));
            MenuItem menuItem2 = new MenuItem(strRemoveTab, new EventHandler(RemoveTab));
            MenuItem[] menuItems = new MenuItem[2] { menuItem1, menuItem2 };

            this.tabControl1.ContextMenu = new ContextMenu(menuItems);

            /*Finish Initialize*/
            this.Visible = true;
            this.SuspendLayout();
            this.SizeChanged += new EventHandler(MainForm_Resize);
            WindowStateChanged += new EventHandler(MainForm_WindowStateChanged);
        }

        private EventHandler WindowStateChanged;

        public FormWindowState WindowState2 {
            get {
                return this.WindowState;
            }
            set {
                this.WindowState = value;
                WindowStateChanged(this, new EventArgs());
            }
        }

        private void MainForm_WindowStateChanged(object sender, EventArgs e) {
            MainForm_Resize(sender, e);
        }

        void MainForm_Resize(object sender, EventArgs e) {
            this.ResumeLayout(true);
            this.SuspendLayout();
        }

        private void menuExitToolStripMenuItem_Click(object sender, EventArgs e) {
            var ex = new FormClosingEventArgs(CloseReason.ApplicationExitCall, false);
            SWB_Exit(sender, ex);
        }

        /// <summary>
        /// SharpWebBrowserを終了します。
        /// </summary>
        /// <param name="sender">呼び出し元のオブジェクト</param>
        /// <param name="e">終了しようとしていることに関するデータを格納した<paramref name="System.Windows.Forms.FormClosingEventArgs"/>。</param>
        private void SWB_Exit(object sender, FormClosingEventArgs e) {
            if (Program.isCalledThreadException) {
                this.Hide();
                return;
            }
            if (MessageBox.Show(strCloseConfirm, strName, MessageBoxButtons.OKCancel) == DialogResult.OK) {
                this.Visible = false;

                if (tabControl1.SelectedTab != null) {
#if !DEBUG
                    try {
#endif
                    wc.Dispose();
#if !DEBUG
                    } catch (System.Runtime.InteropServices.COMException) {
                        Console.WriteLine("WebBrowserController could NOT handle COMException!");
                    }
#endif
                    tabControl1.Dispose();
                    SaveSettings();
                    this.Dispose();
                    Application.Exit();
                } else { //impossible?
                    this.Dispose();
                    Application.Exit();
                }
            } else { e.Cancel = true; }
        }

        bool isAddingTab = false;
        /// <summary>
        /// 新しいタブを開きます。
        /// </summary>
        /// <param name="sender">呼び出し元のオブジェクト</param>
        /// <param name="e">イベント発生時に生成された<paramref name="System.EventArgs"/>。</param>
        private void AddTab(object sender, System.EventArgs e) {
            isAddingTab = true;
            TabPage NewTabPage = new TabPage();
            WebBrowser_NewWindowHacked NewWebBrowserControl = wc.CreateControl();
            NewTabPage.Controls.Add(NewWebBrowserControl);
            NewWebBrowserControl.RegisterAsBrowser = true;
            NewWebBrowserControl.NewWindow2 += new WebBrowserNewWindow2EventHandler(webBrowser_NewWindow2);

            tabControl1.Controls.Add(NewTabPage);
            tabControl1.SelectTab(tabControl1.Controls.Count - 1);

            //Navigate to HomePage
            if (this.Visible) {
                switch ((TabOpenMode)cm.GetObj("NewTabOpenMode", typeof(TabOpenMode))) {
                    case TabOpenMode.blank:
                        NewWebBrowserControl.Navigate("about:blank");
                        break;
                    case TabOpenMode.hp:
                        NewWebBrowserControl.Navigate(cm.GetStr("HomePage"));
                        break;
                    case TabOpenMode.any:
                        NewWebBrowserControl.Navigate(cm.GetStr("AnyPageInNewTab"));
                        break;
                }
            } else
                NewWebBrowserControl.Navigate(cm.GetStr("HomePage"));
            isAddingTab = false;
        }

        /// <summary>
        /// 新しいタブを追加します。
        /// </summary>
        /// <param name="uri">新しいタブで開くページのURL。</param>
        public void AddTab(string uri) {
            isAddingTab = true;
            TabPage NewTabPage = new TabPage();
            WebBrowser_NewWindowHacked NewWebBrowserControl = wc.CreateControl();
            NewTabPage.Controls.Add(NewWebBrowserControl);
            NewWebBrowserControl.RegisterAsBrowser = true;
            NewWebBrowserControl.NewWindow2 += new WebBrowserNewWindow2EventHandler(webBrowser_NewWindow2);

            tabControl1.Controls.Add(NewTabPage);
            tabControl1.SelectTab(tabControl1.Controls.Count - 1);

            NewWebBrowserControl.Navigate(uri);
            isAddingTab = false;
        }

        private void AddTab() {
            isAddingTab = true;
            TabPage newTabPage = new TabPage();
            WebBrowser_NewWindowHacked newBrowserCtrl = wc.CreateControl();
            newTabPage.Controls.Add(newBrowserCtrl);
            newBrowserCtrl.RegisterAsBrowser = true;
            newBrowserCtrl.NewWindow2 += new WebBrowserNewWindow2EventHandler(webBrowser_NewWindow2);

            newBrowserCtrl.Navigate("about:blank");

            tabControl1.TabPages.Add(newTabPage);
            tabControl1.SelectTab(tabControl1.TabPages.Count - 1);
            isAddingTab = false;
        }

        /// <summary>
        /// タブを削除する
        /// </summary>
        /// <param name="sender">呼び出し元のオブジェクト</param>
        /// <param name="e">イベント発生時に生成された<paramref name="System.EventArgs"/>。</param>
        private void RemoveTab(object sender, EventArgs e) {
            if (tabControl1.SelectedTab.Controls.Count != 0)
                wc.RemoveControl((WebBrowser_NewWindowHacked)tabControl1.SelectedTab.Controls[0]);
            tabControl1.TabPages.Remove(tabControl1.SelectedTab);
        }

        void tabControl1_SelectedIndexChanged(object sender, System.EventArgs e) {
            AutoTabRemove();

            statusLabel.Text = string.Empty;
            toolStripProgressBar1.Visible = false;

            if (tabControl1.SelectedTab == null) { AddTab(sender, e); } else
                foreach (WebBrowser_NewWindowHacked wb in tabControl1.TabPages[tabControl1.SelectedIndex].Controls) {
                    statusLabel.Text = wb.StatusText;

                    if (wb.DocumentTitle == "") { this.Text = strName; } else { this.Text = wb.DocumentTitle + " - " + strName; }

                    try { tbURL.Text = wb.Url.ToString(); } catch { tbURL.Text = string.Empty; }

                    //REMIND: WebBrowserのプロパティを見ているので
                    //        出来るならWebBrowserControllerに処理させたい
                    if (wb.EncryptionLevel == WebBrowserEncryptionLevel.Insecure ||
                        wb.EncryptionLevel == WebBrowserEncryptionLevel.Unknown)
                        tbURL.BackColor = Color.FromKnownColor(KnownColor.Window);
                    else
                        tbURL.BackColor = Color.Khaki;
                }
        }

        private void tbURL_KeyPress(object sender, KeyPressEventArgs e) {
            if (e.KeyChar == (char)(Keys.Return)) {
                WB_Navigate(tbURL.Text);
            }
        }

        //TOFIX: webBrowser_Navigatedなど、ほかのイベントもトリガーにしてやれば、検出率は向上するだろう。
        //ただし、開いているタブが多ければ多いほど、処理が重くなるし、トリガーを増やしすぎても重くなるので、
        //どのイベントをトリガーにするかは重要なポイントになるだろう。
        bool isRemovingTab = false;
        /// <summary>
        /// コントロールのみが廃棄されてしまったタブページを閉じます。
        /// </summary>
        private void AutoTabRemove() {
            if (!isAddingTab && !isRemovingTab && !isTabClosing) {
                foreach (TabPage tp in tabControl1.TabPages) {
                    try {
                        var url = ((WebBrowser_NewWindowHacked)tp.Controls[0]).Url.OriginalString;
                    } catch {
                        isRemovingTab = true;
                        tabControl1.TabPages.Remove(tp);
                    }
                }
                isRemovingTab = false;
            }
        }

        /// <summary>
        /// アクティブなタブでURIへナビゲートを行う。正常に処理が実行できなかった場合、設定されたエンジンでの検索に切り替える。
        /// </summary>
        /// <param name="uri">URI</param>
        public void WB_Navigate(string uri) {
            if (tabControl1.SelectedTab == null)
                AddTab(null, new EventArgs());
            foreach (WebBrowser_NewWindowHacked wb in tabControl1.SelectedTab.Controls) {
                if ((System.Text.RegularExpressions.Regex.IsMatch(uri, "[.]+") && !System.Text.RegularExpressions.Regex.IsMatch(uri, "[.]$")) |
                    System.Text.RegularExpressions.Regex.IsMatch(uri, "^(about:)") | System.Text.RegularExpressions.Regex.IsMatch(uri, "^http://") |
                    System.Text.RegularExpressions.Regex.IsMatch(uri, "^localhost") || System.Text.RegularExpressions.Regex.IsMatch(uri, "^https://")) {
                    wb.Navigate(uri);
                } else {
                    uri = System.Text.RegularExpressions.Regex.Replace(uri, "#", "%23");
                    wb.Navigate(String.Format(query, uri));
                }
                wb.Focus();
            }
        }

        private void butNavi_Click(object sender, EventArgs e) {
            WB_Navigate(tbURL.Text);
        }

        private void LoadBookmarkItems() {
            menuBookmarkToolStripMenuItem.DropDownItems.Clear();
            menuBookmarkToolStripMenuItem.DropDownItems.AddRange(bm.GetMenu(menuStrip1));

        }

        void webBrowser_NewWindow2(object sender, WebBrowserNewWindow2EventArgs e) {
            //AddTab(sender, new EventArgs());
            AddTab();
            WebBrowser_NewWindowHacked newBrowserControl = null;//要割り当て
            foreach (WebBrowser_NewWindowHacked c in tabControl1.TabPages[tabControl1.TabPages.Count - 1].Controls) {
                newBrowserControl = c;
            }
            e.ppDisp = newBrowserControl.Application;
            newBrowserControl.RegisterAsBrowser = true;
            /*  実装例
                // 新しくウィンドウを開かれようとするときに発生する
                void webBrowser_NewWindow2(object sender, WebBrowserNewWindow2EventArgs e)
                {
                    // 新しい WebBrowser の初期化
                    var newBrowser = new WebBrowserEx();
                    newBrowser.Dock = DockStyle.Fill;

                    // 新しい WebBrowser のコンテナ（下記はタブの場合）
                    //var tabPage = new TabPage();
                    //tabPage.Controls.Add(newBrowser);
                    //tabControl1.TabPages.Add(tabPage);

                    // 新しい WebBrowser に表示させる設定
                    e.ppDisp = newBrowser.AxApplication;
                    newBrowser.RegisterAsBrowser = true;
    
                    // 新しい WebBrowser からさらにウィンドウを開かれるときも同じようにする
                    newBrowser.NewWindow2 += webBrowser_NewWindow2;
                }
             */
        }


        #region Menu Items

        private void menuVerInfoToolStripMenuItem_Click(object sender, EventArgs e) {
            frmVer.ShowDialog();
        }

        private void menuWUToolStripMenuItem_Click(object sender, EventArgs e) {
            WB_Navigate(strWU);
        }

        private void menuPrintToolStripMenuItem_Click(object sender, EventArgs e) {
            foreach (WebBrowser_NewWindowHacked wb in tabControl1.SelectedTab.Controls)
                wb.ShowPrintDialog();
        }

        private void menuPrintPrwviewToolStripMenuItem_Click(object sender, EventArgs e) {
            foreach (WebBrowser_NewWindowHacked wb in tabControl1.SelectedTab.Controls)
                wb.ShowPrintPreviewDialog();
        }

        private void menuPlgInfoToolStripMenuItem_Click(object sender, EventArgs e) {
            frmPlgInfo.ShowDialog();
        }

        private void menuNewTabToolStripMenuItem_Click(object sender, EventArgs e) {
            AddTab(sender, e);
        }

        private void menuOpenToolStripMenuItem_Click(object sender, EventArgs e) {
            var op = new OpenFileDialog() {
                Multiselect = false,
                CheckFileExists = true,
                CheckPathExists = true,
                AutoUpgradeEnabled = true,
                Title = strOpenFileDialog
            };
            op.ShowDialog();
            if (op.FileName != "") {
                if (cm.GetStr("Behavior_Open") == "Current") {
                    WB_Navigate(op.FileName);
                } else {
                    AddTab(op.FileName);
                }
            }
        }

        private bool isTabClosing = false;
        private void menuCloseTabToolStripMenuItem_Click(object sender, EventArgs e) {
            isTabClosing = true;
            RemoveTab(sender, e);
            isTabClosing = false;
        }

        private void menuSaveAsToolStripMenuItem_Click(object sender, EventArgs e) {
            ((WebBrowser_NewWindowHacked)(tabControl1.SelectedTab.Controls[0])).ShowSaveAsDialog();
        }

        private void menuPageSetupToolStripMenuItem_Click(object sender, EventArgs e) {
            ((WebBrowser_NewWindowHacked)(tabControl1.SelectedTab.Controls[0])).ShowPageSetupDialog();
        }

        private void menuConfigToolStripMenuItem_Click(object sender, EventArgs e) {
            cm.ShowConfigDialog();
        }

        private void menuInetOptionToolStripMenuItem_Click(object sender, EventArgs e) {
            string syspath = Environment.GetEnvironmentVariable("SystemRoot") + "\\System32\\rundll32.exe";
            System.Diagnostics.Process.Start(syspath, strInetCpl);
        }

        #endregion

        private void btReload_Click(object sender, EventArgs e) {
            try { ((WebBrowser)tabControl1.SelectedTab.Controls[0]).Refresh(); } catch { AutoTabRemove(); }
        }

        private void btGoHome_Click(object sender, EventArgs e) {
            try { WB_Navigate(cm.GetStr("HomePage")); } catch { AutoTabRemove(); }
        }

        private void btStop_Click(object sender, EventArgs e) {
            try {
                ((WebBrowser)tabControl1.SelectedTab.Controls[0]).Stop();
                toolStripProgressBar1.Visible = false;
            } catch { AutoTabRemove(); }
        }

        private void btPrint_ButtonClick(object sender, EventArgs e) {
            try { ((WebBrowser)tabControl1.SelectedTab.Controls[0]).ShowPrintDialog(); } catch { AutoTabRemove(); }
        }

        private void menuPrintPreview_Click(object sender, EventArgs e) {
            try { ((WebBrowser)tabControl1.SelectedTab.Controls[0]).ShowPrintPreviewDialog(); } catch { AutoTabRemove(); }
        }

        private void menuPageConfig_Click(object sender, EventArgs e) {
            try { ((WebBrowser)tabControl1.SelectedTab.Controls[0]).ShowPageSetupDialog(); } catch { AutoTabRemove(); }
        }

        private void btAddBookmark_Click(object sender, EventArgs e) {
            //TODO:BookmarkManagerからデリゲート引っ張ってきて登録してしまいたい

            //TODO:アクティブなタブから名前とURLを取得して新しいBookmarkItemをつくってmanage.ShowDialogに送る

            bm.addBookmarkMenu_Click(((WebBrowser)tabControl1.SelectedTab.Controls[0]).DocumentTitle, ((WebBrowser)tabControl1.SelectedTab.Controls[0]).Url.ToString());

        }

        private void btSearch_Click(object sender, EventArgs e) {
            var search = System.Text.RegularExpressions.Regex.Replace(tbSearch.Text, "#", "%23");
            WB_Navigate(String.Format(query, search));
        }

        private void btGoBack_ButtonClick(object sender, EventArgs e) {
            try { ((WebBrowser)tabControl1.SelectedTab.Controls[0]).GoBack(); } catch { AutoTabRemove(); }
        }

        private void btGoForward_ButtonClick(object sender, EventArgs e) {
            try { ((WebBrowser)tabControl1.SelectedTab.Controls[0]).GoForward(); } catch { AutoTabRemove(); }
        }

        private void tbSearch_KeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e) {
            if (e.KeyChar == (char)(Keys.Return)) {
                btSearch_Click(sender, e);
            }
        }

    }//class MainForm

}//namespace Bis5Products.SharpWebBrowser  