/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.connect;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.StringTokenizer;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import jdbcacsess2.connect.ConnectTableModel.AlreadyExistsException;
import jdbcacsess2.connect.ConnectView.ViewConnect;
import jdbcacsess2.main.Config;
import jdbcacsess2.main.Jdbcacsess2;
import jdbcacsess2.main.ShowDialog;
import jdbcacsess2.sqlService.DataBaseConnection;
import jdbcacsess2.sqlService.DataBaseConnectionListener;
import jdbcacsess2.sqlService.DataBaseListener;
import jdbcacsess2.sqlService.DataBaseTransactionListener;
import net.java.ao.DBParam;
import net.java.ao.Query;

@SuppressWarnings("serial")
public class ConnectController {

	/**
	 * 設定情報にアクセスする
	 */
	private Config config;
	/**
	 * データベースコネクション及びデータトランザクションを通知するリスナーの登録要求リスト
	 */
	private final List<DataBaseListener> requestListeners = new CopyOnWriteArrayList<DataBaseListener>(); // @jve:decl-index=0:
	/**
	 * 接続画面のダイアログ
	 */
	private ConnectView view;
	/**
	 * 接続先一覧のテーブルモデル
	 */
	private ConnectTableModel connectTableModel;

	/**
	 * 接続先一覧の選択に連動して、接続詳細情報を表示する
	 */
	private final ListSelectionListener selectionConnectList = new ListSelectionListener() {
		@Override
		public void valueChanged(ListSelectionEvent e) {
			Jdbcacsess2.logger.finer(e.toString());

			if (e.getValueIsAdjusting()) {
				return;
			}

			ConfigConnect row = view.getSelectedConfigConnect();
			if (row == null) {
				return;
			}

			view.getJTextFieldConnectName().setText(row.getConnectName());
			view.getJComboBoxDBMS().setSelectedItem(row.getConfigDbms());

			view.getJTextFieldUser().setText(row.getConnectUser());
			view.getJPasswordFieldPassword().setText(row.getPassword());

			view.getJTextFieldHostName().setText(row.getHostName());
			view.getJTextFieldPortNumber().setText(row.getPortNumber());
			view.getJTextFieldDatabaseName().setText(row.getDatabaseName());

			view.getJTextFieldClassName().setText(row.getClassName());
			view.getJTextAreaDriverFile().setText(row.getDriverFile());
			view.getJTextFieldUrlOption().setText(row.getUrlOption());
		}
	};

	/**
	 *　入力に連動して、url文字列を表示する
	 */
	private class JdbcUrlChnageListener implements DocumentListener, ActionListener {
		private void jdbcUrlConvert() {
			ConfigDbms selectDbms = (ConfigDbms) view.getJComboBoxDBMS().getSelectedItem();
			if (selectDbms == null) {
				return;
			}

			String subprotocol = selectDbms.getSubprotocol();
			String subname = selectDbms.getSubname();
			String portNumber = view.getJTextFieldPortNumber().getText();

			StringBuffer url = new StringBuffer("jdbc" + ("".equals(subprotocol) ? "" : ":" + subprotocol)
			                                    + ("".equals(subname) ? "" : ":" + subname));

			replaceString(url, "%HostName%", view.getJTextFieldHostName().getText());
			replaceString(url, "%DataBaseName%", view.getJTextFieldDatabaseName().getText());

			replaceString(url, "%PortNumber%", ("".equals(portNumber) ? "" : ":") + portNumber);
			replaceString(url, "%PortNumberRAW%", portNumber);

			url.append(view.getJTextFieldUrlOption().getText());

			view.getJTextAreaURL().setText(url.toString());
		}

		private void replaceString(StringBuffer org, String b, String a) {
			int start = org.indexOf(b);
			if (start == -1) {
				return;
			}
			org.replace(start, start + b.length(), a);
		}

		@Override
		public void removeUpdate(DocumentEvent e) {
			jdbcUrlConvert();
		}

		@Override
		public void insertUpdate(DocumentEvent e) {
			jdbcUrlConvert();
		}

		@Override
		public void changedUpdate(DocumentEvent e) {
			jdbcUrlConvert();
		}

		@Override
		public void actionPerformed(ActionEvent e) {
			jdbcUrlConvert();
		}
	}

	private final JdbcUrlChnageListener changeJdbcUrl = new JdbcUrlChnageListener();

	/**
	 * DBMSの選択に連動してクラス名を変更する
	 */
	private final ItemListener changeDbms = new ItemListener() {
		@Override
		public void itemStateChanged(ItemEvent e) {
			if (e.getStateChange() != ItemEvent.SELECTED) {
				// 選択イベントと解除イベントのうち、解除イベントは捨てる。
				return;
			}
			ConfigDbms selectDbms = (ConfigDbms) view.getJComboBoxDBMS().getSelectedItem();
			if (selectDbms == null) {
				return;
			}
			view.getJTextFieldClassName().setEditable(selectDbms.getDriverclass().equals(""));
			view.getJTextFieldClassName().setOpaque(selectDbms.getDriverclass().equals(""));
			view.getJTextFieldClassName().setText(selectDbms.getDriverclass());
		}
	};

	/**
	 * ドライバファイルの参照アクション
	 */
	private final Action actionFileChoose = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			final JFileChooser jFileChooser = new JFileChooser();
			jFileChooser.setMultiSelectionEnabled(true);
			jFileChooser.addActionListener(new ActionListener() {
				@Override
				public void actionPerformed(ActionEvent e) {
					if (e.getActionCommand().equals(JFileChooser.APPROVE_SELECTION)) {
						addDriverFile(jFileChooser);
					}
				}
			});
			jFileChooser.showOpenDialog(view);
		}
	};

	/**
	 * ファイル選択結果を画面の入力欄に追記する
	 * 
	 * @param jFileChooser
	 */
	private void addDriverFile(JFileChooser jFileChooser) {
		// 現在の画面入力値を分解
		ArrayList<String> files = new ArrayList<String>();
		StringTokenizer st = new StringTokenizer(view.getJTextAreaDriverFile().getText(), File.pathSeparator);
		while (st.hasMoreTokens()) {
			String s = st.nextToken();
			if (s.equals("")) {
				continue;
			}
			files.add(s);
		}

		// 選択ファイルを追加
		for (File f : jFileChooser.getSelectedFiles()) {
			files.add(f.getAbsolutePath());
		}

		// 連結して画面に設定
		StringBuilder sb = new StringBuilder();
		for (String f : files) {
			if (sb.length() != 0) {
				sb.append(File.pathSeparator);
			}
			sb.append(f);
		}
		view.getJTextAreaDriverFile().setText(sb.toString());
	}

	/**
	 * ボタン以外のシステムウィンドウクローズイベント
	 */
	private final WindowAdapter windowClosing = new WindowAdapter() {
		@Override
		public void windowClosing(WindowEvent e) {
			// component保存
			try {
				ViewConnect viewConnect = config.getEntity(ConnectView.ViewConnect.class);
				view.collectViewConnect(viewConnect);
				viewConnect.save();
			} catch (SQLException ex) {
				ShowDialog.errorMessage(ex);
			}
		}
	};

	/**
	 * 閉じるアクション
	 */
	private final Action actionClose = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			// component保存
			try {
				ViewConnect viewConnect = config.getEntity(ConnectView.ViewConnect.class);
				view.collectViewConnect(viewConnect);
				viewConnect.save();
			} catch (SQLException ex) {
				ShowDialog.errorMessage(ex);
			}
			// ウィンドウ破棄は自前で行う
			view.dispose();
		}
	};

	/**
	 * ダブルクリック接続アクション
	 */
	private final MouseListener connectClickMouseListener = new MouseAdapter() {
		@Override
		public void mouseClicked(MouseEvent me) {
			if (me.getClickCount() == 2) {
				view.getJButtonConnect().doClick();
			}
		}
	};
	/**
	 * 接続アクション
	 */
	private final Action actionConnect = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			try {
				DataBaseConnection dbc =
						new DataBaseConnection(
						                       view.getJTextAreaURL().getText(),
						                       view.getJTextFieldUser().getText(),
						                       new String(view.getJPasswordFieldPassword().getPassword()),
						                       view.getJTextFieldClassName().getText(),
						                       DataBaseConnection.splitPath(view.getJTextAreaDriverFile().getText()),
						                       view.getJTextFieldConnectName().getText(),
						                       config.optionValues
								);
				new DbOpen(dbc).execute();
			} catch (Exception e1) {
				ShowDialog.errorMessage(e1);
			}
		}
	};

	/**
	 * プログレスバーで処理経過を表示しながら、connection のopenを行う
	 * 
	 * @author sima
	 * 
	 */
	private class DbOpen extends SwingWorker<Object, Object> implements DataBaseConnectionListener {
		private final DataBaseConnection dbc;

		DbOpen(DataBaseConnection dbc) {
			this.dbc = dbc;
			// progressbarを不確定モードにして、アニメーション開始
			view.getJProgressBar().setIndeterminate(true);
		}

		@Override
		public Object doInBackground() {
			for (DataBaseListener listener : requestListeners) {
				// 登録要求のあったリスナーを全て登録する。
				if (listener instanceof DataBaseConnectionListener)
					dbc.addConnectionListener((DataBaseConnectionListener) listener);
				if (listener instanceof DataBaseTransactionListener)
					dbc.addTransactionListener((DataBaseTransactionListener) listener);
			}
			dbc.addConnectionListener(this);
			try {
				dbc.open();
			} catch (Exception ex) {
				ShowDialog.errorMessage(ex);
			}
			return null;
		}

		@Override
		protected void done() {
			// アニメション終了
			view.getJProgressBar().setIndeterminate(false);
		}

		@Override
		public void dataBaseConnectionOpened(DataBaseConnection dataBaseConnection) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					try {
						if (view.getJCheckBoxAutosave().isSelected()) {
							ConfigConnect rowSave = config.getConfigConnectRow(view.getJTextFieldConnectName().getText());
							collectInputData(rowSave);
							connectTableModel.save(rowSave);
						}

						ConfigConnect row[] =
								config.getEntityManager().find(ConfigConnect.class,
								                               Query.select().where(ConfigConnect.CONNECTNAME + "=?",
								                                                    view.getJTextFieldConnectName()
								                                                    .getText()));
						ConfigConnect connectedConfigConnect = null;
						for (ConfigConnect element : row) {
							Date lastConnectdate = new Date();
							element.setLastConnectdate(lastConnectdate);
							element.save();
							connectedConfigConnect = element;
							break;
						}

						// component保存
						ViewConnect viewConnect = config.getEntity(ConnectView.ViewConnect.class);
						view.collectViewConnect(viewConnect);
						if (connectedConfigConnect != null) {
							viewConnect.setConnect(connectedConfigConnect);
						}
						viewConnect.save();

					} catch (SQLException e) {
						ShowDialog.errorMessage(e);
					}

					// 登録要求リストは、open後にクリアしておく。
					requestListeners.clear();

					// ウィンドウ破棄は自前で行う
					view.dispose();
				}
			});

		}

		@Override
		public void dataBaseConnectionClosing(DataBaseConnection dataBaseConnection) {
			dataBaseConnection.removeConnectionlisteners(this);
		}

		@Override
		public void dataBaseConnectionClosed(DataBaseConnection dataBaseConnection) {
		}
	}

	/**
	 * 接続テストアクション
	 */
	private final Action actionConnectTest = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			try {
				DataBaseConnection dbc =
						new DataBaseConnection(
						                       view.getJTextAreaURL().getText(),
						                       view.getJTextFieldUser().getText(),
						                       new String(view.getJPasswordFieldPassword().getPassword()),
						                       view.getJTextFieldClassName().getText(),
						                       DataBaseConnection.splitPath(view.getJTextAreaDriverFile().getText()),
						                       view.getJTextFieldConnectName().getText(),
						                       config.optionValues);
				new DbTestOpen(dbc).execute();

			} catch (Exception ex) {
				ShowDialog.errorMessage(ex);
			}
		}
	};

	/**
	 * プログレスバーで処理経過を表示しながら、connection のopenを行う。 但し、接続テスト用なので成功したらすぐにcloseする。
	 * 
	 * @author sima
	 * 
	 */
	private class DbTestOpen extends SwingWorker<Object, Object> implements DataBaseConnectionListener {
		private final DataBaseConnection dbc;

		DbTestOpen(DataBaseConnection dbc) {
			this.dbc = dbc;
			// progressbarを不確定モードにして、アニメーション開始
			view.getJProgressBar().setIndeterminate(true);
		}

		@Override
		public Object doInBackground() {
			dbc.addConnectionListener(this);
			try {
				dbc.open();
				dbc.close();
			} catch (Exception ex) {
				ShowDialog.errorMessage(ex);
			}
			return null;
		}

		@Override
		protected void done() {
			// アニメション終了
			view.getJProgressBar().setIndeterminate(false);
		}

		@Override
		public void dataBaseConnectionOpened(DataBaseConnection dataBaseConnection) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					ShowDialog.infomationMessage(view, "url[" + dbc.getUrl() + "]に、\n" + "user[" + dbc.getUser()
					                             + "]で接続成功しました。", "接続テストに成功しました");

					if (view.getJCheckBoxAutosave().isSelected()) {
						try {
							ConfigConnect row = config.getConfigConnectRow(view.getJTextFieldConnectName().getText());
							collectInputData(row);
							connectTableModel.save(row);
						} catch (SQLException e) {
							ShowDialog.errorMessage(e);
						}
					}
				}
			});

		}

		@Override
		public void dataBaseConnectionClosing(DataBaseConnection dataBaseConnection) {
			dataBaseConnection.removeConnectionlisteners(this);
		}

		@Override
		public void dataBaseConnectionClosed(DataBaseConnection dataBaseConnection) {
		}
	}


	/**
	 * 新規保存アクション
	 */
	private final Action actionNewSave = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			try {
				DBParam dbParam = new DBParam(ConfigConnect.CONNECTNAME,
				                              view.getJTextFieldConnectName().getText());
				ConfigConnect configConnect = config.getEntityManager().create(ConfigConnect.class,
				                                                               dbParam);
				collectInputData(configConnect);
				connectTableModel.append(configConnect);
			} catch (SQLException e1) {
				ShowDialog.errorMessage(e1);
			} catch (AlreadyExistsException e2) {
				ShowDialog.errorMessage(e2);

			}
		}
	};

	/**
	 * 上書き保存アクション
	 */
	private final Action actionSave = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			ConfigConnect configConnect = view.getSelectedConfigConnect();
			if (configConnect == null) {
				return;
			}
			collectInputData(configConnect);
			connectTableModel.update(configConnect);
		}
	};

	/**
	 * 接続削除アクション
	 */
	private final Action actionRemove = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			ConfigConnect configConnect = view.getSelectedConfigConnect();
			if (configConnect == null) {
				return;
			}

			int ans = JOptionPane.showConfirmDialog(view,
			                                        configConnect.getConnectName() + "を削除しますか？",
			                                        "削除確認",
			                                        JOptionPane.YES_NO_OPTION);
			if (ans == JOptionPane.YES_OPTION) {
				try {
					connectTableModel.remove(configConnect);
				} catch (SQLException e1) {
					ShowDialog.errorMessage(e1);
				}
			}
		}
	};



	/**
	 * コンストラクタ。 DB接続画面を表示し、ユーザがDB接続先を指示する。
	 * 接続したDBコネクションは、コンストラクタで指定したリスナーにて通知される。
	 * 
	 * @param config
	 *            設定データ
	 * @param owner
	 *            　Diaog表示する時にオーナとなるJFrame
	 * 
	 * @throws SQLException
	 */
	public ConnectController(Config config, JFrame owner) throws SQLException {
		this.config = config;

		// view構築
		view = new ConnectView(owner);

		// リスナー登録
		addListener();

		// レンダラー登録
		view.getJComboBoxDBMS().setRenderer(new DefaultListCellRenderer() {
			private static final long serialVersionUID = -6249257421893883408L;

			/*
			 * (非 Javadoc)
			 * 
			 * @see
			 * javax.swing.DefaultListCellRenderer#getListCellRendererComponent
			 * (javax.swing.JList, java.lang.Object, int, boolean, boolean)
			 */
			@Override
			public Component getListCellRendererComponent(JList<?> list,
			                                              Object value,
			                                              int index,
			                                              boolean isSelected,
			                                              boolean cellHasFocus) {
				if (value != null) {
					ConfigDbms dbms = (ConfigDbms) value;
					return super.getListCellRendererComponent(list, dbms.getDbmsName(), index, isSelected, cellHasFocus);
				}
				return super.getListCellRendererComponent(list, null, index, isSelected, cellHasFocus);
			}
		});

		// モデル構築
		ConfigDbms[] d = config.getEntityManager().find(ConfigDbms.class, Query.select().order(ConfigDbms.DBMSNAME));
		view.getJComboBoxDBMS().setModel(new DefaultComboBoxModel<ConfigDbms>(d));
		view.getJComboBoxDBMS().setSelectedItem(null);

		ConfigConnect[] e = config.getEntityManager().find(ConfigConnect.class, Query.select());
		connectTableModel = new ConnectTableModel(config, e);
		view.getJTableConnect().setModel(connectTableModel);
		view.getJTableConnect().getRowSorter().toggleSortOrder(0);

		// GUIのコンポーネント属性を復元
		ViewConnect[] c = config.getEntityManager().find(ConnectView.ViewConnect.class);
		if (c.length != 0) {
			view.setViewConnect(c[0]);
		}
	}

	/**
	 * リスナーを登録する
	 * 
	 * @throws SQLException
	 */
	private void addListener() throws SQLException {
		view.getJTableConnect().getSelectionModel().addListSelectionListener(selectionConnectList);
		view.getJTableConnect().addMouseListener(connectClickMouseListener);

		KeyStroke enter = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
		view.getJTableConnect().getInputMap().put(enter, "enter");
		view.getJTableConnect().getActionMap().put("enter", actionConnect);

		view.getJButtonConnect().addActionListener(actionConnect);
		view.getJButtonClose().addActionListener(actionClose);

		view.getJButtonFileChoose().addActionListener(actionFileChoose);

		view.getJButtonConnectTest().addActionListener(actionConnectTest);

		view.getJButtonNewSave().addActionListener(actionNewSave);
		view.getJButtonSave().addActionListener(actionSave);
		view.getJButtonRemove().addActionListener(actionRemove);

		view.getJComboBoxDBMS().addItemListener(changeDbms);

		view.getJComboBoxDBMS().addActionListener(changeJdbcUrl);
		view.getJTextFieldHostName().getDocument().addDocumentListener(changeJdbcUrl);
		view.getJTextFieldPortNumber().getDocument().addDocumentListener(changeJdbcUrl);
		view.getJTextFieldDatabaseName().getDocument().addDocumentListener(changeJdbcUrl);
		view.getJTextFieldUrlOption().getDocument().addDocumentListener(changeJdbcUrl);

		view.addWindowListener(windowClosing);
	}

	/**
	 * Viewの入力値を更新用オブジェクト{ConfigTableConnect)に集める。
	 * 
	 * @param row
	 *            入力値を格納する為のConfigTableConnect。
	 */
	private void collectInputData(ConfigConnect row) {
		row.setConnectName(view.getJTextFieldConnectName().getText().trim());
		row.setConfigDbms((ConfigDbms) view.getJComboBoxDBMS().getSelectedItem());

		row.setConnectUser(view.getJTextFieldUser().getText().trim());
		row.setPassword(new String(view.getJPasswordFieldPassword().getPassword()));

		row.setHostName(view.getJTextFieldHostName().getText().trim());
		row.setPortNumber(view.getJTextFieldPortNumber().getText().trim());
		row.setDatabaseName(view.getJTextFieldDatabaseName().getText().trim());

		row.setClassName(view.getJTextFieldClassName().getText().trim());
		row.setDriverFile(view.getJTextAreaDriverFile().getText().trim());
		row.setUrlOption(view.getJTextFieldUrlOption().getText().trim());
	}

	/**
	 * setVisibleを呼び出す。
	 * 
	 * @param b
	 */
	public void setVisible(boolean b) {
		view.setVisible(b);
	}

	/**
	 * データベース接続リスナーの登録を要求する。
	 * 
	 * @param listener
	 *            データベース接続リスナー
	 */
	public void addReauestDataBaseListener(DataBaseListener listener) {
		if (listener == null) {
			throw new NullPointerException();
		}
		requestListeners.add(listener);
	}
}
