/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.main;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Font;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultTreeCellRenderer;

import jdbcacsess2.main.icon.IconImage;
import jdbcacsess2.sqlService.DataBaseConnection;
import jdbcacsess2.sqlService.dbobject.DBObject;
import jdbcacsess2.sqlService.dbobject.DBObjects;
import jdbcacsess2.sqlService.dbobject.DBobjectChangeListener;
import jdbcacsess2.sqlService.dbobject.DBobjectMutableTreeTableNode;

import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.decorator.HighlighterFactory;
import org.jdesktop.swingx.treetable.DefaultTreeTableModel;

public class JPanelSession extends JPanel {

	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------

	private DataBaseConnection dataBaseConnection; // @jve:decl-index=0:

	/**
	 * @param dataBaseConnection
	 *            セットする dataBaseConnection
	 */
	void setDataBaseConnection(DataBaseConnection dataBaseConnection) {
		this.dataBaseConnection = dataBaseConnection;
	}

	/**
	 * @return dataBaseConnection
	 */
	DataBaseConnection getDataBaseConnection() {
		return dataBaseConnection;
	}

	/**
	 * tabnedPaneのtabComponent上のマウス位置を把握する
	 */
	public TabbedPaneMousePointer tabbedPaneMousePointer;

	/**
	 * SQL情報パネルの作成数カウンタ
	 */
	private int jTabbedPanelSqlCounter = 1;

	/**
	 * SQL情報パネルの作成数カウンタを１加算
	 * 
	 * @return　加算したカウンタ
	 */
	public int increaseJTabbedPanelSqlCount() {
		return jTabbedPanelSqlCounter++;
	}

	/**
	 * データベースオブジェクトツリーのセルレンダラー
	 * 
	 * @author sima
	 * 
	 */
	static class ObjectsTreeCellRenderer extends DefaultTreeCellRenderer {

		private static final long serialVersionUID = 988648723443010093L;

		@Override
		public Component getTreeCellRendererComponent(JTree tree,
		                                              Object value,
		                                              boolean selected,
		                                              boolean expanded,
		                                              boolean leaf,
		                                              int row,
		                                              boolean hasFocus) {
			DBObject o = ((DBobjectMutableTreeTableNode) value).getDBObject();

			Icon icon = o.getIconImage();
			setOpenIcon(icon);
			setClosedIcon(icon);
			setLeafIcon(icon);

			Font font = getFont();
			if (font != null) {
				if (o.isStrong()) {
					setFont(getFont().deriveFont(Font.BOLD));
				} else {
					setFont(getFont().deriveFont(Font.PLAIN));
				}
			}

			super.getTreeCellRendererComponent(tree,
			                                   o.isCaching() ? "LOADONG... " + o.getSummary() : o.getSummary(),
			                                   selected,
			                                   expanded,
			                                   leaf,
			                                   row,
			                                   hasFocus);
			return this;
		}
	}

	/**
	 * DBパネルデータベースオブジェクトツリーの子ノードを追加する
	 * 
	 * @param treeNode
	 *            親ノード
	 */
	void addChildren(final DBobjectMutableTreeTableNode treeNode) {
		DBObject parent = treeNode.getDBObject();
		if (parent.isBottom() || !parent.isNoCache()) {
			return;
		}

		parent.getAsyncChildren(getDataBaseConnection(), new DBobjectChangeListener(treeNode) {

			@Override
			public void exception(Throwable t) {
				ShowDialog.errorMessage(t);
			}

			@Override
			public void valueChanged(final DBObjects dbObjects) {
				SwingUtilities.invokeLater(new Runnable() {
					@Override
					public void run() {
						DefaultTreeTableModel treeTableModel =
						        (DefaultTreeTableModel) getJXTreetableDBObjects().getTreeTableModel();

						for (DBObject dbObject : dbObjects.getDBObjects()) {
							DBobjectMutableTreeTableNode child = new DBobjectMutableTreeTableNode(dbObject,
							                                                                      !dbObject.isBottom());
							treeTableModel.insertNodeInto(child, getParentNode(), getParentNode().getChildCount());
						}

					}
				});
			}
		});

	}

	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------

	private static final long serialVersionUID = 4124766267642610593L;

	JLabel jLabelStatus = null;
	private JSplitPane jSplitPaneTree = null;
	private JXTreeTable jxTreeTableDBObjects = null;
	private JPanel jPanel1 = null;
	private JTabbedPane jTabbedPaneSql = null;
	private JPopupMenu jPopupTabbedSql = null;
	private JMenuItem jMenuTabbedSqlNew = null;
	private JMenuItem jMenuTabbedSqlClose = null;
	private JMenuItem jMenuTabbedSqlCloseOther = null;

	private JScrollPane jScrollPaneDBObjects = null;

	private JSplitPane jSplitPaneDBObjects = null;

	private JScrollPane jScrollPaneProperty = null;

	private JmyTable jTableProperty = null;

	private JTabbedPane jTabbedPane = null;

	private JScrollPane jScrollPaneExportedKey = null;

	private JmyTable jTableExportedKey = null;

	private JPopupMenu jPopupMenu = null;

	private JMenuItem jMenuDBobjectsRefresh = null;

	/**
	 * This method initializes
	 * 
	 */
	public JPanelSession() {
		super();
		initialize();
	}

	/**
	 * This method initializes this
	 * 
	 */
	private void initialize() {
		jLabelStatus = new JLabel();
		jLabelStatus.setText("");
		this.setLayout(new BorderLayout());
		this.add(jLabelStatus, java.awt.BorderLayout.SOUTH);
		this.add(getJSplitPaneTree(), java.awt.BorderLayout.CENTER);
	}

	/**
	 * This method initializes jSplitPaneTree
	 * 
	 * @return javax.swing.JSplitPane
	 */
	JSplitPane getJSplitPaneTree() {
		if (jSplitPaneTree == null) {
			jSplitPaneTree = new JSplitPane();
			jSplitPaneTree.setOneTouchExpandable(true);
			jSplitPaneTree.setDividerLocation(250);
			jSplitPaneTree.setLeftComponent(getJSplitPaneDBObjects());
			jSplitPaneTree.setRightComponent(getJPanel1());
		}
		return jSplitPaneTree;
	}

	/**
	 * This method initializes jTreeObjects
	 * 
	 * @return javax.swing.JTree
	 */
	JXTreeTable getJXTreetableDBObjects() {
		if (jxTreeTableDBObjects == null) {
			jxTreeTableDBObjects = new JXTreeTable();
			jxTreeTableDBObjects.setTreeTableModel(new DefaultTreeTableModel());
			jxTreeTableDBObjects.setToggleClickCount(0);
			jxTreeTableDBObjects.setTreeCellRenderer(new ObjectsTreeCellRenderer());
			jxTreeTableDBObjects.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
			jxTreeTableDBObjects.setShowVerticalLines(false);
			jxTreeTableDBObjects.setShowHorizontalLines(false);
			jxTreeTableDBObjects.setOpaque(true);
			jxTreeTableDBObjects.setHighlighters(HighlighterFactory.createSimpleStriping());
			jxTreeTableDBObjects.setComponentPopupMenu(getJPopupMenu());
			FontZoom.setMouseWheel(jxTreeTableDBObjects);
		}
		return jxTreeTableDBObjects;
	}

	/**
	 * This method initializes jPanel1
	 * 
	 * @return javax.swing.JPanel
	 */
	private JPanel getJPanel1() {
		if (jPanel1 == null) {
			jPanel1 = new JPanel();
			jPanel1.setLayout(new BorderLayout());
			jPanel1.add(getJTabbedPaneSql(), java.awt.BorderLayout.CENTER);
		}
		return jPanel1;
	}

	/**
	 * This method initializes jTabbedPaneSql
	 * 
	 * @return javax.swing.JTabbedPane
	 */
	public JTabbedPane getJTabbedPaneSql() {
		if (jTabbedPaneSql == null) {
			jTabbedPaneSql = new JTabbedPane();
			jTabbedPaneSql.setComponentPopupMenu(getJPopupTabbedSql());
			tabbedPaneMousePointer = new TabbedPaneMousePointer(jTabbedPaneSql);
		}
		return jTabbedPaneSql;
	}

	/**
	 * This method initializes jPopupTabbedSql
	 * 
	 * @return javax.swing.JPopupMenu
	 */
	JPopupMenu getJPopupTabbedSql() {
		if (jPopupTabbedSql == null) {
			jPopupTabbedSql = new JPopupMenu();
			jPopupTabbedSql.add(getJMenuTabbedSqlNew());
			jPopupTabbedSql.add(getJMenuTabbedSqlClose());
			jPopupTabbedSql.add(getJMenuTabbedSqlCloseOther());
		}
		return jPopupTabbedSql;
	}

	/**
	 * This method initializes jMenuTabbedSqlNew
	 * 
	 * @return javax.swing.JMenuItem
	 */
	JMenuItem getJMenuTabbedSqlNew() {
		if (jMenuTabbedSqlNew == null) {
			jMenuTabbedSqlNew = new JMenuItem(IconImage.TAB_ADD.getValue());
			jMenuTabbedSqlNew.setText("New");
		}
		return jMenuTabbedSqlNew;
	}

	/**
	 * This method initializes jMenuTabbedSqlClose
	 * 
	 * @return javax.swing.JMenuItem
	 */
	JMenuItem getJMenuTabbedSqlClose() {
		if (jMenuTabbedSqlClose == null) {
			jMenuTabbedSqlClose = new JMenuItem(IconImage.TAB_DELETE.getValue());
			jMenuTabbedSqlClose.setText("Close");
		}
		return jMenuTabbedSqlClose;
	}

	/**
	 * This method initializes jMenuTabbedSqlCloseOther
	 * 
	 * @return javax.swing.JMenuItem
	 */
	JMenuItem getJMenuTabbedSqlCloseOther() {
		if (jMenuTabbedSqlCloseOther == null) {
			jMenuTabbedSqlCloseOther = new JMenuItem(IconImage.TAB_DELETE.getValue());
			jMenuTabbedSqlCloseOther.setText("CloseOther");
		}
		return jMenuTabbedSqlCloseOther;
	}

	/**
	 * This method initializes jScrollPaneDBObjects
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private JScrollPane getJScrollPaneDBObjects() {
		if (jScrollPaneDBObjects == null) {
			jScrollPaneDBObjects = new JScrollPane();
			jScrollPaneDBObjects.setViewportView(getJXTreetableDBObjects());
		}
		return jScrollPaneDBObjects;
	}

	/**
	 * This method initializes jSplitPaneDBObjects
	 * 
	 * @return javax.swing.JSplitPane
	 */
	JSplitPane getJSplitPaneDBObjects() {
		if (jSplitPaneDBObjects == null) {
			jSplitPaneDBObjects = new JSplitPane();
			jSplitPaneDBObjects.setOrientation(JSplitPane.VERTICAL_SPLIT);
			jSplitPaneDBObjects.setDividerLocation(450);
			jSplitPaneDBObjects.setBottomComponent(getJTabbedPane());
			jSplitPaneDBObjects.setTopComponent(getJScrollPaneDBObjects());
		}
		return jSplitPaneDBObjects;
	}

	/**
	 * This method initializes jScrollPaneProperty
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private JScrollPane getJScrollPaneProperty() {
		if (jScrollPaneProperty == null) {
			jScrollPaneProperty = new JScrollPane();
			jScrollPaneProperty.setViewportView(getJTableProperty());
		}
		return jScrollPaneProperty;
	}

	/**
	 * This method initializes jTableProperty
	 * 
	 * @return javax.swing.JTable
	 */
	JmyTable getJTableProperty() {
		if (jTableProperty == null) {
			jTableProperty = new JmyTable();
		}
		return jTableProperty;
	}

	/**
	 * This method initializes jTabbedPane
	 * 
	 * @return javax.swing.JTabbedPane
	 */
	private JTabbedPane getJTabbedPane() {
		if (jTabbedPane == null) {
			jTabbedPane = new JTabbedPane();
			jTabbedPane.addTab("Properties", null, getJScrollPaneProperty(), null);
			jTabbedPane.addTab("ExportedKeys", null, getJScrollPaneExportedKey(), null);
		}
		return jTabbedPane;
	}

	/**
	 * This method initializes jScrollPaneExportedKey
	 * 
	 * @return javax.swing.JScrollPane
	 */
	private JScrollPane getJScrollPaneExportedKey() {
		if (jScrollPaneExportedKey == null) {
			jScrollPaneExportedKey = new JScrollPane();
			jScrollPaneExportedKey.setViewportView(getJTableExportedKey());
		}
		return jScrollPaneExportedKey;
	}

	/**
	 * This method initializes jTableExportedKey
	 * 
	 * @return javax.swing.JTable
	 */
	JmyTable getJTableExportedKey() {
		if (jTableExportedKey == null) {
			jTableExportedKey = new JmyTable();
		}
		return jTableExportedKey;
	}

	/**
	 * This method initializes jPopupMenu
	 * 
	 * @return javax.swing.JPopupMenu
	 */
	private JPopupMenu getJPopupMenu() {
		if (jPopupMenu == null) {
			jPopupMenu = new JPopupMenu();
			jPopupMenu.add(getJMenuDBobjectsRefresh());
		}
		return jPopupMenu;
	}

	/**
	 * This method initializes jMenuDBobjectsRefresh
	 * 
	 * @return javax.swing.JMenuItem
	 */
	JMenuItem getJMenuDBobjectsRefresh() {
		if (jMenuDBobjectsRefresh == null) {
			jMenuDBobjectsRefresh = new JMenuItem(IconImage.REFRESH.getValue());
			jMenuDBobjectsRefresh.setText("Refresh");
		}
		return jMenuDBobjectsRefresh;
	}

}
