/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package jdbcacsess2.main;

import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.KeyStroke;

/**
 * JTable に 1)CTRL+矢印キー で移動する機能 2)選択列/行の拡張する機能 のショートカットキーキーを登録する
 * 
 * @author sima
 * 
 */
public class JTableShortcut {
	static public void setJTableShortcut(JTable jtable) {
		setShortCut(jtable, KeyEvent.VK_UP, InputEvent.CTRL_DOWN_MASK, "columnFirst", actionColumnFirstJump);
		setShortCut(jtable, KeyEvent.VK_DOWN, InputEvent.CTRL_DOWN_MASK, "columnEnd", actionColumnEndJump);
		setShortCut(jtable, KeyEvent.VK_LEFT, InputEvent.CTRL_DOWN_MASK, "rowFirst", actionRowFirstJump);
		setShortCut(jtable, KeyEvent.VK_RIGHT, InputEvent.CTRL_DOWN_MASK, "rowEnd", actionRowEndJump);

		setShortCut(jtable, KeyEvent.VK_SPACE, InputEvent.CTRL_DOWN_MASK, "rowexpand", actionRowSelectExpand);
		setShortCut(jtable, KeyEvent.VK_SPACE, InputEvent.SHIFT_DOWN_MASK, "colexpand", actionColumnSelectexpand);
	}

	/**
	 * inputmap と actionmap を使用し、ショートカットキーを設定する
	 * 
	 * @param jComponent
	 * @param keyCode
	 * @param modifiers
	 * @param key
	 * @param action
	 */
	static private void setShortCut(JComponent jComponent, int keyCode, int modifiers, String key, Action action) {
		InputMap inputMap = jComponent.getInputMap();
		ActionMap actionMap = jComponent.getActionMap();

		KeyStroke keyStroke = KeyStroke.getKeyStroke(keyCode, modifiers);
		inputMap.put(keyStroke, key);
		actionMap.put(key, action);
	}

	private static abstract class JTableCursorJump extends AbstractAction {
		private static final long serialVersionUID = -8725551506660304024L;

		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() instanceof JTable) {
				JTable jTable = (JTable) e.getSource();

				int rowIndex = getRowJumpIndex(jTable.getSelectedRow(), jTable.getRowCount());
				int columnIndex = getColumnJumpIndex(jTable.getSelectedColumn(), jTable.getColumnCount());

				// スクロール
				Rectangle cellRect = jTable.getCellRect(rowIndex, columnIndex, false);
				if (cellRect != null) {
					jTable.scrollRectToVisible(cellRect);
				}

				// 移動
				jTable.changeSelection(rowIndex, columnIndex, false, false);
			}
		}

		/**
		 * 行移動先を計算する
		 * 
		 * @param rowIndex
		 *            現在位置
		 * @param rowCount
		 *            要素数
		 * @return
		 */
		abstract int getRowJumpIndex(int rowIndex, int rowCount);

		/**
		 * 列移動先を計算する
		 * 
		 * @param ColumnIndex
		 *            現在位置
		 * @param columnCount
		 *            要素数
		 * @return
		 */
		abstract int getColumnJumpIndex(int ColumnIndex, int columnCount);
	}

	static private final Action actionRowFirstJump = new JTableCursorJump() {
		private static final long serialVersionUID = -2158918605198692637L;

		@Override
		int getColumnJumpIndex(int ColumnIndex, int columnCount) {
			return 0;
		}

		@Override
		int getRowJumpIndex(int rowIndex, int rowCount) {
			return rowIndex;
		}
	};

	static private final Action actionRowEndJump = new JTableCursorJump() {
		private static final long serialVersionUID = 7159547464095054706L;

		@Override
		int getColumnJumpIndex(int ColumnIndex, int columnCount) {
			return columnCount - 1;
		}

		@Override
		int getRowJumpIndex(int rowIndex, int rowCount) {
			return rowIndex;
		}
	};

	static private final Action actionColumnFirstJump = new JTableCursorJump() {
		private static final long serialVersionUID = 3289572832545730204L;

		@Override
		int getColumnJumpIndex(int ColumnIndex, int columnCount) {
			return ColumnIndex;
		}

		@Override
		int getRowJumpIndex(int rowIndex, int rowCount) {
			return 0;
		}
	};

	static private final Action actionColumnEndJump = new JTableCursorJump() {
		private static final long serialVersionUID = 2404431616648661021L;

		@Override
		int getColumnJumpIndex(int ColumnIndex, int columnCount) {
			return ColumnIndex;
		}

		@Override
		int getRowJumpIndex(int rowIndex, int rowCount) {
			return rowCount - 1;
		}
	};

	/**
	 * 選択範囲を列全体に拡張する。
	 */
	static private final Action actionRowSelectExpand = new AbstractAction() {
		private static final long serialVersionUID = 8769124792157544519L;

		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() instanceof JTable) {
				JTable jTable = (JTable) e.getSource();

				int rowIndex = jTable.getSelectedRow();
				int columnIndex = jTable.getSelectedColumn();

				jTable.addRowSelectionInterval(0, jTable.getRowCount() - 1);

				jTable.changeSelection(rowIndex, columnIndex, true, true);
			}
		}
	};

	/**
	 * 選択範囲を行全体に拡張する。
	 */
	static private final Action actionColumnSelectexpand = new AbstractAction() {
		private static final long serialVersionUID = 7534881048814827865L;

		@Override
		public void actionPerformed(ActionEvent e) {
			if (e.getSource() instanceof JTable) {
				JTable jTable = (JTable) e.getSource();

				int rowIndex = jTable.getSelectedRow();
				int columnIndex = jTable.getSelectedColumn();

				jTable.addColumnSelectionInterval(0, jTable.getColumnCount() - 1);
				jTable.changeSelection(rowIndex, columnIndex, true, true);

			}
		}
	};

}
