/*
 * Copyright 2011 Kazuhiro Shimada
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *	    http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jdbcacsess2.main;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.JComponent;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.JToggleButton;
import javax.swing.KeyStroke;
import javax.swing.RowFilter;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.table.TableRowSorter;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import javax.swing.text.Keymap;
import javax.swing.tree.ExpandVetoException;
import javax.swing.tree.TreePath;

import jdbcacsess2.connect.ConnectController;
import jdbcacsess2.main.JPanelSql.SentenceSeparator;
import jdbcacsess2.main.MainView.ViewMain;
import jdbcacsess2.sqlService.CodeAssistModel;
import jdbcacsess2.sqlService.DataBaseConnection;
import jdbcacsess2.sqlService.DataBaseConnectionListener;
import jdbcacsess2.sqlService.DataBaseTransactionListener;
import jdbcacsess2.sqlService.SqlAsyncExecute;
import jdbcacsess2.sqlService.SqlExecutedListener;
import jdbcacsess2.sqlService.StringConncat;
import jdbcacsess2.sqlService.dbobject.DBObject;
import jdbcacsess2.sqlService.dbobject.DBObjectTableColumns.DBObjectTableColumn;
import jdbcacsess2.sqlService.dbobject.DBObjectTables.DBObjectTable;
import jdbcacsess2.sqlService.dbobject.DBObjectsRoot;
import jdbcacsess2.sqlService.dbobject.DBobjectMutableTreeTableNode;
import jdbcacsess2.sqlService.dbobject.PropertyDBObjectsTableModel;
import jdbcacsess2.sqlService.dbobject.PropertyExportedKeyTableModel;
import jdbcacsess2.sqlService.exception.DbConnectAlreadyException;
import jdbcacsess2.sqlService.exception.DbConnectIllgalStateException;
import jdbcacsess2.sqlService.history.HistryTableModel;
import jdbcacsess2.sqlService.history.Record;
import jdbcacsess2.sqlService.parse.SqlExecuteSentencies.SqlExecuteSentence;

import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.treetable.DefaultTreeTableModel;
import org.jdesktop.swingx.treetable.MutableTreeTableNode;

/**
 * @author sima
 * 
 */
@SuppressWarnings("serial")
public class MainController {

	private final Config config;

	private final HistryTableModel histryTableModel;

	private final MainView view;

	private final CodeAssistView codeAssistView;

	/**
	 * 新しいDBパネルを開く
	 */
	private final ActionListener actionNewPanelSession = new ActionListener() {
		@Override
		public void actionPerformed(ActionEvent e) {
			try {
				createConnectionJPanelSession();
			} catch (Exception e2) {
				ShowDialog.errorMessage(e2);
			}
		}

	};
	private final ActionListener actionReNewPanelSession = new ActionListener() {
		@Override
		public void actionPerformed(ActionEvent e) {
			try {
				view.getCurrentJPanelSession().getDataBaseConnection().close();
			} catch (DbConnectIllgalStateException e1) {
			}
			try {
				view.getCurrentJPanelSession().getDataBaseConnection().open();
			} catch (Exception e2) {
				ShowDialog.errorMessage(e2);
			}
		}
	};
	/**
	 * DBパネルを閉じる
	 */
	private final ActionListener actionClosePanelSession = new ActionListener() {
		@Override
		public void actionPerformed(ActionEvent e) {
			int index = view.tabbedPaneMousePointer.getFocusedComponentIndex();
			if (index == -1) {
				return;
			}
			view.getJTabbedPaneSession().remove(index);
		}
	};

	/**
	 * ショートカットキーによる実行＆停止のアクション
	 */
	private final Action actionExecutionShortcut = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			view.getCurrentJPanelSql().getJToggleButtonExecution().doClick();
		}
	};
	/**
	 * 実行＆停止のアクション
	 */
	private final Action actionExecution = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			JPanelSql jPanelSql = view.getCurrentJPanelSql();

			if (jPanelSql.getJToggleButtonExecution().isSelected()) {
				exec(jPanelSql);
			} else {
				stop(jPanelSql);
			}
		}

		/**
		 * 実行のアクション
		 */
		private void exec(JPanelSql jPanelSql) {
			String sql = jPanelSql.getRSyntaxTextAreaSql().getSelectedText();
			if (sql == null) {
				sql = jPanelSql.getRSyntaxTextAreaSql().getText();
			}

			SqlAsyncExecute sqlAsyncExecute =
					new SqlAsyncExecute(sql,
					                    ((SentenceSeparator) jPanelSql.getJComboBoxSentenceSeparator()
					                    		.getSelectedItem()).getRegVal(),
					                    		config.optionValues);
			if (sqlAsyncExecute.isDuplxSelect()) {
				ShowDialog.warningMessage("複数のSELECT文は実行出来ません", "複数のSELECT文");
				jPanelSql.getJToggleButtonExecution().setSelected(false);
				return;
			}

			for (SqlExecuteSentence ses : sqlAsyncExecute.getSqlExecuteSentencies()) {
				for (String s : ses.getSqlInputParameter().getInputItemNames()) {
					// TODO パラメータを設定する
					Jdbcacsess2.logger.fine(ses.getSqlCommand() + " " + s);
				}
			}

			try {
				sqlAsyncExecute.addSqlExcutedListener(jPanelSql);
				sqlAsyncExecute.addSqlExcutedListener((SqlExecutedListener) jPanelSql.getJTable().getModel());
				sqlAsyncExecute.addSqlExcutedListener(new Record(config,
				                                                 jPanelSql.getDataBaseConnection().getConnectName(),
				                                                 jPanelSql.getDataBaseConnection().getUrl(),
				                                                 jPanelSql.getDataBaseConnection().getUser(),
				                                                 histryTableModel));
				// sql実行
				sqlAsyncExecute.executeAsync(jPanelSql.getDataBaseConnection());

			} catch (Exception ex) {
				ShowDialog.errorMessage(ex);
			}
		}

		/**
		 * 停止のアクション
		 * 
		 * @param jPanelSql
		 */
		private void stop(JPanelSql jPanelSql) {
			jPanelSql.taskCancel();
		}
	};

	/**
	 * 接続と切断のアクション
	 */
	private final Action actionConnection = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {

			if (view.getCurrentJPanelSql().getDataBaseConnection() == null) {
				return;
			}

			JToggleButton tb = (JToggleButton) e.getSource();
			try {
				if (tb.isSelected()) {
					try {
						view.getCurrentJPanelSql().getDataBaseConnection().open();
					} catch (DbConnectAlreadyException e1) {
						e1.printStackTrace();
					}
				} else {
					try {
						view.getCurrentJPanelSql().getDataBaseConnection().close();
					} catch (DbConnectIllgalStateException e1) {
						e1.printStackTrace();
					}
				}
			} catch (SQLException ex) {
				ShowDialog.errorMessage(ex);
				tb.setSelected(false);
			}
		}
	};

	/**
	 * 検索継続のアクション
	 */
	private final Action actionContinue = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			view.getCurrentJPanelSql().taskWakuUp();
		}
	};

	/**
	 * コミットボタンのアクション
	 */
	private final Action actionCommit = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			if (view.getCurrentJPanelSql().getDataBaseConnection() == null) {
				return;
			}
			try {
				view.getCurrentJPanelSql().getDataBaseConnection().commit();
				Jdbcacsess2.logger.info("commit");
			} catch (Exception ex) {
				ShowDialog.errorMessage(ex);
			}
		}
	};

	/**
	 * ロールバックボタンのアクション
	 */
	private final Action actionRollback = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			if (view.getCurrentJPanelSql().getDataBaseConnection() == null) {
				return;
			}
			try {
				view.getCurrentJPanelSql().getDataBaseConnection().rollback();
				Jdbcacsess2.logger.info("rollback");
			} catch (Exception ex) {
				ShowDialog.errorMessage(ex);
			}
		}
	};

	/**
	 * Aplicationの終了
	 */
	private final Action actionExit = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			// windowClosingイベントを発生させる
			WindowEvent closingEvent = new WindowEvent(view, WindowEvent.WINDOW_CLOSING);
			view.getToolkit().getSystemEventQueue().postEvent(closingEvent);
		}
	};

	/**
	 * GUIのコンポーネント属性を閉じる時に保存する。
	 */
	private final WindowAdapter windowClosing = new WindowAdapter() {
		@Override
		public void windowClosing(WindowEvent e) {
			Jdbcacsess2.logger.info("application termination start:" + e);
			try {
				ViewMain row = config.getEntity(MainView.ViewMain.class);
				view.collectViewMain(row);
				row.save();
				config.close();

				// hierarchyChanged を発生させるためにコンポーネントを削除
				view.removeAll();

			} catch (SQLException ex) {
				ShowDialog.errorMessage(ex);
			}
			Jdbcacsess2.logger.info("application  termination end");
		}
	};

	/**
	 * {@link JPanelSession}のdbコネクションの状態変更
	 */
	private class ChangeConnectionPanelSession implements DataBaseConnectionListener {
		private final JPanelSession jPanelSession;

		ChangeConnectionPanelSession(JPanelSession jPanelSession) {
			this.jPanelSession = jPanelSession;
		}

		@Override
		public void dataBaseConnectionOpened(final DataBaseConnection dbc) {
			Jdbcacsess2.logger.info(dbc + " " + dbc.getUrl());

			// 再接続時なにもしない
			if (jPanelSession.getDataBaseConnection() != null) {
				Jdbcacsess2.logger.info("reopen");
				return;
			}

			jPanelSession.setDataBaseConnection(dbc);
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					// 各種リスナーの登録
					jPanelSession.getJMenuTabbedSqlNew().addActionListener(actionNewPanelSql);
					jPanelSession.getJMenuTabbedSqlClose().addActionListener(actionClosePanelSql);
					jPanelSession.getJMenuTabbedSqlCloseOther().addActionListener(actionCloseOtherPanelSql);

					JXTreeTable treeTable = jPanelSession.getJXTreetableDBObjects();
					treeTable.addTreeWillExpandListener(eventObjectExpand);
					treeTable.addTreeSelectionListener(eventObjectSelection);
					treeTable.addMouseListener(eventObjectsPressed);
					setShortCut(treeTable, KeyEvent.VK_F5, 0, "refresh", actionObjectRefresh);
					setShortCut(treeTable, KeyEvent.VK_ENTER, 0, "enter", actionObjectSelect);
					setShortCut(treeTable, KeyEvent.VK_SPACE, 0, "expand_collapse", actionObjectExpandOrCollapse);

					jPanelSession.getJMenuDBobjectsRefresh().addActionListener(actionObjectRefresh);

					// 画面状態の復元
					view.setViewMainPanelSesion(jPanelSession, getConfigViewMain());

					// tabに追加してから、HierarchyListenerを登録
					JTabbedPane jTabbedPane = view.getJTabbedPaneSession();
					jTabbedPane.setSelectedComponent(jTabbedPane.add(dbc.getConnectName(), jPanelSession));

					// tabcomponentを用意する
					TabComponent tabComponent = new TabComponent(jTabbedPane);
					tabComponent.addButtunActionListener(actionClosePanelSession);

					jTabbedPane.setTabComponentAt(jTabbedPane.indexOfComponent(jPanelSession), tabComponent);
					jPanelSession.addHierarchyListener(eventHierarchyParentChanged);

					// SQL実行履歴のフィルタをjtableのオブジェクト単位に作成
					TableRowSorter<HistryTableModel> sorter = new TableRowSorter<HistryTableModel>(histryTableModel);
					sorter.setRowFilter(RowFilter.regexFilter("^" + dbc.getConnectName() + "$",
					                                          HistryTableModel.FILTERCLOUMN));

					List<RowSorter.SortKey> sortKeys = new ArrayList<RowSorter.SortKey>();
					sortKeys.add(new RowSorter.SortKey(0, SortOrder.DESCENDING));

					// SQL実行履歴パネルを作成し、jtableに、フィルタとカラム幅自動変更を設定する
					JPanelHIstory jPanelHIstory = new JPanelHIstory();
					JTable table = jPanelHIstory.getTable();
					table.setModel(histryTableModel);
					table.setRowSorter(sorter);
					table.getRowSorter().setSortKeys(sortKeys);
					new TableColumnFit(table).setEnableAutoFit(true);

					// TableRowFilterSupport.forTable(table).searchable(true).apply();

					// SQL実行履歴パネルをsqlパネルのtabに追加
					jPanelSession.getJTabbedPaneSql().add(jPanelHIstory, "History", 0);

					jPanelSession.jLabelStatus.setText("NAME=" + dbc.getConnectName() + "   " + "URL=" + dbc.getUrl()
					                                   + "   " + "USER=" + dbc.getUser());

					// SQLパネルの表示
					try {
						createConnectionJpanelSql(dbc);
					} catch (Exception e) {
						ShowDialog.errorMessage(e);
					}
				}
			});

			// DBツリーオブジェクトの初期設定
			try {

				DBObjectsRoot o = new DBObjectsRoot(dbc.getMetaData());
				DBobjectMutableTreeTableNode root = new DBobjectMutableTreeTableNode(o.getDBObjects().get(0), true);
				DefaultTreeTableModel model =
						(DefaultTreeTableModel) jPanelSession.getJXTreetableDBObjects().getTreeTableModel();
				model.setRoot(root);

				jPanelSession.addChildren(root);

				// addChildren内でSwingUtilities.invokeLater してるので、
				// たぶん、ツリーオブジェクトが作成されてから、属性復元が行われる。
				SwingUtilities.invokeLater(new Runnable() {
					@Override
					public void run() {
						// GUIのコンポーネント属性を復元
						view.setTreeTableColumnWidth(jPanelSession, getConfigViewMain());
					}
				});
			} catch (Exception e) {
				ShowDialog.errorMessage(e);
			}

		}

		@Override
		public void dataBaseConnectionClosed(DataBaseConnection dbc) {
			Jdbcacsess2.logger.info(jPanelSession.getDataBaseConnection() + " "
					+ jPanelSession.getDataBaseConnection().getUrl());
		}

		@Override
		public void dataBaseConnectionClosing(DataBaseConnection dbc) {
		}

	}

	/**
	 * {@link JPanelSql}のdbコネクションの状態変更
	 */
	private class ChangeConnectionPanelSql implements DataBaseConnectionListener {
		JPanelSql jPanelSql;

		ChangeConnectionPanelSql(JPanelSql jPanelSql) {
			this.jPanelSql = jPanelSql;
		}

		@Override
		public void dataBaseConnectionOpened(DataBaseConnection dbc) {
			Jdbcacsess2.logger.info(dbc + " " + dbc.getUrl());

			// sessionボタンで、接続/切断を行った時は、接続後処理を行わない。
			if (view.getCurrentJPanelSession().getJTabbedPaneSql().indexOfComponent(jPanelSql) == -1) {
				SwingUtilities.invokeLater(openedWorker);
			}

			ShowDialog.infomationMessage(view, "新しいセッションが開始されました", dbc.getConnectName());
		}

		@Override
		public void dataBaseConnectionClosing(DataBaseConnection dbc) {
		}

		@Override
		public void dataBaseConnectionClosed(DataBaseConnection dbc) {
			Jdbcacsess2.logger.info(dbc + " " + dbc.getUrl());
			ShowDialog.infomationMessage(view, "セッションを終了しました", dbc.getConnectName());
		}

		private final Runnable openedWorker = new Runnable() {
			@Override
			public void run() {
				// 各種リスナーの登録
				jPanelSql.getJToggleButtonExecution().addActionListener(actionExecution);
				jPanelSql.getJToggleButtonConnection().addActionListener(actionConnection);
				jPanelSql.getJButtonContinue().addActionListener(actionContinue);
				jPanelSql.getJButtonCommit().addActionListener(actionCommit);
				jPanelSql.getJButtonRollback().addActionListener(actionRollback);

				Keymap keymap = jPanelSql.getRSyntaxTextAreaSql().getKeymap();
				keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, InputEvent.CTRL_DOWN_MASK),
				                             actionExecutionShortcut);
				keymap.addActionForKeyStroke(KeyStroke.getKeyStroke(KeyEvent.VK_SPACE, InputEvent.CTRL_DOWN_MASK),
				                             actionCodeAssistDisplay);

				// セル移動のショートカットを登録する
				JTableShortcut.setJTableShortcut(jPanelSql.getJTable());

				// JPanel用JTabbedPaneを取得する。
				JTabbedPane jTabbedPane = view.getCurrentJPanelSession().getJTabbedPaneSql();

				// tabtitleを用意する。
				String title = "SQL-" + Integer.toString(view.getCurrentJPanelSession().increaseJTabbedPanelSqlCount());

				// tabにcomponentとabComponentを追加してから、HierarchyListenerを登録
				jTabbedPane.setSelectedComponent(jTabbedPane.add(title, jPanelSql));

				// tabcomponentを用意する。
				TabComponent tabComponent = new TabComponent(jTabbedPane);
				tabComponent.addButtunActionListener(actionClosePanelSql);
				jTabbedPane.setTabComponentAt(jTabbedPane.indexOfComponent(jPanelSql), tabComponent);

				jPanelSql.addHierarchyListener(eventHierarchyParentChanged);

				// 画面状態の復元
				view.setViewMainPanelSql(jPanelSql, getConfigViewMain());

			}
		};
	}

	/**
	 * {@link JPanelSql}のトランザクションの状態変更
	 */
	private static class ChangeTransactionPanelSql implements DataBaseTransactionListener {
		@Override
		public void autoCommitChange(DataBaseConnection dbc, boolean autoCommit) {
			Jdbcacsess2.logger.info(dbc + " " + "autocommit is " + autoCommit);
		}

		@Override
		public void commitEnd(DataBaseConnection dbc) {
			Jdbcacsess2.logger.info(dbc + " " + "commit end ");
		}

		@Override
		public void rollbackEnd(DataBaseConnection dbc) {
			Jdbcacsess2.logger.info(dbc + " " + "rollback end ");
		}
	}

	/**
	 * 新しいSQLパネルを開く
	 */
	private final Action actionNewPanelSql = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			DataBaseConnection dbc = view.getCurrentJPanelSession().getDataBaseConnection();
			try {
				createConnectionJpanelSql(dbc);
			} catch (Exception e1) {
				ShowDialog.errorMessage(e1);
			}
		}
	};

	/**
	 * SQLパネルを閉じる
	 */
	private final Action actionClosePanelSql = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			int index = view.getCurrentJPanelSession().tabbedPaneMousePointer.getFocusedComponentIndex();
			if (index == -1) {
				return;
			}
			JTabbedPane jTabbedPane = view.getCurrentJPanelSession().getJTabbedPaneSql();
			if (jTabbedPane.getComponentAt(index) instanceof JPanelSql) {
				jTabbedPane.remove(index);
			}

		}
	};

	/**
	 * SQLパネル選択中以外を全て閉じる
	 */
	private final Action actionCloseOtherPanelSql = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			JTabbedPane tab = view.getCurrentJPanelSession().getJTabbedPaneSql();

			int index = view.getCurrentJPanelSession().tabbedPaneMousePointer.getFocusedComponentIndex();
			if (index == -1) {
				return;
			}

			Component c = tab.getComponentAt(index);
			for (Component component : new CopyOnWriteArrayList<Component>(tab.getComponents())) {
				if (component instanceof JPanelSql && component != c) {
					tab.remove(component);
				}
			}
		}
	};

	/**
	 * データベースオブジェクトツリーの再構築を行う
	 */
	private final Action actionObjectRefresh = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			JXTreeTable treeTable = view.getCurrentJPanelSession().getJXTreetableDBObjects();

			// ノード削除を行うためにモデルを取得する
			DefaultTreeTableModel treeTableModel = (DefaultTreeTableModel) treeTable.getTreeTableModel();

			// 選択中のツリーパス配列を取得
			TreePath tps[] = treeTable.getTreeSelectionModel().getSelectionPaths();

			for (TreePath tp : tps) {
				// ツリーパスからノードを取得
				DBobjectMutableTreeTableNode node = (DBobjectMutableTreeTableNode) tp.getLastPathComponent();

				Jdbcacsess2.logger.fine("refresh:" + node.getDBObject().getSummary());

				for (MutableTreeTableNode mttn : java.util.Collections.list(node.children())) {
					// ノードの削除は、モデルに対して行う。ノードに対して削除すると例外が発生する。
					treeTableModel.removeNodeFromParent(mttn);
				}

				// キャッシュをクリアし、子ノードを追加する
				node.getDBObject().clearCache();
				view.getCurrentJPanelSession().addChildren(node);
			}
		}
	};

	/**
	 * データベースオブジェクトツリーの展開と折畳アクション
	 */
	private final Action actionObjectExpandOrCollapse = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			JXTreeTable treeTable = view.getCurrentJPanelSession().getJXTreetableDBObjects();
			TreePath[] tps = treeTable.getTreeSelectionModel().getSelectionPaths();
			if (tps == null) {
				return;
			}
			for (TreePath tp : tps) {
				// 展開と折畳を反転させる
				if (treeTable.isExpanded(tp)) {
					treeTable.collapsePath(tp);
				} else {
					treeTable.expandPath(tp);
				}
			}
		}
	};

	/**
	 * データベースオブジェクトツリーのリーフを展開する
	 */
	private final TreeWillExpandListener eventObjectExpand = new TreeWillExpandListener() {
		@Override
		public void treeWillCollapse(TreeExpansionEvent arg0) throws ExpandVetoException {
		}

		@Override
		public void treeWillExpand(TreeExpansionEvent arg0) throws ExpandVetoException {
			DBobjectMutableTreeTableNode n = (DBobjectMutableTreeTableNode) arg0.getPath().getLastPathComponent();
			view.getCurrentJPanelSession().addChildren(n);
		}
	};

	/**
	 * データベースオブジェクトのプロパティを表示する
	 */
	private final TreeSelectionListener eventObjectSelection = new TreeSelectionListener() {
		@Override
		public void valueChanged(TreeSelectionEvent e) {
			DBobjectMutableTreeTableNode node = (DBobjectMutableTreeTableNode) e.getPath().getLastPathComponent();
			DBObject dbObject = node.getDBObject();

			// 出力したいProperty情報を持っている時のみ表示する
			if (dbObject.isPropertyEnable()) {
				JmyTable jTableProperty = view.getCurrentJPanelSession().getJTableProperty();
				jTableProperty.setModel(new PropertyDBObjectsTableModel(dbObject.getProperties()));
			}

			// TABLEカラムの場合は親ノード(=TABLE)に書き換える
			if (dbObject instanceof DBObjectTableColumn) {
				DBobjectMutableTreeTableNode parent = (DBobjectMutableTreeTableNode) node.getParent();
				dbObject = parent.getDBObject();
			}

			// TABLE 系の時のみexportkey情報を表示する
			JmyTable jTable = view.getCurrentJPanelSession().getJTableExportedKey();
			if (dbObject instanceof DBObjectTable) {
				jTable.setModel(new PropertyExportedKeyTableModel(((DBObjectTable) dbObject).getExportedKey()));
			}
		}
	};

	/**
	 * マウスのダブルクリックでデータベースオブジェクトツリーの名称をSQL文に挿入する
	 */
	private final MouseListener eventObjectsPressed = new MouseAdapter() {
		@Override
		public void mouseClicked(MouseEvent e) {
			if (e.getClickCount() != 2) {
				return;
			}

			TreePath tp = view.getCurrentJPanelSession().getJXTreetableDBObjects().getPathForLocation(e.getX(),
			                                                                                          e.getY());
			if (tp == null) {
				return;
			}

			DBObject o = ((DBobjectMutableTreeTableNode) tp.getLastPathComponent()).getDBObject();
			if (o.getName() == null) {
				return;
			}

			JPanelSql jPanelSql = view.getCurrentJPanelSql();
			if (jPanelSql == null) {
				return;
			}

			jPanelSql.insertStringPreviousCaret(o.getName());
		}
	};
	/**
	 * エンターキーでデータベースオブジェクトツリーの名称をSQL文に挿入する
	 */
	private final Action actionObjectSelect = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			TreePath[] tps = ((JXTreeTable) e.getSource()).getTreeSelectionModel().getSelectionPaths();
			if (tps == null) {
				return;
			}

			StringConncat s = new StringConncat(", ");
			for (TreePath tp : tps) {
				DBObject o = (DBObject) ((DBobjectMutableTreeTableNode) tp.getLastPathComponent()).getUserObject();
				if (o.getName() != null) {
					s.append(o.getName());
				}
			}

			view.getCurrentJPanelSql().insertStringPreviousCaret(s.toString());
		}
	};

	/**
	 * コードアシスト画面を表示する
	 */
	private final Action actionCodeAssistDisplay = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			JTextComponent jTextComponent = (JTextComponent) e.getSource();
			try {
				codeAssistView.setLocation(jTextComponent);
				codeAssistView.setModel(new CodeAssistModel());
				codeAssistView.setVisible(true);
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	};

	/**
	 * コードアシストで決定されて文字列をSQL文に挿入する
	 */
	private final Action actionSelectCodeAssist = new AbstractAction() {
		@Override
		public void actionPerformed(ActionEvent e) {
			String value = codeAssistView.getJList().getSelectedValue().getEditorValue();
			if (!value.equals("")) {
				view.getCurrentJPanelSql().insertStringPreviousCaret(value);
			}
			codeAssistView.setVisible(false);
		}
	};

	/**
	 * コンストラクタ
	 * 
	 * @param config
	 */
	public MainController(Config config) {
		this.config = config;

		view = new MainView();
		ShowDialog.setJFrame(view);

		histryTableModel = new HistryTableModel(config);

		codeAssistView = new CodeAssistView(view);
		codeAssistView.setActionSelected(actionSelectCodeAssist);

		// リスナーを登録する
		view.addWindowListener(windowClosing);
		view.getExitMenuItem().addActionListener(actionExit);
		view.getJMenuTabbedSessionNew().addActionListener(actionNewPanelSession);
		view.getJMenuTabbedSessionReNew().addActionListener(actionReNewPanelSession);
		view.getJMenuTabbedSessionClose().addActionListener(actionClosePanelSession);

		// GUIのコンポーネント属性を復元
		view.setViewMain(getConfigViewMain());
		view.setVisible(true);

		// 接続ダイアログをモーダルで表示
		try {
			createConnectionJPanelSession();
		} catch (Exception e2) {
			ShowDialog.errorMessage(e2);
		}
	}

	/**
	 * main view のcommponent属性保持データを検索する
	 */
	private ViewMain getConfigViewMain() {
		try {
			ViewMain[] c = config.getEntityManager().find(MainView.ViewMain.class);
			if (c.length != 0) {
				return c[0];
			}
		} catch (SQLException e1) {
			ShowDialog.errorMessage(e1);
		}
		return null;
	}

	/**
	 * {@link JPanelSession}の生成、DBコネクションリスナーの登録、コネクションのopenを行う。
	 * 
	 * @throws Exception
	 */
	private void createConnectionJPanelSession() throws Exception {
		JPanelSession jPanelSession = new JPanelSession();

		ChangeConnectionPanelSession changeConnectionPanelSession = new ChangeConnectionPanelSession(jPanelSession);

		ConnectController dialog = new ConnectController(config, view);
		dialog.addReauestDataBaseListener(changeConnectionPanelSession);

		dialog.setVisible(true);
	}

	/**
	 * {@link JPanelSql}の生成、DBコネクションリスナーの登録、コネクションのopenを行う。
	 * 
	 * @param dbc
	 * @throws Exception
	 */
	private void createConnectionJpanelSql(DataBaseConnection dbc) throws Exception {
		JPanelSql jpanelSql = new JPanelSql();

		// 既存DB接続の接続先情報で新しいDB接続を作成し、オープンする。
		DataBaseConnection newdbc = new DataBaseConnection(dbc);

		newdbc.addConnectionListener(new ChangeConnectionPanelSql(jpanelSql));
		newdbc.addConnectionListener(jpanelSql);
		newdbc.addTransactionListener(new ChangeTransactionPanelSql());
		newdbc.addTransactionListener(jpanelSql);

		newdbc.open();
	}

	/**
	 * パネルが削除された時に、DB接続を切断する
	 */
	HierarchyListener eventHierarchyParentChanged = new HierarchyListener() {
		@Override
		public void hierarchyChanged(HierarchyEvent e) {
			if ((e.getChangeFlags() & HierarchyEvent.PARENT_CHANGED) != 0) {
				Component c = e.getComponent();
				Jdbcacsess2.logger.info("PARENT_CHANGED:" + e);

				if (c instanceof JPanelSql) {
					JPanelSql jPanelSql = (JPanelSql) c;
					if (jPanelSql.getDataBaseConnection() != null) {
						try {
							jPanelSql.getDataBaseConnection().close();
						} catch (DbConnectIllgalStateException e1) {
							e1.printStackTrace();
						}
					}
				} else if (c instanceof JPanelSession) {
					JPanelSession jPanelSql = (JPanelSession) c;
					if (jPanelSql.getDataBaseConnection() != null) {
						try {
							jPanelSql.getDataBaseConnection().close();
						} catch (DbConnectIllgalStateException e1) {
							e1.printStackTrace();
						}
					}
				}
			}
		}
	};

	/**
	 * inputmap と actionmap を使用し、ショートカットキーを設定する
	 * 
	 * @param jComponent
	 * @param keyCode
	 * @param modifiers
	 * @param key
	 * @param action
	 */
	private void setShortCut(JComponent jComponent, int keyCode, int modifiers, String key, Action action) {
		InputMap inputMap = jComponent.getInputMap();
		ActionMap actionMap = jComponent.getActionMap();

		KeyStroke keyStroke = KeyStroke.getKeyStroke(keyCode, modifiers);
		inputMap.put(keyStroke, key);
		actionMap.put(key, action);
	}

}
