﻿using System;
using System.Collections.Generic;
using System.Text;

namespace SlothLib.LinearAlgebra.Vector
{
	/// <summary>
	/// IVectorからさらに、演算機能を実装したクラス。
	/// </summary>
	/// <remarks>
	/// 
	/// <newpara>[2007-05-01][shirasuna]作成</newpara>
    /// <newpara>[2007-11-12][kabutoya]修正</newpara>
	/// </remarks>
	abstract public class AbstractVector : IVector
	{
		/// <summary>
		/// 加算オペレーター
		/// </summary>
		/// <param name="vector1"></param>
		/// <param name="vector2"></param>
		/// <returns></returns>
		public static IVector operator +(AbstractVector vector1, IVector vector2) {
			return VectorOperator.Add(vector1, vector2);
		}

		/// <summary>
		/// 減算オペレーター
		/// </summary>
		/// <param name="vector1"></param>
		/// <param name="vector2"></param>
		/// <returns></returns>
		public static IVector operator -(AbstractVector vector1, IVector vector2)
		{
			return VectorOperator.Subtract(vector1, vector2);
		}

		/// <summary>
		/// 負オペレーター
		/// </summary>
		/// <param name="vector"></param>
		/// <returns></returns>
		public static IVector operator -(AbstractVector vector) {
			return VectorOperator.Negate(vector);
		}

		/// <summary>
		/// 乗算オペレーター
		/// </summary>
		/// <param name="scalar"></param>
		/// <param name="vector"></param>
		/// <returns></returns>
		public static IVector operator *(double scalar, AbstractVector vector) {
			return VectorOperator.Multiply(scalar, vector);
		}

        /// <summary>
        /// 指定した 2 つのベクトルの値が同一かどうかを判断します。
        /// </summary>
        /// <param name="vector1"></param>
        /// <param name="vector2"></param>
        /// <returns>同一であれば true、さもなければ false</returns>
        public static bool operator ==(AbstractVector vector1, IVector vector2)
        {
            return VectorOperator.Equals(vector1, vector2);
        }

        /// <summary>
        /// 指定した 2 つのベクトルの値が同一かどうかを判断します。
        /// </summary>
        /// <param name="vector1"></param>
        /// <param name="vector2"></param>
        /// <returns>同一であれば false、さもなければ true</returns>
        public static bool operator !=(AbstractVector vector1, IVector vector2)
        {
            return !VectorOperator.Equals(vector1, vector2);
        }

		/// <summary>
		/// ベクトルの次元
		/// </summary>
		abstract public int Dimension { get; }

		/// <summary>
		/// 指定された次元の値を取得する
		/// </summary>
		/// <param name="dimension"></param>
		/// <returns></returns>
		abstract public double this[int dimension]
		{
			get;
			set;
		}

		/// <summary>
		/// 配列によるベクトル取得
		/// </summary>
		/// <returns></returns>
		public abstract double[] GetPlainVector();

		/// <summary>
		/// クローン
		/// </summary>
		/// <returns></returns>
		public abstract object Clone();

        /// <summary>
        /// このベクトルと、指定した SlothLib.LinearAlgebra.IVector オブジェクトの値が同一かどうかを判断します。
        /// </summary>
        /// <param name="value">SlothLib.LinearAlgebra.IVector。</param>
        /// <returns></returns>
        public abstract bool Equals(IVector value);

        /// <summary>
        /// このベクトルと、指定した System.Object オブジェクトの値が同一かどうかを判断します。
        /// </summary>
        /// <param name="obj">System.Object。</param>
        /// <returns></returns>
        public override bool Equals(object obj)
        {
            IVector other = obj as IVector;

            if (ReferenceEquals(other, null))
            {
                return false;
            }

            return this.Equals(other);
        }

        /// <summary>
        /// このベクトルのハッシュコードを返します。
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            int hashCode = this.Dimension;

            for (int i = 0; i < this.Dimension; i++)
            {
                hashCode = hashCode ^ this[i].GetHashCode();
            }

            return hashCode;
        }
    }
}
