﻿using System;
using System.Collections.Generic;
using System.Text;

namespace SlothLib.LinearAlgebra.FeatureVector
{
	/// <summary>
	/// Inverse Document Frequency の重みを持つベクトルを作成する
	/// Value = Log(VectorCount / Value + addValue, baseValue)
	/// addValue が省略された場合は1.0
	/// baseValue が省略された場合は2.0が用いられる
	/// </summary>
	public class InverseDFVector<T> : BasicVector<T>, IDocumentFrequencyVector<T>
	{

		// もとになったDFベクトルの文書数
		private int vectorCount;

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public InverseDFVector(double baseValue, double addValue, IDocumentFrequencyVector<T> dfVector)
		{
			Calculate(baseValue, addValue, dfVector);
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public InverseDFVector(double baseValue, double addValue, IEnumerable<IVector<T>> vectors)
		{
			Calculate(baseValue, addValue, vectors);
		}

/*		
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public InverseDFVector(double baseValue, double addValue, params IVector<T>[] vectors)
		{
			Calculate(baseValue,addValue,vectors);
		}
 */
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public InverseDFVector(IDocumentFrequencyVector<T> dfVector)
			: this(2.0, 1.0, dfVector)
		{ }

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public InverseDFVector(IEnumerable<IVector<T>> vectors)
			: this(2.0, 1.0, vectors)
		{ }

/*
		/// <summary>
		/// コンストラクタ
		/// </summary>
		public InverseDFVector(params IVector<T>[] vectors)
			: this(2.0, 1.0, vectors)
		{ }
 */

		private void Calculate(double baseValue, double addValue, IEnumerable<IVector<T>> vectors)
		{
			Calculate(baseValue, addValue, new DocumentFrequencyVector<T>(vectors));
		}

		private void Calculate(double baseValue, double addValue, IDocumentFrequencyVector<T> dfVector)
		{
			foreach (T key in dfVector.Keys)
			{
				this[key] = Math.Log(dfVector.VectorCount / dfVector[key] + addValue, baseValue);
			}
			this.vectorCount = dfVector.VectorCount;
		}


		/// <summary>
		/// ベクトルの構成数
		/// </summary>
		public int VectorCount
		{
			get
			{
				return this.VectorCount;
			}
		}

	}
}
