﻿using System;
using System.Collections.Generic;
using System.Text;

namespace SlothLib.NLP
{
	/// <summary>
	/// フィルタのリスト
	/// </summary>
	public class MorphemeFilterList : IMorphemeFilter, IMorphemeToStringFilter, IStringFilter
	{

		#region privateフィールド

		private List<IMorphemeFilter> morphemeFilterList;
		private IMorphemeToStringFilter morphemeToStringFilter;
		private List<IStringFilter> stringFilterList;

		#endregion

		#region コンストラクタ

		/// <summary>
		/// コンストラクタ
		/// </summary>
		public MorphemeFilterList()
		{
			this.morphemeFilterList = new List<IMorphemeFilter>();
			this.morphemeToStringFilter = null;
			this.stringFilterList = new List<IStringFilter>();
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="morphemeFilters">入力が形態素で出力も形態素のフィルタのリスト</param>
		public MorphemeFilterList(IEnumerable<IMorphemeFilter> morphemeFilters)
		{
			this.morphemeFilterList = new List<IMorphemeFilter>(morphemeFilters);
			this.morphemeToStringFilter = null;
			this.stringFilterList = new List<IStringFilter>();
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="morphemeFilters">入力が形態素で出力も形態素のフィルタのリスト</param>
		public MorphemeFilterList(params IMorphemeFilter[] morphemeFilters)
		{
			this.morphemeFilterList = new List<IMorphemeFilter>(morphemeFilters);
			this.morphemeToStringFilter = null;
			this.stringFilterList = new List<IStringFilter>();
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="morphemeToStringFilter">入力が形態素で出力が文字列フィルタ</param>
		public MorphemeFilterList(IMorphemeToStringFilter morphemeToStringFilter)
		{
			this.morphemeFilterList = new List<IMorphemeFilter>();
			this.morphemeToStringFilter = morphemeToStringFilter;
			this.stringFilterList = new List<IStringFilter>();
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="stringFilters">入力が文字列で出力も文字列のフィルタのリスト</param>
		public MorphemeFilterList(IEnumerable<IStringFilter> stringFilters)
		{
			this.morphemeFilterList = new List<IMorphemeFilter>();
			this.morphemeToStringFilter = null;
			this.stringFilterList = new List<IStringFilter>(stringFilters);
		}

		/// <summary>
		/// コンストラクタ
		/// </summary>
		/// <param name="morphemeFilters">入力が形態素で出力も形態素のフィルタのリスト</param>
		/// <param name="morphemeToStringFilter">入力が形態素で出力が文字列フィルタ</param>
		/// <param name="stringFilters">入力が文字列で出力も文字列のフィルタのリスト</param>
		public MorphemeFilterList(IEnumerable<IMorphemeFilter> morphemeFilters,
						  IMorphemeToStringFilter morphemeToStringFilter,
						  IEnumerable<IStringFilter> stringFilters)
		{
			this.morphemeFilterList = new List<IMorphemeFilter>(morphemeFilters);
			this.morphemeToStringFilter = morphemeToStringFilter;
			this.stringFilterList = new List<IStringFilter>(stringFilters);

		}
		#endregion

		#region プロパティ

		/// <summary>
		/// MorphemeToStringFilterが登録されているかどうかを取得
		/// </summary>
		public bool HasMorphemeToStringFilter
		{
			get
			{
				if (this.morphemeToStringFilter == null)
				{
					return false;
				}
				return true;
			}
		}

		#endregion

		#region Add

		/// <summary>
		/// フィルタを追加する
		/// </summary>
		/// <param name="morphemeFilter">入力が形態素で出力も形態素のフィルタ</param>
		public void Add(IMorphemeFilter morphemeFilter)
		{
			this.morphemeFilterList.Add(morphemeFilter);
		}

		/// <summary>
		/// フィルタを追加する
		/// </summary>
		/// <param name="morphemeToStringFilter">入力が形態素で出力が文字列フィルタ</param>
		public void Add(IMorphemeToStringFilter morphemeToStringFilter)
		{
			this.morphemeToStringFilter = morphemeToStringFilter;
		}

		/// <summary>
		/// フィルタを追加する
		/// </summary>
		/// <param name="stringFilter">入力が文字列で出力も文字列のフィルタ</param>
		public void Add(IStringFilter stringFilter)
		{
			this.stringFilterList.Add(stringFilter);
		}

		#endregion

		#region DoMorphemeFilter

		/// <summary>
		/// 入力が形態素で出力も形態素のフィルタを実行する
		/// </summary>
		/// <param name="morpheme">フィルタを適用する形態素</param>
		/// <returns>フィルタ適用後の形態素</returns>
		public IMorpheme DoMorphemeFilter(IMorpheme morpheme)
		{
			IMorpheme resultMorpheme = morpheme;
			foreach (IMorphemeFilter filter in this.morphemeFilterList)
			{
				resultMorpheme = filter.DoFilter(resultMorpheme);
				if (resultMorpheme == null)
				{
					return null; //結果がnullになったらその場でnullを返す
				}
			}
			return resultMorpheme;
		}

		/// <summary>
		/// 入力が形態素で出力も形態素のフィルタを実行する
		/// </summary>
		/// <param name="morphemes">フィルタを適用する形態素のリスト</param>
		/// <returns>フィルタ適用後の形態素の配列</returns>
		public IMorpheme[] DoMorphemeFilter(IEnumerable<IMorpheme> morphemes)
		{
			IMorpheme[] resultMorphemes = new List<IMorpheme>(morphemes).ToArray();
			foreach (IMorphemeFilter filter in this.morphemeFilterList)
			{
				resultMorphemes = filter.DoFilter(resultMorphemes);
			}
			return resultMorphemes;
		}

		#endregion

		#region DoStringFilter

		/// <summary>
		/// 入力が文字列で出力も文字列のフィルタを実行する
		/// </summary>
		/// <param name="str">フィルタを適用する文字列</param>
		/// <returns>フィルタ適用後の文字列</returns>
		public string DoStringFilter(string str)
		{
			string resultString = str;
			foreach (IStringFilter filter in this.stringFilterList)
			{
				resultString = filter.DoFilter(resultString);
				if (resultString == null)
				{
					return null; //もし結果がnullになったらその場でnullを返す
				}
			}
			return resultString;
		}
		/// <summary>
		/// 入力が文字列で出力も文字列のフィルタを実行する
		/// </summary>
		/// <param name="strings">フィルタを適用する文字列のリスト</param>
		/// <returns>フィルタ適用後の文字列の配列</returns>
		public string[] DoStringFilter(IEnumerable<string> strings)
		{
			string[] resultStrings = new List<string>(strings).ToArray();
			foreach (IStringFilter filter in this.stringFilterList)
			{
				resultStrings = filter.DoFilter(resultStrings);
			}
			return resultStrings;
		}

		#endregion

		#region DoMorphemeToStringFilter

		//全てのフィルターを通す
		/// <summary>
		/// 登録されている全てのフィルタを適用する
		/// </summary>
		/// <param name="morpheme">フィルタを適用する形態素</param>
		/// <returns>フィルタ適用後の文字列</returns>
		public string DoMorphemeToStringFilter(IMorpheme morpheme)
		{
			if (this.morphemeToStringFilter == null)
			{
				throw new Exception("IMorphemeToStringFilterがありません");
			}
			morpheme = DoMorphemeFilter(morpheme);
			if (morpheme == null) return null;
			string str = this.morphemeToStringFilter.DoFilter(morpheme);
			if (str == null) return null;
			return DoStringFilter(str);
		}

		/// <summary>
		/// 登録されている全てのフィルタを適用する
		/// </summary>
		/// <param name="morphemes">フィルタを適用する形態素のリスト</param>
		/// <returns>フィルタ適用後の文字列の配列</returns>
		public string[] DoMorphemeToStringFilter(IEnumerable<IMorpheme> morphemes)
		{
			if (this.morphemeToStringFilter == null)
			{
				throw new Exception("IMorphemeToStringFilterがありません");
			}
			morphemes = DoMorphemeFilter(morphemes);
			string[] strings = this.morphemeToStringFilter.DoFilter(morphemes);
			return DoStringFilter(strings);
		}

		#endregion


		#region インターフェースのメンバ

		#region IMorphemeFilter メンバ

		IMorpheme IMorphemeFilter.DoFilter(IMorpheme morpheme)
		{
			return DoMorphemeFilter(morpheme);
		}

		IMorpheme[] IMorphemeFilter.DoFilter(IEnumerable<IMorpheme> morphemes)
		{
			return DoMorphemeFilter(morphemes);
		}

		#endregion

		#region IStringFilter メンバ

		string IStringFilter.DoFilter(string str)
		{
			return DoStringFilter(str);
		}

		string[] IStringFilter.DoFilter(IEnumerable<string> strings)
		{
			return DoStringFilter(strings);
		}

		#endregion

		#region IMorphemeToStringFilter メンバ

		string IMorphemeToStringFilter.DoFilter(IMorpheme morpheme)
		{
			return DoMorphemeToStringFilter(morpheme);
		}

		string[] IMorphemeToStringFilter.DoFilter(IEnumerable<IMorpheme> morphemes)
		{
			return DoMorphemeToStringFilter(morphemes);
		}

		#endregion

		#endregion
	}
}
