﻿// [ohshima][2007-03-04] 修正
// [tyamamot][2007-05-07] 修正

using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Web;
using System.Net;
using System.IO;

namespace SlothLib.Web.Search
{
	/// <summary>
	/// Yahoo!動画検索Webサービス
	/// http://developer.yahoo.co.jp/search/video/V1/videoSearch.html
	/// </summary>
	public class YahooJpVideoSearch : IVideoSearch
	{

		#region プライベートフィールド

		/// <summary>
		/// アプリケーションID
		/// </summary>
		private string applicationID;

		private SearchType type;
		private SearchFormat format;
		private bool adultOk;
		private string[] site;
        private WebProxy proxy;

        //結果が繰り返さないようにするために必要
        Dictionary<string, bool> alreadyGotURL = new Dictionary<string, bool>();

        // 空のXMLのためのループ回数カウンタ
        private int loopCount = 0;

        #endregion


		#region コンストラクタ

		/// <summary>
		/// コンストラクタ　アプリケーションIDを指定する
		/// </summary>
		/// <param name="applicationID">プログラムで用いるアプリケーションID</param>

		public YahooJpVideoSearch(string applicationID)
		{
			this.applicationID = applicationID;

			this.type = SearchType.all;
			this.format = SearchFormat.any;
			this.adultOk = false;
			this.site = null;
		}

		#endregion


		#region public メソッド

		#region DoSearch

		/// <summary>
		/// Yahoo!動画検索を実行する
		/// </summary>
		/// <param name="query">検索クエリ</param>
		/// <param name="resultNum">検索結果取得数</param>
		/// <returns>YahooVideoSearchResultの検索結果</returns>
		public YahooJpVideoSearchResult DoSearch(string query, int resultNum)
		{
            return DoSearch(query, resultNum, 1);
		}

		/// <summary>
		/// Yahoo!動画検索を実行する
		/// </summary>
		/// <param name="query">検索クエリ</param>
		/// <param name="resultNum">検索結果取得数</param>
		/// <param name="start">検索開始の先頭位置　デフォルトは1</param>
        /// <returns>YahooVideoSearchResultの検索結果</returns>
		public YahooJpVideoSearchResult DoSearch(string query, int resultNum, int start)
		{
            return DoSearchOver(query, resultNum, start);
		}

		#endregion

		/// <summary>
		/// 検索HIT数のみを取得するメソッド
		/// </summary>
		/// <param name="query">検索クエリ</param>
		/// <returns>HIT数</returns>
        public long GetTotalNumber(string query)
        {
            return DoSearch(query, 10).TotalResultsAvailable;
        }
		#endregion


		#region private メソッド

		/// <summary>
		/// 50件以内の検索結果を行う
		/// </summary>
		/// <param name="query">検索クエリ</param>
		/// <param name="results">検索結果取得数</param>
		/// <param name="start">検索開始の先頭位置</param>
		/// <returns>YahooVideoSearchResult型の検索結果</returns>
		private YahooJpVideoSearchResult DoSearchOriginal(string query, int results, int start)
		{
			string requestURL = MakeRequestURL(query, this.type, results, start, this.format, this.adultOk, this.site);
			//System.Diagnostics.Debug.WriteLine(requestURL);
			XmlDocument xmlDoc = new XmlDocument();

			HttpStatusCode statusCode;

			HttpWebRequest req = (HttpWebRequest)WebRequest.Create(requestURL);
            if (this.proxy != null)
            {
                req.Proxy = proxy;
            }
			using (HttpWebResponse res = (HttpWebResponse)req.GetResponse())
			using (Stream st = res.GetResponseStream())
			{
				xmlDoc.Load(st);
				statusCode = res.StatusCode;
			}

			// ルートの要素を取得
			XmlElement xmlRoot = xmlDoc.DocumentElement;

			//Errorが帰ってきた場合
			//if (xmlRoot.Name.Equals("Error"))
			if (statusCode != HttpStatusCode.OK)
			{
				// 例外メッセージを取得
				string errorMessage = GetElementString(xmlRoot.GetElementsByTagName("Message"));
				// 例外を投げる。
				this.ThrowException(statusCode, errorMessage);
			}

            // [inagawa] 空のXMLが返ってくる問題に暫定的対処【要検討】
            if (string.IsNullOrEmpty(xmlRoot.InnerXml) && string.IsNullOrEmpty(xmlRoot.GetAttribute("totalResultsAvailable")))
            {
                if (++loopCount > 5)
                {
                    // ループ回数が5回を超えたら例外を投げる
                    throw new Exception("Yahoo!から適切なXMLを取得できません。");
                }
                System.Diagnostics.Debug.WriteLine("returned XML is empty");
                return DoSearchOriginal(query, results, start);
            }

            // <Result>要素のtotalResultsAvailableの属性値を取得
			long totalResultsAvailable = long.Parse(xmlRoot.GetAttribute("totalResultsAvailable"));
			int totalResultsReturned = int.Parse(xmlRoot.GetAttribute("totalResultsReturned"));
			int firstResultPosition = int.Parse(xmlRoot.GetAttribute("firstResultPosition"));

			// Resultを入れていく
			List<YahooJpVideoElement> ResultElementList = new List<YahooJpVideoElement>();

            //[inagawa]もし結果の数が0だったら空の結果リストを返す【要検討】
            if (totalResultsReturned == 0)
            {
                return new YahooJpVideoSearchResult(query, totalResultsAvailable, totalResultsReturned, firstResultPosition, ResultElementList.ToArray());
            }

            XmlNodeList xmlResultList = xmlRoot.GetElementsByTagName("Result");
			int rank = 1;

            string firstURL = GetElementString(((XmlElement)xmlResultList[0]).GetElementsByTagName("Url"));
            if (alreadyGotURL.ContainsKey(firstURL))
            {
                return new YahooJpVideoSearchResult(query, 0, 0, -1, new YahooJpVideoElement[0]);
            }
            else
            {
                alreadyGotURL.Add(firstURL, true);
            }

			foreach (XmlElement xmlResult in xmlResultList)
			{
				string title = GetElementString(xmlResult.GetElementsByTagName("Title"));
				string summary = GetElementString(xmlResult.GetElementsByTagName("Summary"));
				string url = GetElementString(xmlResult.GetElementsByTagName("Url"));
				string clickUrl = GetElementString(xmlResult.GetElementsByTagName("ClickUrl"));
				string refererUrl = GetElementString(xmlResult.GetElementsByTagName("RefererUrl"));
				string fileSize = GetElementString(xmlResult.GetElementsByTagName("FileSize"));
				string fileFormat = GetElementString(xmlResult.GetElementsByTagName("FileFormat"));
				string height = GetElementString(xmlResult.GetElementsByTagName("Height"));
				string width = GetElementString(xmlResult.GetElementsByTagName("Width"));
				string duration = GetElementString(xmlResult.GetElementsByTagName("Duration"));
				string streaming = GetElementString(xmlResult.GetElementsByTagName("Streaming"));
				string channels = GetElementString(xmlResult.GetElementsByTagName("Channels"));
				string restrictions = GetElementString(xmlResult.GetElementsByTagName("Restrictions"));
				string thumbnailUrl = string.Empty;
				string thumbnailHeight = string.Empty;
				string thumbnailWidth = string.Empty;
				string publisher = GetElementString(xmlResult.GetElementsByTagName("Publisher"));
				string copyright = GetElementString(xmlResult.GetElementsByTagName("Copyright"));

				XmlNodeList xmlCacheNode = xmlResult.GetElementsByTagName("Thumbnail");
				if (xmlCacheNode != null)
				{
					foreach (XmlElement xmlCacheElement in xmlCacheNode)
					{
						thumbnailUrl = GetElementString(xmlCacheElement.GetElementsByTagName("Url"));
						thumbnailHeight = GetElementString(xmlCacheElement.GetElementsByTagName("Height"));
						thumbnailWidth = GetElementString(xmlCacheElement.GetElementsByTagName("Width"));
					}
				}
				YahooJpVideoElement result
					= new YahooJpVideoElement(rank, title, summary, url, clickUrl, refererUrl, fileSize, fileFormat,
					height, width, duration, channels, streaming, thumbnailUrl, thumbnailHeight, thumbnailWidth, publisher
					, restrictions, copyright);
				ResultElementList.Add(result);
				rank++;

			}

			return new YahooJpVideoSearchResult(query, totalResultsAvailable, totalResultsReturned, firstResultPosition, ResultElementList.ToArray());
		}

		/// <summary>
		/// 50件以上の検索を行う
		/// </summary>
		/// <param name="query">検索クエリ</param>
		/// <param name="results">検索結果取得数</param>
		/// <param name="start">検索開始の先頭位置</param>
		/// <returns>YahooJpVideoSearchResult型の検索結果</returns>
		private YahooJpVideoSearchResult DoSearchOver(string query,int results, int start)
		{
			int i;
			int loop = (results - 1) / 50;
            //List<ISearchResultElement> result = new List<ISearchResultElement>();
            List<YahooJpVideoElement> result = new List<YahooJpVideoElement>();
            long totalResultsAvailable = 0;
			YahooJpVideoSearchResult r;

			for (i = 0; i < loop; i++)
			{
                this.loopCount = 0;
                r = DoSearchOriginal(query, 50, i * 50 + 1);
                if (r.FirstResultPosition == -1) { break; }
                totalResultsAvailable = r.TotalResultsAvailable;
                result.AddRange(r.ResultElements);
			}
            this.loopCount = 0;
            r = DoSearchOriginal(query, results - (loop * 50), loop * 50 + 1);
            //result.AddRange(r);
            result.AddRange(r.ResultElements);
            if (r.FirstResultPosition == -1)
            {
            }
            else
            {
                totalResultsAvailable = r.TotalResultsAvailable;
            }

			return new YahooJpVideoSearchResult(query, totalResultsAvailable, result.Count, 1, result.ToArray());
		}

		#endregion


		#region 雑用メソッド


		/// <summary>
		/// 例外を投げる
		/// </summary>
		/// <param name="errorCode"></param>
		/// <param name="errorMessage"></param>
		private void ThrowException(HttpStatusCode errorCode, string errorMessage)
		{
			switch (errorCode)
			{
				case HttpStatusCode.BadRequest: // 400
					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.BadRequest, errorMessage);
				case HttpStatusCode.Forbidden: // 403
					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.Forbidden, errorMessage);
				case HttpStatusCode.ServiceUnavailable: // 503
					throw new YahooJpSearchException(YahooJpSearchException.HttpCode.ServiceUnavailable, errorMessage);
				default:
					throw new Exception("YahooWebSearchで想定外のHTTPエラーが発生しました。（エラーコード: " + (int)errorCode + "）" + errorMessage);

			}
		}

		/// <summary>
		/// XmlNodeListの初めのノードのテキストを取得する
		/// </summary>
		/// <param name="nodeList">XmlNodeList</param>
		/// <returns>XmlNodeListの初めのノードのInnerText
		///          XmlNodeListが空であれば空文字列を返す</returns>
		private string GetElementString(XmlNodeList nodeList)
		{
			if (nodeList.Count == 0)
			{
				return string.Empty;
			}
			else
			{
				return nodeList[0].InnerText;
			}
		}

		/// <summary>
		/// リクエストURLを作成する
		/// </summary>
		/// <param name="query">検索クエリ</param>
		/// <param name="type">指定検索の種類</param>
		/// <param name="results">検索結果取得数</param>
		/// <param name="start">検索開始の先頭位置</param>
		/// <param name="format">検索するファイルの種類　デフォルトはany</param>
		/// <param name="adultOk">アダルトコンテンツの検索結果を含めるかどうか　デフォルトはfalse</param>
		/// <param name="site">検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます
		/// デフォルトはnull</param>
		/// <returns>URL</returns>
		private string MakeRequestURL(string query, SearchType type, int results, int start,
			SearchFormat format, bool adultOk, string[] site)
		{
			string strType = string.Empty;
			string strFormat = string.Empty;
			string strAdult = string.Empty;
			string strSite = string.Empty;

			switch (type)
			{
				case SearchType.all:
					break;
				case SearchType.any:
					strType = "&type=any";
					break;
				case SearchType.phrase:
					strType = "&type=phrase";
					break;
			}

			switch (format)
			{
				case SearchFormat.any:
					break;
				case SearchFormat.avi:
					strFormat = "&format=avi";
					break;
				case SearchFormat.flash:
					strFormat = "&format=flash";
					break;
				case SearchFormat.mpeg:
					strFormat = "&format=mpeg";
					break;
				case SearchFormat.msmedia:
					strFormat = "&format=msmedia";
					break;
				case SearchFormat.quicktime:
					strFormat = "&format=quicktime";
					break;
				case SearchFormat.realmedia:
					strFormat = "&format=realmedia";
					break;
			}

			if (adultOk == true)
			{
				strAdult = "&adulat_ok=1";
			}

			if (site != null)
			{
				if (site.Length > 30)
				{
					throw new ArgumentException("siteに指定できるドメインは30個までです", "site");
				}
				else
				{
					foreach (string s in site)
					{
						strSite += "&site=" + s;
					}
				}
			}

			string requestURL = "http://api.search.yahoo.co.jp/VideoSearchService/V1/videoSearch?"
			+ "appid=" + this.applicationID
			+ "&query=" + System.Web.HttpUtility.UrlEncode(query, Encoding.UTF8)
			+ strType
			+ "&results=" + results.ToString()
			+ "&start=" + start.ToString()
			+ strFormat
			+ strAdult
			+ strSite;
			Uri url = new Uri(requestURL);
			return url.AbsoluteUri;
		}

		#endregion


		#region 引数で指定するための列挙型

		/// <summary>
		/// 指定検索の種類
		/// </summary>
		public enum SearchType
		{
			/// <summary>
			/// デフォルト
			/// </summary>
			all,
			/// <summary>
			/// 
			/// </summary>
			any,
			/// <summary>
			/// 
			/// </summary>
			phrase
		}

		/// <summary>
		/// 検索するファイルの種類 デフォルトはany
		/// </summary>
		public enum SearchFormat
		{
			/// <summary>
			/// デフォルト
			/// </summary>
			any,
			/// <summary>
			/// 
			/// </summary>
			avi,
			/// <summary>
			/// 
			/// </summary>
			flash,
			/// <summary>
			/// 
			/// </summary>
			mpeg,
			/// <summary>
			/// 
			/// </summary>
			msmedia,
			/// <summary>
			/// 
			/// </summary>
			quicktime,
			/// <summary>
			/// 
			/// </summary>
			realmedia
		}

		#endregion


		#region プロパティ

		/// <summary>
		/// 指定検索の種類 デフォルトはall
		/// </summary>
		public SearchType Type
		{
			set
			{
				this.type = value;
			}
		}

		/// <summary>
		/// 検索するファイルの種類　デフォルトはany
		/// </summary>
		public SearchFormat Format
		{
			set
			{
				this.format = value;
			}
		}

		/// <summary>
		/// アダルトコンテンツの検索結果を含めるかどうかを指定します。デフォルトはfalse
		/// </summary>
		public bool AdultOk
		{
			set
			{
				this.adultOk = value;
			}
		}

		/// <summary>
		/// >検索するドメイン（例えば www.yahoo.co.jp）を制限します。30ドメインまで指定することができます。デフォルトはnull
		/// </summary>
		public string[] Site
		{
			set
			{
				this.site = value;
			}
		}

        /// <summary>
        /// プロクシを取得・設定する。
        /// </summary>
        public string Proxy
        {
            set
            {
                if (string.IsNullOrEmpty(value))
                {
                    this.proxy = null;
                }
                else
                {
                    this.proxy = new WebProxy(value);
                }
            }
            get
            {
                return this.proxy.Address.AbsoluteUri;
            }
        }


		#endregion


		#region IVideoSearch メンバ

		IVideoSearchResult IVideoSearch.DoSearch(string query, int resultNum)
		{
			return this.DoSearch(query, resultNum);
		}

		#endregion


		#region ISearch メンバ

		ISearchResult ISearch.DoSearch(string query, int resultNum)
		{
			return this.DoSearch(query, resultNum);
		}

		#endregion

	}
}
