//using System;
//using System.Collections.Generic;
//using System.Text;
//using SlothLib.External.Text;

package slothLib.text;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import slothLib.SlothLibException;
import slothLib.portUtil.PortUtil;

	/**
	 * Javaの自動コード判別機能を利用して文字コード判別、変換を行う。
	 * 元の文字コードが分からないのが難点．使わないと言うことはないだろうか． 
	 */
	public class AutoDetectConverter implements IEncodingConverter
	{

		/**
		 * コンストラクタ
		 */
		public AutoDetectConverter()
		{ }

		/**
		 * テキストファイルの文字コード変換を行う。
		 * @param textFilePath テキストファイルのパス
		 */
		public EncodingConverterResult convertTextFile(String textFilePath)
		throws SlothLibException
		{
			try {
				InputStream is = new FileInputStream(textFilePath);
				int len = (int)(new File(textFilePath)).length();
				byte [] buffer = new byte[len];
				PortUtil.readFully(is, buffer, 0, len);
				return convertByteArray(buffer);
			} catch (IOException e) {
				throw new SlothLibException(e);
			}
		}

		/**
		 * バイト列の文字コード変換を行う。
		 * @param byteArray バイト列
		 */
		public EncodingConverterResult convertByteArray(byte [] byteArray)
		throws SlothLibException
		{
			try {
				return convertByteArray(new ByteArrayInputStream(byteArray));
			} catch (IOException e) {
				throw new SlothLibException(e);
			}
		}

		private EncodingConverterResult convertByteArray(InputStream is) 
		throws IOException
		{
			is.mark(3000);
			String result = null;
			boolean isUTF8 = isUTF8Sequence(is, 3000); 
			System.out.println (isUTF8);
			is.reset();
			if (isUTF8)
				result = PortUtil.readFully(is, "UTF-8");
			else
				result = PortUtil.readFully(is, "JISAutoDetect");
			return new EncodingConverterResult(result, "unknown");
		}
		
		
		private boolean isUTF8Sequence(InputStream is, int len) throws IOException{
			int counter = 0;
			int sjisCounter = 0;
			int sjisPoint = 0;
			for (int i = 0; i < len; i++){
				int ch = (int)is.read();
				if (ch < 0) return true;
				if (sjisCounter == 0 && ch == 0x1B)
					sjisCounter++;
				else if (sjisCounter == 1 && (ch == 0x24 || ch == 0x28))
					sjisCounter++;
				else if (sjisCounter == 2 && (ch == 0x42)){
					sjisCounter = 0;
					if (++sjisPoint >= 2)
						return false;      // looks more like SJIS than UTF8 
				}
				
				if (counter == 0) {
					if ((ch & 0x80) == 0x00)
						continue;
					if ((ch & 0xE0) == 0xC0){
						counter = 1;
						continue;
					}
					if ((ch & 0xF0) == 0xE0) {
						counter = 2;
						continue;
					}
					if ((ch & 0xF8) == 0xF0) {
						counter = 3;
						continue;
					}
					if ((ch & 0xFC) == 0xF8) {
						counter = 4;
						continue;
					}
					if ((ch & 0xFE) == 0xFC) {
						counter = 5;
						continue;
					}
					return false;
				} else {
					if ((ch & 0xC0) == 0xF0) {
						counter--;
						continue;
					}
				}
			}
			return true;
		}

	}
