//using System;
//using System.Collections.Generic;
//using System.Text;
//using SlothLib.Web.Search;

//using SlothLib.Web.Search.com.msn.search.soap;

package slothLib.web.search;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import slothLib.SlothLibException;
import slothLib.portUtil.PortUtil;

import com.microsoft.schemas.msnsearch._2005._09.fex.MSNSearchServiceStub;

    /**
     * LiveSearchを行うクラス
     * http://msdn2.microsoft.com/en-us/library/bb266180.aspx
     */
    public class LiveWebSearch implements IWebSearch
    {

        // privateフィールド

        private String appID;
        private SearchFlags searchFlags = SearchFlags.None;
        private SafeSearchOptions safeSearchOptions = SafeSearchOptions.Moderate;
        private Language language = Language.Japanese;
        private String fileType = "";

        
        public static final String KEY_PROP = "slothlib.web.search.LiveWebSearch.key";
        static {slothLib.SlothLib.init();}


        // コンストラクタ

        /**
         * コンストラクタ
         * @param appID アプリケーションID
         */
        public LiveWebSearch(String appID)
        {
            this.appID = appID;
        }

        /**
         * コンストラクタ 
         * プロパティ slothlib.web.search.LiveWebSearch.key の値をキーとする．
         */
        public LiveWebSearch() throws WebSearchException
        {
        	String tmp = System.getProperty(KEY_PROP);
        	if (tmp == null)
        		throw new WebSearchException("cannot find property: " + KEY_PROP);
        	this.appID = tmp.trim();
        }
        
        


        // publicメソッド

        /**
         * LiveSearch検索を行う
         * @param query 検索クエリー
         * @param resultNum 検索結果総数
         * @return LiveWebSearch型の検索結果
         */
        public LiveWebSearchResult doSearch(String query, int resultNum) throws WebSearchException
        {
            try {
				return this.doSearchOver(query, resultNum);
			} catch (RemoteException e) {
				throw new WebSearchException(e);
			}
        }

//        //#region IWebSearch メンバ
//
//        IWebSearchResult IWebSearch.doSearch(String query, int resultNum)
//        {
//            return this.DoSearch(query, resultNum);
//        }
//
//        //#endregion
//
//        //#region ISearch メンバ
//
//        ISearchResult ISearch.DoSearch(String query, int resultNum)
//        {
//            return this.DoSearch(query, resultNum);
//        }





        // private関数

        private MSNSearchServiceStub.SearchFlags createSearchFlags(MSNSearchServiceStub.SearchFlags_type0 type0)
        {
        	MSNSearchServiceStub.SearchFlags flags;
        	flags = new MSNSearchServiceStub.SearchFlags();
        	flags.setSearchFlags_type0(new MSNSearchServiceStub.SearchFlags_type0[]{type0});
        	return flags;
        }
        private MSNSearchServiceStub.ResultFieldMask createMask(MSNSearchServiceStub.ResultFieldMask_type0 type0)
        {
        	MSNSearchServiceStub.ResultFieldMask mask = 
        		new MSNSearchServiceStub.ResultFieldMask();
        	mask.setResultFieldMask_type0(new MSNSearchServiceStub.ResultFieldMask_type0[]{type0});
        	return mask;
        }
        
        private LiveWebSearchResult doSearchOriginal(String query, int resultNum, int offset, int trueResultNum) throws RemoteException
        {
            MSNSearchServiceStub searchService = new MSNSearchServiceStub();
            MSNSearchServiceStub.SearchRequest searchRequest = new MSNSearchServiceStub.SearchRequest();

            searchRequest.setAppID(appID);

            
            // to force Http1.0 used
            org.apache.axis2.client.Options option = searchService._getServiceClient().getOptions();
            option.setProperty(org.apache.axis2.transport.http.HTTPConstants.HTTP_PROTOCOL_VERSION, "HTTP/1.0");
            option.setProperty(org.apache.axis2.transport.http.HTTPConstants.CHUNKED,Boolean.FALSE);

            searchService._getServiceClient().setOptions(option);

            //検索する言語の指定
            switch (language)
            {
                case Japanese:
                    searchRequest.setCultureInfo("ja-JP");
                    break;
                case English:
                    searchRequest.setCultureInfo("en-US");
                    break;
            }

            searchRequest.setQuery(query);

            switch (this.safeSearchOptions)
            {
                case Moderate:
                    searchRequest.setSafeSearch(MSNSearchServiceStub.SafeSearchOptions.Moderate);
                    break;
                case Strict:
                	searchRequest.setSafeSearch(MSNSearchServiceStub.SafeSearchOptions.Strict);
                    break;
                case None:
                	searchRequest.setSafeSearch(MSNSearchServiceStub.SafeSearchOptions.Off);
                    break;
            }

            switch (this.searchFlags)
            {
                case DisableHostCollapsing:
                	searchRequest.setFlags(createSearchFlags(MSNSearchServiceStub.SearchFlags_type0.DisableHostCollapsing));                    		
                    break;
                case MarkQueryWords:
                	searchRequest.setFlags(createSearchFlags(MSNSearchServiceStub.SearchFlags_type0.MarkQueryWords));
                	break;
                case None:
                	searchRequest.setFlags(createSearchFlags(MSNSearchServiceStub.SearchFlags_type0.None));
                    break;
            }


            int arraySize = 1;
            MSNSearchServiceStub.SourceRequest[] sourceRequests = new MSNSearchServiceStub.SourceRequest[arraySize];

            sourceRequests[0] = new MSNSearchServiceStub.SourceRequest();
            sourceRequests[0].setCount(25);
            sourceRequests[0].setOffset(offset);
            sourceRequests[0].setResultFields(createMask(MSNSearchServiceStub.ResultFieldMask_type0.All));
            sourceRequests[0].setSource(MSNSearchServiceStub.SourceType.Web);
            if (!PortUtil.isNullOrEmpty(fileType))
            {
                sourceRequests[0].setFileType(fileType);
            }
            MSNSearchServiceStub.ArrayOfSourceRequestRequests array = new MSNSearchServiceStub.ArrayOfSourceRequestRequests();
            array.setSourceRequest(sourceRequests);
            searchRequest.setRequests(array);
            
            //結果の取得開始
            MSNSearchServiceStub.Search search = new MSNSearchServiceStub.Search();
            search.setRequest(searchRequest);
            MSNSearchServiceStub.SearchResponse0 searchResponse = searchService.Search(search);

            long totalNumber = 0;
            List<LiveWebElement> resultElements = new ArrayList<LiveWebElement>();

            MSNSearchServiceStub.ArrayOfSourceResponseResponses results = searchResponse.getResponse().getResponses();
            
            for (MSNSearchServiceStub.SourceResponse response: results.getSourceResponse())
            {

                    totalNumber = response.getTotal();

                    int count = 0;
                MSNSearchServiceStub.Result[] elements = response.getResults().getResult();
                for (MSNSearchServiceStub.Result re: elements)
                {
                    LiveWebElement element = new LiveWebElement(safeGet(re.getTitle()),
                    		safeGet(re.getDescription()), safeGet(re.getUrl()), 
                    		safeGet(re.getDisplayUrl()), safeGet(re.getCacheUrl()), safeGet(re.getSearchTags()));
                    resultElements.add(element);
                    count++;
                    if(count >= trueResultNum){
                        break;
                    }


                }

            }

            return new LiveWebSearchResult(query, totalNumber, resultElements.toArray(new LiveWebElement[0]));

        }

        private String safeGet(String str)
        {
            return PortUtil.isNullOrEmpty(str) ? "" : str;
        }

        private LiveWebSearchResult doSearchOver(String query, int resultNum) throws RemoteException
        {
            int loop = (resultNum - 1) / 25;
            long total = 0;
            List<LiveWebElement> resultElements = new ArrayList<LiveWebElement>();
            for (int i = 0; i < loop; i++)
            {
                resultElements.addAll(Arrays.asList(doSearchOriginal(query, 25, 25 * i,25).getResultElements()));
            }
            LiveWebSearchResult searchResult = doSearchOriginal(query, 25, loop * 25,resultNum - (loop * 25));
            total = searchResult.getTotalNumber();
            resultElements.addAll(Arrays.asList(searchResult.getResultElements()));

            return new LiveWebSearchResult(query, total, resultElements.toArray(new LiveWebElement[0]));

        }



        // プロパティ

        /**
         * 検索に関するフラグ
         */
        public SearchFlags getSearchFlag()
        {
        	return this.searchFlags;
        }
        public void setSerachFlag(SearchFlags value)
        {
                this.searchFlags = value;
        }

        /**
         * 有害サイトのフィルタリングオプション
         */
        public SafeSearchOptions getSafeSearchOption()
        {
        	return this.safeSearchOptions;
        }
        public void setSafeSearchOption(SafeSearchOptions value)
        {
        	this.safeSearchOptions = value;
        }

        /**
         * 検索するときの言語
         */
        public Language getSearchLanguage()
        {
        	return this.language;
        }
        public void setSearchLanguage(Language value)
        {
        	this.language = value;
        }

        /**
         * ファイルの種類
         */
        public String getFileType()
        {
        	return this.fileType;
        }
        
        public void setFileType(String value)
        {
                this.fileType = value;
        }



        // 列挙型

        /**
         * 検索に関するフラグ
         */
        public enum SearchFlags
        {
            /**
             * 検索結果がURLによって集約されることを無効にする
             */
            DisableHostCollapsing,

            /**
             * 検索結果中のクエリを強調する
             */
            MarkQueryWords,

            /**
             * 何も指定しない
             */
            None,
        }

        /**
         * 有害サイトのフィルタリングオプション
         */
        public enum SafeSearchOptions
        {
            /**
             * 有害な画像のみフィルタリングを行う
             */
            Moderate,
            /**
             * 有害な画像とテキストの両方のフィルタリングを行う
             */
            Strict,
            /**
             * 何もフィルタリングを行わない
             */
            None,
        }

        /**
         * 検索するときの言語
         */
        public enum Language
        {
            /**
             * 日本語
             */
            Japanese,
            /**
             * 英語
             */
            English,
        }


    }

