//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.Xml;
//using System.Web;
//using System.IO;
//using System.Net;
//using System.Net.Sockets;

package slothLib.web.search;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.Proxy;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import slothLib.portUtil.PortUtil;

    /**
     * YouTubeタグ検索を行うクラス
     */
    public class YouTubeSearch implements IVideoSearch
    {

        // privateフィールド

        /**
         * YouTube API の dev_key
         */
        private String dev_id;
        private Proxy proxy;



        // コンストラクタ

        /**
         * コンストラクタ　アプリケーションIDを指定する
         * @param dev_id プログラムで用いるアプリケーションID
         */

        public YouTubeSearch(String dev_id)
        {
            this.dev_id = dev_id;
        }




        // public関数

        /**
         * YouTube検索を行う
         * @param query 検索クエリ
         * @param resultNum 検索結果取得数
         * @return 検索結果
         */
        public YouTubeSearchResult doSearch(String query, int resultNum)
        throws WebSearchException
        {
            return this.doSearchOver(query, resultNum);
        }




        // private関数

        private YouTubeSearchResult doSearchOriginal(String query, int resultNum, int page)
        throws WebSearchException
        {
            List<YouTubeElement> result = new ArrayList<YouTubeElement>();

            // YouTube API REST リクエストURL
            String url = makeYouTubeQuery(query, page, resultNum);
            System.err.println(url);

            Document xmlDoc = PortUtil.getDocumentFromURL(url, this.proxy);
            // ルートの要素を取得
            Element rootElement = xmlDoc.getDocumentElement();

            long totalNumber = 0;

            try
            {
               totalNumber =  Long.parseLong(rootElement.getChildNodes().item(0).getChildNodes().item(0).getTextContent());
            }
            catch (NumberFormatException e)
            {
                totalNumber = 0;
            }


            // 成功したか否かのチェック
            {
                String status = rootElement.getAttribute("status");
                if (status.equals("fail"))
                {
                	String code = rootElement.getElementsByTagName("code").item(0).getTextContent();
                	//例外を発生させる
                	throw (new YouTubeException(code));
                }
            }

            

            NodeList videoList = rootElement.getElementsByTagName("video");

            for (int i = 0; i < videoList.getLength(); i++)
            {
            	Node video = videoList.item(i);
            	String author =          video.getChildNodes().item(0).getTextContent();
                String id =              video.getChildNodes().item(1).getTextContent();
                String title =           video.getChildNodes().item(2).getTextContent();
                String length_seconds = video.getChildNodes().item(3).getTextContent();
                String rating_avg =     video.getChildNodes().item(4).getTextContent();
                String rating_count =   video.getChildNodes().item(5).getTextContent();
                String description =    video.getChildNodes().item(6).getTextContent();
                String view_count =     video.getChildNodes().item(7).getTextContent();
                String date =           video.getChildNodes().item(8).getTextContent();
                String commnet_count = video.getChildNodes().item(9).getTextContent();

                String tags = video.getChildNodes().item(10).getTextContent();
                // タグを[]で囲う
                tags = "[" + tags + "]";
                tags = tags.replace(" ", "] [");
                // snippetにtagsとsnippetをつける


                String uri = video.getChildNodes().item(11).getTextContent(); // ページへのURL （flvへの直URLではない）

                String thumbnail_url = video.getChildNodes().item(12).getTextContent();
                
                // コメントを取得
                // snippet += GetComment(dev_id, id);

                //変数を元に結果クラスを作成し、リストに追加
                YouTubeElement element = new YouTubeElement(safeString(author), 
                    safeString(id), 
                    safeString(title),
                    strToInt(length_seconds), 
                    strToDouble(rating_avg),
                    strToInt(rating_count),
                    safeString(description),
                    strToInt(view_count),
                    UnixTimeStampToDateTime(date),
                    strToInt(commnet_count),
                    safeString(tags),
                    safeString(uri),
                    safeString(thumbnail_url));
                result.add(element);
                System.err.println(element.toString());
            }

            return new YouTubeSearchResult(query, totalNumber, result.toArray(new YouTubeElement[0]));

        }

        /**
         * YouTubeに投げるURLを出力する
         * @param tag the tag to search for
         * @param page the "page" of results you want to retrieve (e.g. 1, 2, 3)
         * @param per_page the number of results you want to retrieve per page (default 20, maximum 100) 
         * @return 
         */
        private String makeYouTubeQuery(String tag, int page, int per_page)
        {
           
            String requestURL = "http://www.youtube.com/api2_rest?method=youtube.videos.list_by_tag&"
            + "dev_id=" + dev_id + "&tag=" +tag
            + "&page=" + page + "&per_page=" + per_page;

            return PortUtil.absoluteUri(requestURL);
        }

        // GetComment
        /**
         * videos.get_details メソッドでコメントを取得する
         * @param dev_id アプリケーションID
         * @param video_id ビデオID
         * @return コメントをつなげたもの
         */
        public String getComment(String dev_id, String video_id)
        throws WebSearchException
        {
            String comment_str = "";

            // コメントを取得
            String detail_url = "http://www.youtube.com/api2_rest?method=youtube.videos.get_details&" + "dev_id=" + dev_id + "&video_id=" + video_id;
            Document detail_xmlDoc = PortUtil.getDocumentFromURL(detail_url, this.proxy);

            Element detail_rootElement = detail_xmlDoc.getDocumentElement();

            {
                String status = detail_rootElement.getAttribute("status");
                if (status.equals("ok"))
                { 
                	// do nothing
                } 
                else if (status.equals("fail"))
                {
                	NodeList nl = detail_rootElement.getElementsByTagName("code");
                	String code = nl.item(0).getTextContent();
                	//例外を発生させる
                	throw (new YouTubeException(code));
                }
            }

            NodeList commentList = detail_rootElement.getElementsByTagName("comment");

            if (commentList != null)
            {
                for (int index = 0; index < commentList.getLength(); index++)
                {
                	Node comment = commentList.item(index);
                	// author = comment.ChildNodes[3].InnerText;
                    comment_str += " " + comment.getChildNodes().item(1).getTextContent(); // comment
                    // time = comment.ChildNodes[3].InnerText;
                }
            }

            return comment_str;
        }


        // GetSwfUrl
        /**
         * FlashのプレイヤーURLを取得
         * @param videoID ビデオID
         * @return FlashのプレイヤーURL
         */
        public String getSwfUrl(String videoID)
        {
            // HTTP Request
            String request = "GET /watch_video?v=" + videoID + " HTTP/1.0\r\n";
            request += "Connection: Close\r\n";
            request += "\r\n";

            // tを取得するリクエストメッセージ
            String server = "www.youtube.com";
            String path = "/watch_video?v=" + videoID;
            //リクエストメッセージを作成する
            String reqMsg = "GET " + path + " HTTP/1.0\r\n" + "Connection: Close\r\n\r\n";

            // ソケット通信
            String resMsg = socketConnect(server, reqMsg);


            // tを取得
            int start = resMsg.indexOf("&t=") + 3;
            int end = resMsg.indexOf("\n", start);
            String t = resMsg.substring(start, end - start - 1);

            String swfURL = "http://www.youtube.com/player2.swf?video_id=" + videoID + "&t=" + t;
            return swfURL;

        }


        // GetVideoUrl
        /**
         * videoの直URLを取得
         * @param videoID videoID
         * @return 直URL
         */
        public String getVideoUrl(String videoID)
        {
            // HTTP Request
            String request = "GET /watch_video?v=" + videoID + " HTTP/1.0\r\n";
            request += "Connection: Close\r\n";
            request += "\r\n";

            // tを取得するリクエストメッセージ
            String server = "www.youtube.com";
            String path = "/watch_video?v=" + videoID;
            //リクエストメッセージを作成する
            String reqMsg = "GET " + path + " HTTP/1.0\r\n" + "Connection: Close\r\n\r\n";

            // ソケット通信
            String resMsg = socketConnect(server, reqMsg);


            // tを取得
            int start = resMsg.indexOf("&t=") + 3;
            int end = resMsg.indexOf("\n", start);
            String t = resMsg.substring(start, end - start - 1);


            // video URL を取得するリクエストメッセージ
            String path2 = "/get_video?video_id=" + videoID + "&t=" + t;
            //リクエストメッセージを作成する
            String reqMsg2 = "GET " + path2 + " HTTP/1.0\r\n" + "Connection: Keep-Alive\r\n\r\n";


            // ソケット通信
            String resMsg2 = socketConnect(server, reqMsg2);

            // video URL を取得
            int start2 = resMsg2.indexOf("Location:") + 10;
            int end2 = resMsg2.indexOf("\r\n", start2);
            String videoURL = resMsg2.substring(start2, end2 - start2);

            return videoURL;

        }


        // socketConnect
        /**
         * ソケット通信を行なう
         * @param server サーバ
         * @param reqMsg リクエストメッセージ
         * @return レスポンスメッセージ
         */
        private String socketConnect(String server, String reqMsg)
        {
//            //文字列をbyte配列に変換
//            Encoding enc = Encoding.ASCII;
//            byte[] reqBytes = enc.GetBytes(reqMsg);
//
//            //ホスト名からIPアドレスを取得
//            IPAddress hostadd = Dns.GetHostEntry(server).AddressList[0];
//
//            //IPEndPointを取得
//            IPEndPoint ephost = new IPEndPoint(hostadd, 80);

            //Socketの作成,	接続
        	try {
        		Socket sock = new Socket(server, 80);
        		PrintWriter pw = new PrintWriter(new OutputStreamWriter(sock.getOutputStream()));
        		pw.write(reqMsg);
        		LineNumberReader lnr = new LineNumberReader(new InputStreamReader(sock.getInputStream()));
        		StringBuilder resMsg = new StringBuilder(); 
        		String str = null;
        		while ((str = lnr.readLine()) != null){
        			resMsg.append(str);
        			resMsg.append("\n");
        		}
        		sock.close();
        		return resMsg.toString();
        	} catch (IOException e){
        		return "Error";
        	}
            

        }


        private Date UnixTimeStampToDateTime(String date)
        {

            //UNIXタイムスタンプをDateTime型に変換
            try
            {
                double tmp = Double.parseDouble(date);
                return new Date((long)(tmp * 1000));
            }
            catch (NumberFormatException e)
            {
                return new Date();
            }

        }

        private int strToInt(String str)
        {
            try
            {
                return Integer.parseInt(str);
            }
            catch (NumberFormatException e)
            {
                return 0;
            }
        }

        private double strToDouble(String str)
        {
            try
            {
                return Double.parseDouble(str);
            }
            catch (NumberFormatException e)
            {
                return 0.0;
            }
        }

        private String safeString(String str)
        {
            if (str == null)
            {
                return "";
            }
            else
            {
                return str;
            }
        }


        private YouTubeSearchResult doSearchOver(String query, int resultNum)
        throws WebSearchException
        {
            int loop = (resultNum - 1) / 100;
            List<YouTubeElement> resultElements = new ArrayList<YouTubeElement>();
            for (int i = 0; i < loop; i++)
            {
                resultElements.addAll(Arrays.asList(doSearchOriginal(query, 100, i + 1).getResultElements()));
            }
            YouTubeSearchResult searchResult = doSearchOriginal(query, resultNum - (loop * 100), loop + 1);
            resultElements.addAll(Arrays.asList(searchResult.getResultElements()));

            return new YouTubeSearchResult(query, searchResult.getTotalNumber(), resultElements.toArray(new YouTubeElement[0]));
        }



        // プロパティ

        /**
         * プロクシを取得・設定する。
         */
    	public void setProxy(String value)
    	{
    		if ( PortUtil.isNullOrEmpty(value))
    		{
    			this.proxy = null;
    		}
    		else
    		{
    			this.proxy = PortUtil.createProxy(value);
    		}
    	}
    	public String getProxy()
    	{
    		return PortUtil.getProxyURL(this.proxy);
    	}




//        //#region IVideoSearch メンバ
//
//        IVideoSearchResult IVideoSearch.DoSearch(String query, int resultNum)
//        {
//            return this.DoSearch(query, resultNum);
//        }
//
//        //#endregion
//
//        //#region ISearch メンバ
//
//        ISearchResult ISearch.DoSearch(String query, int resultNum)
//        {
//            return this.DoSearch(query, resultNum);
//        }
//
//        //#endregion
    }

