//using System;
//using System.Collections.Generic;
//using System.Text;
//using System.Threading;
//using System.IO;
//using System.Net;

package slothLib.web.webLoader;


import slothLib.portUtil.PortUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*; 

    /**
     * マルチスレッドを用いて複数URLの内容を取得するクラス
     */
    public class MultiWebGet
    {
        // privateフィールド

        private int threadCount;
        private String proxy;
        private String userAgent;
        private int count;


        // コンストラクタ

        /**
         * コンストラクタ
         */
        public MultiWebGet()
        {
            this.proxy = null;
            this.userAgent = null;
            this.threadCount = DEFAULT_THREAD_COUNT;
            this.count = 0;
           
        }



        // DoFetch

        /**
         * マルチスレッドを利用して複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @return MultiWebGetResults型の結果
         */
        public MultiWebGetResults doFetch(String[] urlList)
        {
            return this.doFetch(urlList, null, null);
        }

        /**
         * マルチスレッドを利用して複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @param outputFileList 取得するURLのリストに対応した出力ファイル名のリスト
         * @return MultiWebGetResults型の結果
         */
        public MultiWebGetResults doFetch(String[] urlList, String[] outputFileList)
        {
            return this.doFetch(urlList, outputFileList, null, null);
        }



        /**
         * マルチスレッドを利用して複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @param allowContentTypes 取得するURLのCotentTypeのリスト
         * @param excludeContentTypes 取得しないURLのCotentTypeのリスト
         * @return MultiWebGetResults型の結果
         */
        public MultiWebGetResults doFetch(String[] urlList, String[] allowContentTypes, String[] excludeContentTypes)
        {
            String[] tmpFileNameList = new String[urlList.length];

            for (int i = 0; i < urlList.length; i++)
            {
                tmpFileNameList[i] = getTempFileName(urlList[i]);
            }

            return this.doFetch(urlList, tmpFileNameList, allowContentTypes, excludeContentTypes);
            
        }

        /**
         * マルチスレッドを利用して複数URLのデータを取得する
         * @param urlList データを取得するURLのリスト
         * @param outputFileList 取得するURLのリストに対応した出力ファイル名のリスト
         * @param allowContentTypes 取得するURLのCotentTypeのリスト
         * @param excludeContentTypes 取得しないURLのCotentTypeのリスト
         * @return 
         */
        public MultiWebGetResults doFetch(String[] urlList, String[] outputFileList, String[] allowContentTypes, String[] excludeContentTypes) 
        {
            if (urlList.length > outputFileList.length)
            {
                System.err.println("MultiWebGet:取得するURLリストと対応する出力ファイル名リストの要素数が対応していません");
                return null;
            }

            List< Future <WebGetResult>> futures = new ArrayList<Future <WebGetResult>>();
                                   
            WebGetResult [] results = new WebGetResult [urlList.length];  
            
            for (int i = 0; i < urlList.length; i++){
            	WebGetManager wgm = new WebGetManager(urlList[i], outputFileList[i], allowContentTypes, excludeContentTypes, this.proxy,this.userAgent);	
            	futures.add(threadPool.submit(wgm));
            }
            for (int i = 0; i < urlList.length; i++)
            {
            	try {
            		results[i] = futures.get(i).get();
            	} catch (InterruptedException e) {
            		e.printStackTrace();
            	} catch (ExecutionException e) {
            		e.printStackTrace();
            	}
            }

            return new MultiWebGetResults(results);
        }




        /**
         * URLから一意のファイル名を得る
         * @param url 
         * @return 
         */
        private String getHashName(String url)
        {
            return url.hashCode() + "." +System.currentTimeMillis() + ".tmp";
        }

        /**
         * URLから一時ファイル名を得る
         * @param url 
         * @return 
         */
        private String getTempFileName(String url)
        {
            return System.getProperty("java.io.tmpdir") + "/" + getHashName(url);
        }

        /**
         * HTTP通信時に用いるproxyサーバ
         */
        public void setProxy(String value)
        {
        	this.proxy = value;
        }
        public String getProxy()
        {
        	return proxy;
        }

        /**
         * HTTP通信時に用いるUserAgent
         */
        public void setUserAgent(String value)
        {
        	this.userAgent = value;
        }
        public String getUserAgent()
        {
        	return this.userAgent;
        }

        /**
         * Webの取得に利用するスレッドの数
         */
        public int getThreadCount()
        {
        	return this.threadCount;
        }

        public void setThreadCount(int value)
        {
        	if (value > MAX_THREAD_COUNT)
        	{
        		this.threadCount = MAX_THREAD_COUNT;
        	}
        	else if (value < 0)
        	{
        		this.threadCount = 1;
        	}
        	else
        	{
        		this.threadCount = value;
        	}
    		threadPool.setMaximumPoolSize(this.threadCount);
        }



        // 定数

        private static final int MAX_THREAD_COUNT = 20;
        private static final int DEFAULT_THREAD_COUNT = 10;

        private static ThreadPoolExecutor threadPool =
        	new ThreadPoolExecutor(2, DEFAULT_THREAD_COUNT,  10, TimeUnit.MINUTES, new LinkedBlockingDeque<Runnable>()); 


    }

