﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;
using System.IO;
using System.Text;

namespace Somali.Base
{
    class XIO
    {
        private XElement _XRoot = null;
        /// <summary>
        /// ルート要素
        /// </summary>
        public XElement RootElement
        {
            set { this._XRoot = value; }
            get { return this._XRoot; }
        }

        #region コンストラクタ

        public XIO()
        {
        }

        public XIO( XElement root )
        {
            if ( root != null )
            {
                this.RootElement = root;
            }
            else
            {
                throw new ArgumentNullException( "root" );
            }
        }

        #endregion

        /// <summary>
        /// ルート要素を持っているかどうかを示す値を取得します。
        /// </summary>
        /// <returns>True:存在する<br/>False:存在しない</returns>
        public bool HasRoot()
        {
            return (RootElement != null);
        }

        /// <summary>
        /// XMLファイルを読み込みます。
        /// </summary>
        /// <param name="filePath">読み込み元XMLファイルパス</param>
        public void LoadXML( string filePath )
        {
            if ( string.IsNullOrEmpty( filePath ) == false )
            {
                if ( File.Exists( filePath ) )
                {
                    RootElement = XElement.Load( filePath );
                }
                else
                {
                    RootElement = null;
                }
            }
            else
            {
                RootElement = null;
            }
        }

        /// <summary>
        /// XMLファイルを書き出します。
        /// </summary>
        /// <param name="filePath">保存先XMLファイルパス</param>
        public void SaveXML( string filePath )
        {
            if ( RootElement != null )
            {
                RootElement.Save( filePath );
            }
        }

        /// <summary>
        /// ルート要素(XIO.RootElement)の持つ1つ以上の要素を取得します。
        /// </summary>
        /// <returns>ルート要素のXMLツリー下の要素を返却します。<br/>
        /// 要素を持たない場合はnullを返却します。</returns>
        public IEnumerable<XElement> GetAllElements()
        {
            return GetAllElements( RootElement );
        }

        /// <summary>
        /// 指定されたXElementの持つ1つ以上の要素を取得します。
        /// </summary>
        /// <param name="element">XElement</param>
        /// <returns>指定されたXElementのXMLツリー下の要素を返却します。<br/>
        /// 要素を持たない場合はnullを返却します。</returns>
        public static IEnumerable<XElement> GetAllElements( XElement element )
        {
            if ( element.HasElements == true )
            {
                return element.Elements();
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// ルート要素(XIO.RootElement)から指定されたタグの一番最初の要素を取得します。
        /// </summary>
        /// <param name="tag"></param>
        /// <returns></returns>
        public XElement GetElementByTag( string tag )
        {
            return GetElementByTag( RootElement, tag );
        }

        /// <summary>
        /// 指定されたXElementから指定されたタグの一番最初の要素を取得します。
        /// </summary>
        /// <param name="element"></param>
        /// <param name="tag"></param>
        /// <returns></returns>
        public static XElement GetElementByTag( XElement element, string tag )
        {
            if ( element.HasElements == true )
            {
                return element.Descendants( tag ).First();
            }
            else
            {
                return null;
            }
        }

        public IEnumerable<XElement> GetElementsListByTag( string tag )
        {
            IEnumerable<XElement> query = RootElement.Descendants( tag );

            if ( query.ToList().Count == 1 )
            {
                return query;
            }
            else
            {
                return null;
            }
        }

        public static string GetValueByTag( XElement element, string tag )
        {
            if ( element.HasElements == true )
            {
                return element.Descendants( tag ).First().Value;
            }
            else
            {
                return null;
            }
        }

        #region Replace

        public void ReplaceElementByTag( XElement element, string tag )
        {

        }

        /// <summary>
        /// ルート要素(XIO.RootElement)の要素の値を変更します。
        /// </summary>
        /// <param name="tag"></param>
        /// <param name="value"></param>
        /// <remarks>ルート要素(XIO.RootElement)から指定されたタグで取得できる一番最初の要素の値を指定された値で変更します。</remarks>
        public void ReplaceValueByTag( string tag, object value )
        {
            ReplaceValueByTag( RootElement, tag, value );
        }

        /// <summary>
        /// 指定されたXElementの要素の値を変更します。
        /// </summary>
        /// <param name="element"></param>
        /// <param name="tag"></param>
        /// <param name="value"></param>
        /// <remarks>指定されたXElementから指定されたタグで取得できる一番最初の要素の値を指定された値で変更します。</remarks>
        public static void ReplaceValueByTag( XElement element, string tag, object value )
        {
            XElement target = GetElementByTag( element, tag );
            if ( target == null || target.IsEmpty == true )
            {
                throw new Exception( "element is null or empty." );
            }
            else
            {
                target.SetValue( value );
            }
        }
        #endregion

    }
}
