package my.logic.sokoban;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;


public class Sokoban {
    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Sokoban <input file> [output file]");
            return;
        }
        
        Sokoban sokoban = new Sokoban();
        
        try {
            sokoban.process(args[0], args.length > 1 ? args[1] : null);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    int size = 0;
    int move = 0;
    
    List<Point[]> map = new ArrayList<Point[]>();
    
    Point start;
    Point goal;
    
    class Point {
        int row;
        int col;
        char ch;
        int steps;
        
        Point(char ch, int row, int col) {
            this.ch = ch;
            this.row = row;
            this.col = col;
            this.steps = -1;
        }
        
        public String toString() {
            return ch + "(" + row + "," + col + "," + steps + ")";
        }
    }
    
    private void constructMap(String input) throws Exception {
        BufferedReader br = new BufferedReader(new FileReader(input));

        String line;
        int row = 0;
        while ((line = br.readLine()) != null) {
            char[] ca = line.toCharArray();
            Point[] pa = new Point[ca.length];
            for (int col = 0; col < line.length(); col++) {
                Point p = new Point(line.charAt(col), row, col);
                pa[col] = p;
                if (p.ch == 'S') {
                    start = p;
                }
                else if (p.ch == 'G') {
                    goal = p;
                }
            }
            map.add(pa);
            row++;
            size += ca.length;
        }
    }
    
    private void move(Point p, int r, int c) {
        Point p2 = getPoint(p.row + r, p.col + c);
        if (p2 != null && p2.ch != '*' && (p2.steps == -1 || p.steps + 1 < p2.steps)) {
            p2.steps = p.steps + 1;
            move(p2);
        }
    }

    private void move(Point p) {
        move++;
        move(p, -1, 0); //north
        move(p, 0, 1); //east
        move(p, 1, 0); //south
        move(p, 0, -1); //west
    }
    
    private Point getPoint(int row, int col) {
        if (row < 0 || row >= map.size()) {
            return null;
        }
        
        Point[] pa = map.get(row);
        if (col < 0 || col >= pa.length) {
            return null;
        }
        
        return pa[col];
    }
    
    private Point small(Point p1, Point p2) {
        if (p1 == null) {
            return p2;
        }
        if (p2 == null) {
            return p1;
        }
        if (p1.steps >= 0 && (p2.steps < 0 || p1.steps < p2.steps)) {
            return p1;
        }
        return p2;
    }
    
    private void back(Point p) throws Exception {
//        System.out.println(p);
//        printMap(null);
        
        Point p2 = null;
        
        Point pn = getPoint(p.row - 1, p.col);
        p2 = small(pn, p2);
        
        Point pe = getPoint(p.row, p.col + 1);
        p2 = small(pe, p2);
        
        Point ps = getPoint(p.row + 1, p.col);
        p2 = small(ps, p2);
        
        Point pw = getPoint(p.row, p.col - 1);
        p2 = small(pw, p2);
        
        if (p2 != null && p2.steps > 0) {
            p2.ch = '$';
            back(p2);
        }
    }
    
    private void printMap(String output) throws Exception {
        PrintStream ps = System.out;
        if (output != null) {
            ps = new PrintStream(output);
        }

        for (Point[] pa : map) {
            for (Point p : pa) {
                ps.print(p.ch);
            }
            ps.println();
        }
        if (output != null) {
            ps.close();
        }
    }
    
    public void process(String input, String output) throws Exception {
        constructMap(input);

        long st = System.currentTimeMillis();

        start.steps = 0;
        move(start);
        
        if (goal.steps > 0) {
            back(goal);
        }
        
        long et = System.currentTimeMillis();

        System.out.println("Sokoban <" + input
            + "> - steps: " + goal.steps
            + ", size: " + size
            + ", move: " + move
            + ", time: " + (et - st));
        
        printMap(output);
    }
}

