package my.logic.sokoban;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import nuts.core.util.TreeNode;


public class Sokoban1 {

	public static void main(String[] args) {
		if (args.length < 1) {
			System.out.println("Sokoban <input file> [output file]");
			return;
		}
		try {
			Sokoban1 sokoban = new Sokoban1();
			sokoban.process(args[0], args.length > 1 ? args[1] : null);
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	long trys = 0;
	List<char[]> map;
	TreeNode<Point> root;
	
	private int getArea() {
		int area = 0;
		for (char[] s : map) {
			area += s.length;
		}
		return area;
	}
	
	public void process(String in, String out) throws Exception {
		BufferedReader br = new BufferedReader(new FileReader(in));
		
		map = new ArrayList<char[]>();
		while (true) {
			String line = br.readLine();
			if (line == null) {
				break;
			}
			map.add(line.toCharArray());
		}
		br.close();

		int area = getArea();

		long start = System.currentTimeMillis();
		int steps = solve();
		long end = System.currentTimeMillis();
		
		System.out.println("Sokoban <" + in 
			+ "> - steps: " + steps 
			+ ", area: " + area
			+ ", trys: " + trys
			+ ", times: " + (end - start));
		if (steps < 1) {
			System.out.println("The goal is not reachable!");
		}
		
		PrintStream ps;
		if (out != null) {
			ps = new PrintStream(out);
		}
		else {
			ps = System.out;
		}
		for (char[] line : map) {
			ps.println(line);
		}
		if (out != null) {
			ps.close();
		}
	}
	
	class Point {
		char c;
		int x;
		int y;
		
		public Point(char c, int x, int y) {
			this.c = c;
			this.x = x;
			this.y = y;
		}
		
		public String toString() {
			return c + "(" + x + "," + y + ")";
		}
		
		public boolean equals(Object o) {
			if (o instanceof Point) {
				Point p = (Point)o;
				return this.x == p.x && this.y == p.y;
			}
			return false;
		}

		private Point move(int x, int y) {
			char c = getChar(x, y);
			if (c == '*') {
				return null;
			}
			return new Point(c, x, y);
		}
		
		public Point left() {
			if (x > 0) {
				return move(x - 1, y);
			}
			return null;
		}
		
		public Point right() {
			char[] s = map.get(y);
			if (x < s.length - 1) {
				return move(x + 1, y);
			}
			return null;
		}
		
		public Point up() {
			if (y > 0) {
				return move(x, y - 1);
			}
			return null;
		}
		
		public Point down() {
			if (y < map.size() - 1) {
				return move(x, y + 1);
			}
			return null;
		}
		
	}
	
	private Point find(char c) throws Exception {
		for (int y = 0; y < map.size(); y++) {
			char[] s = map.get(y);
			for (int x = 0; x < s.length; x++) {
				if (s[x] == c) {
					return new Point(c, x, y);
				}
			}
		}
		return null;
	}
	
	private char getChar(int x, int y) {
		char[] s = map.get(y);
		return s[x];
	}
	
	private void setChar(Point p, char c) {
		char[] s = map.get(p.y);
		s[p.x] = c;
	}
	
	private int solve() throws Exception {
		Point start = find('S');
		Point goal = find('G');
		
		root = new TreeNode<Point>(start);

		move(root);

		TreeNode<Point> gn = root.findNode(goal);
		
		if (gn != null && !gn.isRoot()) {
			TreeNode<Point> n = gn.getParent();
			while (!n.isRoot()) {
				setChar(n.getValue(), '$');
				n = n.getParent();
			}
			return gn.getLevel();
		}
		
		return 0;
	}
	
	private void addToTree(TreeNode<Point> n, Point p) {
		TreeNode<Point> a = root.findNode(p);
		if (a != null) {
			if (n.getLevel() + 1 < a.getLevel()) {
				a.remove();
				n.createChild(p);
			}
		}
		else {
			n.createChild(p);
		}
	}
	
	private void move(TreeNode<Point> node) {
		trys++;
		
		Point start = node.getValue();
		
		Point up = start.up();
		if (up != null) {
			addToTree(node, up);
		}
		
		Point right = start.right();
		if (right != null) {
			addToTree(node, right);
		}
		
		Point down = start.down();
		if (down != null) {
			addToTree(node, down);
		}

		Point left = start.left();
		if (left != null) {
			addToTree(node, left);
		}

		if (node.hasChild()) {
			for (TreeNode<Point> c : node.getChildren()) {
				move(c);
			}
		}
	}
}
