package my.logic.sokoban;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;

import nuts.core.util.TreeNode;


public class Sokoban11 {

	public static void main(String[] args) {
		if (args.length < 1) {
			System.out.println("Sokoban <input file> [output file]");
			return;
		}
		try {
			Sokoban11 sokoban = new Sokoban11();
			sokoban.process(args[0], args.length > 1 ? args[1] : null);
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}

	long trys = 0;
	List<char[]> charMap;
	List<TreeNode<Point>[]> nodeMap;
	TreeNode<Point> root;
	
	private int getArea() {
		int area = 0;
		for (char[] s : charMap) {
			area += s.length;
		}
		return area;
	}
	
	public void process(String in, String out) throws Exception {
		BufferedReader br = new BufferedReader(new FileReader(in));
		
		nodeMap = new ArrayList<TreeNode<Point>[]>();
		charMap = new ArrayList<char[]>();
		while (true) {
			String line = br.readLine();
			if (line == null) {
				break;
			}
			charMap.add(line.toCharArray());
			nodeMap.add(new TreeNode[line.length()]);
		}
		br.close();

		int area = getArea();

		long start = System.currentTimeMillis();
		int steps = solve();
		long end = System.currentTimeMillis();
		
		System.out.println("Sokoban <" + in 
			+ "> - steps: " + steps 
			+ ", area: " + area
			+ ", trys: " + trys
			+ ", times: " + (end - start));
		if (steps < 1) {
			System.out.println("The goal is not reachable!");
		}
		
		PrintStream ps;
		if (out != null) {
			ps = new PrintStream(out);
		}
		else {
			ps = System.out;
		}
		printMap(ps);
		if (out != null) {
			ps.close();
		}
	}
	
	private void printMap(PrintStream ps) {
		for (char[] line : charMap) {
			ps.println(line);
		}
	}
	
	class Point {
		char c;
		int x;
		int y;
		TreeNode<Point> node;
		
		public Point(char c, int x, int y) {
			this.c = c;
			this.x = x;
			this.y = y;
		}
		
		public String toString() {
			return c + "(" + x + "," + y + ")";
		}
		
		public boolean equals(Object o) {
			if (o instanceof Point) {
				Point p = (Point)o;
				return this.x == p.x && this.y == p.y;
			}
			return false;
		}

		private Point move(int x, int y) {
			char c = getChar(x, y);
			if (c == '*') {
				return null;
			}
			return new Point(c, x, y);
		}
		
		public Point left() {
			if (x > 0) {
				return move(x - 1, y);
			}
			return null;
		}
		
		public Point right() {
			char[] s = charMap.get(y);
			if (x < s.length - 1) {
				return move(x + 1, y);
			}
			return null;
		}
		
		public Point up() {
			if (y > 0) {
				return move(x, y - 1);
			}
			return null;
		}
		
		public Point down() {
			if (y < charMap.size() - 1) {
				return move(x, y + 1);
			}
			return null;
		}
		
	}
	
	private Point find(char c) throws Exception {
		for (int y = 0; y < charMap.size(); y++) {
			char[] s = charMap.get(y);
			for (int x = 0; x < s.length; x++) {
				if (s[x] == c) {
					return new Point(c, x, y);
				}
			}
		}
		return null;
	}
	
	private char getChar(int x, int y) {
		char[] s = charMap.get(y);
		return s[x];
	}
	
	private void setChar(Point p, char c) {
		char[] s = charMap.get(p.y);
		s[p.x] = c;
	}
	
	private TreeNode<Point> getNode(int x, int y) {
		TreeNode<Point>[] l = nodeMap.get(y);
		return l[x];
	}
	
	private void setNode(int x, int y, TreeNode<Point> n) {
		TreeNode<Point>[] l = nodeMap.get(y);
		l[x] = n;
	}

	private int solve() throws Exception {
		Point start = find('S');
		Point goal = find('G');
		
		root = new TreeNode<Point>(start);
		setNode(start.x, start.y, root);
		
		move(root);

		TreeNode<Point> gn = getNode(goal.x, goal.y);
		
		if (gn != null && !gn.isRoot()) {
			TreeNode<Point> n = gn.getParent();
			while (!n.isRoot()) {
				setChar(n.getValue(), '$');
				n = n.getParent();
			}
			return gn.getLevel();
		}
		
		return 0;
	}
	
	private void addToTree(TreeNode<Point> n, Point p) {
		TreeNode<Point> a = getNode(p.x, p.y);
		if (a != null) {
			if (n.getLevel() + 1 < a.getLevel()) {
				a.remove();
				TreeNode<Point> c = n.createChild(p);
				setNode(p.x, p.y, c);
			}
		}
		else {
			TreeNode<Point> c = n.createChild(p);
			setNode(p.x, p.y, c);
		}
	}
	
	private void move(TreeNode<Point> node) throws Exception {
		trys++;
		
//		if (trys == 184) {
//			System.out.println();
//		}
//		
		Point start = node.getValue();
//		setChar(start, '@');
//		printMap(System.out);
//		System.out.println();
//		Thread.sleep(500);
		
//		System.out.println(trys + " " + start);

		Point up = start.up();
		if (up != null) {
			addToTree(node, up);
		}

		Point right = start.right();
		if (right != null) {
			addToTree(node, right);
		}
		
		Point down = start.down();
		if (down != null) {
			addToTree(node, down);
		}

		Point left = start.left();
		if (left != null) {
			addToTree(node, left);
		}
		
		if (node.hasChild()) {
			for (TreeNode<Point> c : node.getChildren()) {
				move(c);
			}
		}

//		setChar(start, start.c);
//		printMap(System.out);
//		System.out.println();
//		Thread.sleep(500);
	}
}
