package my.tools;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import nuts.core.tool.CommandTool;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.io.IOUtils;


/**
 * Unzip
 */
public class Unzip {
    /**
     * Main class for Unzip
     */
    public static class Main extends CommandTool {
        /**
         * @param args arguments
         */
        public static void main(String[] args) {
            Main cgm = new Main();
            
            Object cg = new Unzip();

            cgm.execute(cg, args);
        }

        @SuppressWarnings("static-access")
        protected void addCommandLineOptions(Options options) throws Exception {
            super.addCommandLineOptions(options);
            
            options.addOption(OptionBuilder.withArgName("zipfile")
                .hasArg()
                .withDescription("zipfile to unzip")
                .create("i"));

            options.addOption(OptionBuilder.withArgName("output")
                .hasArg()
                .withDescription("output directory")
                .create("o"));
        }

        protected void getCommandLineOptions(CommandLine cl, Options options) throws Exception {
            super.getCommandLineOptions(cl, options);
            
            if (cl.hasOption("i")) {
                setParameter("src", new File(cl.getOptionValue("i")));
            }
            else {
                errorRequired(options, "zipfile");
            }

            if (cl.hasOption("o")) {
                setParameter("dir", new File(cl.getOptionValue("o")));
            }
        }

    }

    //---------------------------------------------------------------------------------------
    // properties
    //---------------------------------------------------------------------------------------
    private File src;
    private File dir = new File(".");
    private ZipFile zipfile;
    
    /**
     * @param src the src to set
     */
    public void setSrc(File src) {
        this.src = src;
    }

    /**
     * @param dir the dir to set
     */
    public void setDir(File dir) {
        this.dir = dir;
    }

    protected void checkParameters() throws Exception {
    	CommandTool.checkRequired(src, "src");
        if (!src.exists()) {
            throw new IllegalArgumentException("zipfile [" + src.getPath() + "] does not exists.");
        }
        zipfile = new ZipFile(src);
        
    	CommandTool.checkRequired(dir, "dir");
    }

    /**
     * execute
     * @throws Exception if an error occurs
     */
    public void execute() throws Exception {
        checkParameters();
        
        System.out.println("Extracting zipfile: " + src.getPath());
        
        int successed = 0;
        int failed = 0;
        for (Enumeration<? extends ZipEntry> e = zipfile.entries(); e.hasMoreElements(); ) {
            ZipEntry ze = e.nextElement();
            if (ze.isDirectory()) {
                continue;
            }
    
            System.out.print(ze.getName());
            System.out.print(" ... ");
            
            InputStream zis = zipfile.getInputStream(ze);
            if (zis == null) {
                System.out.println("FAILED");
                failed++;
            }
            else {
            	OutputStream os;
            	
            	try {
	                File f = new File(dir, ze.getName());
	                f.getParentFile().mkdirs();
	                os = new FileOutputStream(f);
            	}
            	catch (FileNotFoundException ex) {
                    String name = ze.getName().replaceAll("[^\\w\\-\\.\\\\\\/]", "");
	                File f = new File(dir, name);
	                f.getParentFile().mkdirs();
	                os = new FileOutputStream(f);
            	}
                IOUtils.copy(zis, os);
                
                zis.close();
                os.close();
    
                System.out.println("OK");
                successed++;
            }
        }
        
        System.out.print(successed + " files extract successfully");
        if (failed > 0) {
            System.out.println(", " + failed + " files extract failed!");
        }
        else {
            System.out.println(".");
        }
    }
    
}

