/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.affiliate.action.admin;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import nuts.core.collections.CollectionUtils;
import nuts.core.io.IOUtils;
import nuts.core.lang.StringUtils;
import nuts.exts.fileupload.UploadManager;

import org.apache.commons.vfs2.FileObject;

import squirrels.affiliate.action.admin.category.CategoryAction;
import squirrels.affiliate.bean.ImageInfo;
import squirrels.affiliate.constant.AC;
import squirrels.affiliate.constant.RC;
import squirrels.affiliate.model.bean.Category;
import squirrels.affiliate.model.dao.CategoryDAO;
import squirrels.affiliate.model.example.CategoryExample;


/**
 * CategoryExAction
 */
@SuppressWarnings("serial")
public class CategoryExAction extends CategoryAction {
	/**
	 * Constructor
	 */
	public CategoryExAction() {
		super();
	}

	/**
	 * queryListByExample
	 * @param example example
	 */
	@Override
	protected void queryListByExample(CategoryExample example) throws Exception {
		example.content().exclude();
		super.queryListByExample(example);
	}

	/**
	 * @return image list
	 */
	public List<ImageInfo> getImageList() {
		try {
			if (images == null && getData() != null && getData().getId() != null) {
				images = new ArrayList<ImageInfo>();

				FileObject dir = utils().getCategoryImagesPath(getData());
				if (dir.exists()) {
					for (FileObject f : dir.getChildren()) {
						ImageInfo ii = new ImageInfo();
						ii.setName(f.getName().getBaseName());
						ii.setTime(f.getContent().getLastModifiedTime());
						ii.setDisplay(ii.getName().equals(getData().getIcon()));
						images.add(ii);
					}
					Collections.sort(images);
				}
			}
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
		return images;
	}
	
	private void processIcon(Category data) throws Exception {
		if (CollectionUtils.isEmpty(images)) {
			data.setIcon(null);
		}
		else {
			data.setIcon(null);
			for (ImageInfo ii : images) {
				if (ii.isDisplay()) {
					if (StringUtils.isNotEmpty(ii.getName())
							&& !ImageInfo.DELETE.equalsIgnoreCase(ii.getAction())) {
						data.setIcon(ii.getName());
						break;
					}
					else if (StringUtils.isNotEmpty(ii.getTemp())
							&& ImageInfo.INSERT.equalsIgnoreCase(ii.getAction())) {
						data.setIcon(ii.getTemp());
						break;
					}
				}
			}
		}
	}

	private void processImages(Category data) throws Exception {
		if (CollectionUtils.isNotEmpty(images)) {
			FileObject ci = utils().getCategoryImagesPath(data);
			if (!ci.exists()) {
				ci.createFolder();
			}

			for (ImageInfo ii : images) {
				if (ImageInfo.INSERT.equalsIgnoreCase(ii.getAction())
						&& StringUtils.isNotEmpty(ii.getTemp())) {
					FileObject ft = UploadManager.resolveUploadFile(ii.getTemp());
					FileObject fi = ci.resolveFile(ft.getName().getBaseName());

					ft.moveTo(fi);
				}
				else if (ImageInfo.DELETE.equalsIgnoreCase(ii.getAction())
						&& StringUtils.isNotEmpty(ii.getName())) {
					FileObject fi = ci.resolveFile(ii.getName());
					if (fi.exists()) {
						fi.delete();
					}
				}
			}
			images = null;
		}
	}
	

	/**
	 * checkOnInsert
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	@Override
	protected boolean checkOnInsert(Category data) throws Exception {
		processIcon(data);
		return super.checkOnInsert(data);
	}

	/**
	 * commitInsert
	 * @param data data
	 * @throws Exception  if an error occurs
	 */
	@Override
	protected void commitInsert(Category data) throws Exception {
		super.commitInsert(data);
		
		addReloadMessage();

		processImages(data);
	}

	/**
	 * checkOnUpdate
	 * @param data data
	 * @param srcData srcData
	 * @throws Exception if an error occurs
	 */
	@Override
	protected boolean checkOnUpdate(Category data, Category srcData) throws Exception {
		processIcon(data);
		return super.checkOnUpdate(data, srcData);
	}

	/**
	 * commitUpdate
	 * @param data data
	 * @param srcData source data
	 * @throws Exception  if an error occurs
	 */
	@Override
	protected void commitUpdate(Category data, Category srcData) throws Exception {
		super.commitUpdate(data, srcData);
		
		addReloadMessage();

		processImages(data);
	}

	/**
	 * checkOnDelete
	 * @param data data
	 * @param srcData source data
	 * @return true - continue, false - skip
	 */
	@Override
	protected boolean checkOnDelete(Category data, Category srcData) throws Exception {
		if (!super.checkOnDelete(data, srcData)) {
			return false;
		}
		
		CategoryDAO dao = new CategoryDAO(getDataAccessSession());
		CategoryExample example = dao.createExample();
		example.parentId().equalTo(data.getId());
		if (dao.countByExample(example) > 0) {
			addActionError(getText("error-delete-with-child"));
			return false;
		}

		return true;
	}
	
	/**
	 * commitDelete
	 * @param data data
	 */
	@Override
	protected void commitDelete(Category data) throws Exception {
		super.commitDelete(data);

		IOUtils.deltree(utils().getCategoryImagesPath(data));
	}

	/**
	 * checkOnBulkDelete
	 * @param data data
	 * @return true - continue, false - skip
	 */
	@Override
	protected boolean checkOnBulkDelete(List<Category> dataList) throws Exception {
		if (!super.checkOnBulkDelete(dataList)) {
			return false;
		}

		CategoryDAO cDao = new CategoryDAO(getDataAccessSession());

		for (Category c : dataList) {
			CategoryExample cExp = cDao.createExample();
			cExp.parentId().equalTo(c.getId()).setLimit(1);
			if (cDao.countByExample(cExp) > 0) {
				addActionError(getText("error-bdelete-with-child", new String[] { c.getId().toString() }));
			}
		}

		return !hasActionErrors();
	}
	
	/**
	 * commitBulkDelete
	 * @param dataList data list
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void commitBulkDelete(List<Category> dataList) throws Exception {
		super.commitBulkDelete(dataList);
		
		addReloadMessage();

		for (Category c : dataList) {
			IOUtils.deltree(utils().getCategoryImagesPath(c));
		}
	}

	private void addReloadMessage() {
		addApplicationMessage(AC.CATEGORY_RELOAD, getText(RC.MESSAGE_CATEGORY_RELOAD));
	}
}

