/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.affiliate.action.task;

import java.util.List;

import nuts.core.collections.CollectionUtils;
import nuts.core.lang.NumberUtils;
import nuts.core.lang.StringUtils;
import nuts.core.util.Sorter;

import squirrels.affiliate.constant.GC;
import squirrels.affiliate.model.bean.Category;


/**
 */
@SuppressWarnings("serial")
public class GenreAction extends AbstractTaskAction {
	private static final String GENRE_LOAD_STATUS = "genre.load.status";
	private static final String DEFAULT_GENRE_STATUS = " : : ";
	private static final char GENRE_LEVEL_SEPARATOR = ':';
	private static final char GENRE_INDEX_SEPARATOR = '.';
	private static final int MAX_GENRE_LEVEL = StringUtils.split(DEFAULT_GENRE_STATUS, GENRE_LEVEL_SEPARATOR).length;
	private boolean loaded = false;
	
	/**
	 * clear
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	@SuppressWarnings("unchecked")
	public String clear() throws Exception {
		getCache().put(GENRE_LOAD_STATUS, "");
		return SUCCESS;
	}

	private void loadKeyword() throws Exception {
		getCache().remove(GC.KEYWORDS_CACHE_KEY);
		utils().getKeywordList();
	}

	/**
	 * load
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	@SuppressWarnings("unchecked")
	public String load() throws Exception {
		String status = (String)getCache().get(GENRE_LOAD_STATUS);

		Category rc = utils().getRootCategory();
		if (StringUtils.isEmpty(status)) {
			loadCategory(rc);
			loadKeyword();
			status = DEFAULT_GENRE_STATUS;
		}
		else {
			String[] ss = StringUtils.split(status, GENRE_LEVEL_SEPARATOR);
			if (ss.length != MAX_GENRE_LEVEL) {
				log.warn("Invalid " + GENRE_LOAD_STATUS + ": " + status);
				loadCategory(rc);
				status = DEFAULT_GENRE_STATUS;
			}
			else {
				_load(rc, ss, 0);
				if (loaded) {
					status = StringUtils.join(ss, GENRE_LEVEL_SEPARATOR);
				}
				else {
					status = DEFAULT_GENRE_STATUS;
				}
			}
		}
		
		getCache().put(GENRE_LOAD_STATUS, status);
		
		log.info(GENRE_LOAD_STATUS + ": " + status);
		return SUCCESS;
	}

	private void loadCategory(Category rc) throws Exception {
		rc.getChildren();
		
		getItemModel().ranking(rc.getId());

		int limit = getTextAsInt("list-pager-limit", 30);
		String sort = getText("list-sorter-column", "reviewCount");
		boolean asc = !getText("list-sorter-direction", "DESC").equals(Sorter.DESC);

		getItemModel().search(rc.getId(), null,
			sort, asc,
			1, limit);

		loaded = true;
	}

	private void _load(Category pc, String[] ss, int level) throws Exception {
		List<Category> cs = pc.getChildren();
		if (CollectionUtils.isEmpty(cs)) {
			return;
		}
		
		String[] ts = StringUtils.split(ss[level], GENRE_INDEX_SEPARATOR);
		int tlast = NumberUtils.toInt(ts[0], -1);
		int clast = 0;
		if (ts.length > 1) {
			clast = NumberUtils.toInt(ts[1], 0);
		}

		// load next category
		if (tlast + 1 < cs.size()) {
			tlast++;
			loadCategory(cs.get(tlast));
			ss[level] = String.valueOf(tlast) + GENRE_INDEX_SEPARATOR + String.valueOf(clast);
			return;
		}

		if (level + 1 < ss.length) {
			for ( ; clast < cs.size(); clast++) {
				_load(cs.get(clast), ss, level + 1);
				ss[level] = String.valueOf(tlast) + GENRE_INDEX_SEPARATOR + String.valueOf(clast);
				if (loaded) {
					break;
				}
				else {
					// clear children status
					for (int i = level + 1; i < ss.length; i++) {
						ss[i] = " ";
					}
				}
			}
		}
	}
}
