/*
 * This file is part of Nuts.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.affiliate.tool;

import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.util.List;

import nuts.aems.util.AppletHelper;
import nuts.core.io.CsvWriter;
import nuts.core.io.IOUtils;
import nuts.core.lang.StringEncodeUtils;
import nuts.core.lang.StringUtils;
import nuts.core.orm.dao.DataAccessSession;
import nuts.core.orm.dao.DataAccessUtils;
import nuts.core.tool.CommandTool;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import squirrels.affiliate.Application;
import squirrels.affiliate.model.bean.Category;
import squirrels.affiliate.model.dao.CategoryDAO;
import squirrels.affiliate.model.example.CategoryExample;
import squirrels.affiliate.webservice.rakuten.RakutenAPI;



/**
 */
public class CategorySitemap {
	private static Log log = LogFactory.getLog(CategorySitemap.class);

	private CsvWriter csv;
	
	private String ws = RakutenAPI.ID;
	private String root = "0";
	private int depth = 3;
	private String output;
	
	public static class Main extends CommandTool {
		@SuppressWarnings("static-access")
		protected void addCommandLineOptions(Options options) throws Exception {
			options.addOption(OptionBuilder.withArgName("web service")
				.hasArg()
				.withDescription("web service id (rakuten: r, yahoo: y)")
				.create("w"));

			options.addOption(OptionBuilder.withArgName("root")
				.hasArg()
				.withDescription("root category id (default: 0)")
				.create("r"));

			options.addOption(OptionBuilder.withArgName("depth")
				.hasArg()
				.withDescription("output category depth (default: 3)")
				.create("d"));

			options.addOption(OptionBuilder.withArgName("output")
				.hasArg()
				.withDescription("output file (default: stdout)")
				.create("o"));
		}

		protected void getCommandLineOptions(CommandLine cl, Options options) throws Exception {
			if (cl.hasOption("w")) {
				setParameter("ws", cl.getOptionValue("w").trim());
			}
			
			if (cl.hasOption("r")) {
				setParameter("root", cl.getOptionValue("r").trim());
			}
			
			if (cl.hasOption("d")) {
				setParameter("depth", cl.getOptionValue("d").trim());
			}
			
			if (cl.hasOption("o")) {
				setParameter("output", cl.getOptionValue("o").trim());
			}
		}

		/**
		 * main
		 * @param args arguments
		 */
		public static void main(String args[]) {
			Main main = new Main();
			main.execute(new CategorySitemap(), args);
		}
	}

	public CategorySitemap() {
	}

	/**
	 * @return the ws
	 */
	public String getWs() {
		return ws;
	}

	/**
	 * @param ws the ws to set
	 */
	public void setWs(String ws) {
		this.ws = ws;
	}

	/**
	 * @return the output
	 */
	public String getOutput() {
		return output;
	}

	/**
	 * @param output the output to set
	 */
	public void setOutput(String output) {
		this.output = output;
	}

	/**
	 * @return the root
	 */
	public String getRoot() {
		return root;
	}

	/**
	 * @param root the root to set
	 */
	public void setRoot(String root) {
		this.root = root;
	}

	/**
	 * @return the depth
	 */
	public int getDepth() {
		return depth;
	}

	/**
	 * @param depth the depth to set
	 */
	public void setDepth(int depth) {
		this.depth = depth;
	}

	private int count = 0;
	private int cdepth = 0;
	private void output(CategoryDAO cdao, String id) throws Exception {
		if (cdepth >= depth) {
			return;
		}

		cdepth++;

		CategoryExample cexp = cdao.createExample();
		cexp.invalid().isFalse()
			.vendor().equalTo(ws)
			.parentId().equalTo(id)
			.id().asc();

		double rate = (double)(6 - cdepth) / 10;
		List<Category> list = cdao.selectByExample(cexp);
		for (Category c : list) {
			String cid = StringUtils.removeStart(c.getId(), ws);
			csv.writeNext(new String[] { cid, String.valueOf(rate) });
			count++;
			output(cdao, c.getId());
		}
		
		cdepth--;
	}

	public void execute() throws Exception {
		if (depth > 5) {
			throw new IllegalArgumentException("depth must < 6");
		}
		if (StringUtils.isEmpty(output)) {
			csv = new CsvWriter(new OutputStreamWriter(System.out));
		}
		else {
			csv = new CsvWriter(new OutputStreamWriter(new FileOutputStream(output), StringEncodeUtils.UTF_8));
		}

		log.debug("depth=" + depth);
		log.debug("ws=" + ws);
		log.debug("root=" + root);
		log.debug("output=" + output);
		
		AppletHelper.init(Application.class, false);

		DataAccessSession das = Application.get().openDataAccessSession();

		CategoryDAO cdao = new CategoryDAO(das);
		try {
			output(cdao, ws + root);
			log.info(count + " records outputed to " + (output == null ? "stdout" : output));
		}
		finally {
			IOUtils.closeQuietly(csv);
			DataAccessUtils.closeQuietly(das);
			Application.get().destroy();
		}
	}
}
