/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.affiliate.action;

import java.util.ArrayList;
import java.util.List;

import nuts.core.util.Pager;
import nuts.core.util.SimpleQuery;
import nuts.core.util.Sorter;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

import squirrels.affiliate.model.ItemModel;
import squirrels.affiliate.model.bean.Category;
import squirrels.affiliate.model.bean.Item;

/**
 */
@SuppressWarnings("serial")
public class ItemAction extends AbstractAction {

	protected Pager pager = new Pager();

	protected Sorter sorter = new Sorter();

	protected SimpleQuery query = new SimpleQuery();
	
	protected String categoryId;

	protected Category category;

	protected List<Category> categoryLocation;
	
	protected String itemId;
	
	protected Item item;

	protected List itemList;

	protected List rankList;

	/**
	 * @return the categoryId
	 */
	public String getCategoryId() {
		return categoryId;
	}

	/**
	 * @param categoryId the categoryId to set
	 */
	public void setCategoryId(String categoryId) {
		this.categoryId = StringUtils.substringBefore(categoryId, "-");
	}

	/**
	 * @return the category
	 */
	public Category getCategory() {
		return category;
	}

	/**
	 * @return the categoryLocation
	 */
	public List<Category> getCategoryLocation() {
		return categoryLocation;
	}

	/**
	 * @return the itemId
	 */
	public String getItemId() {
		return itemId;
	}

	/**
	 * @param itemId the itemId to set
	 */
	public void setItemId(String itemId) {
		this.itemId = itemId;
	}

	/**
	 * @return the item
	 */
	public Item getItem() {
		return item;
	}

	/**
	 * @return the itemList
	 */
	public List getItemList() {
		return itemList;
	}

	/**
	 * @return the rankList
	 */
	public List getRankList() {
		return rankList;
	}

	//-----------------------------------------------------------------------
	// short name
	//-----------------------------------------------------------------------
	/**
	 * @return the pager
	 */
	public Pager getPg() {
		return pager;
	}

	/**
	 * @param pager the pager to set
	 */
	public void setPg(Pager pager) {
		this.pager = pager;
	}

	/**
	 * @return the sorter
	 */
	public Sorter getSo() {
		return sorter;
	}

	/**
	 * @param sorter the sorter to set
	 */
	public void setSo(Sorter sorter) {
		this.sorter = sorter;
	}

	/**
	 * @return the query
	 */
	public SimpleQuery getQq() {
		return query;
	}

	/**
	 * @param query the query to set
	 */
	public void setQq(SimpleQuery query) {
		this.query = query;
	}

	/**
	 * @return the itemId
	 */
	public String getIid() {
		return itemId;
	}

	/**
	 * @param itemId the itemId to set
	 */
	public void setIid(String itemId) {
		this.itemId = itemId;
	}

	/**
	 * @return the categoryId
	 */
	public String getGid() {
		return categoryId;
	}

	/**
	 * @param categoryId the categoryId to set
	 */
	public void setGid(String categoryId) {
		setCategoryId(categoryId);
	}

	private void verifySort() {
		String[] ss = new String[] { "reviewAverage", "reviewCount", "itemPrice"};
		if (!ArrayUtils.contains(ss, sorter.getColumn())) {
			sorter.setColumn(ss[0]);
			sorter.setDirection(Sorter.DESC);
		}
	}
	
	/**
	 * rank
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String rank() throws Exception {
		setCategoryId(utils().getActionName());
		if (categoryId != null) {
			processLocation(categoryId);
			
			rankList = getItemModel().ranking(categoryId);
		}

		return LIST;
	}
	
	/**
	 * category
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String category() throws Exception {
		setCategoryId(utils().getActionName());
		return list();
	}
	
	public String item() throws Exception {
		String itemCode = utils().getActionName();
		item = getItemModel().getItem(itemCode);
		return SUCCESS;
	}
	
	/**
	 * list
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String list() throws Exception {
		query.setKey(null);

		utils().loadLimitParams(pager);
		utils().loadSorterParams(sorter);
		verifySort();
		
		if (categoryId != null) {
			processLocation(categoryId);
			
			int p = pager.getPage();
			if (p > 100) {
				p = 100;
				pager.setStart((p - 1) * pager.getLimit());
			}

			ItemModel.Result result = getItemModel().search(categoryId, null,
				(sorter.getDirection().equals(Sorter.DESC) ? "-" : "+") + sorter.getColumn(),
				p, pager.getLimit());

			pager.setTotal(result.getCount());
			pager.normalize();
			itemList = result.getItems();
			
			rankList = getItemModel().ranking(categoryId);
		}

		utils().saveLimitParams(pager);
		utils().saveSorterParams(sorter);

		return LIST;
	}

	/**
	 * search
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String search() throws Exception {
		if (StringUtils.isEmpty(query.getKey())) {
			return list();
		}

		utils().loadLimitParams(pager);
		utils().loadSorterParams(sorter);

		verifySort();

		if (categoryId != null) {
			processLocation(categoryId);
		}

		int p = pager.getPage();
		if (p > 100) {
			p = 100;
			pager.setStart((p - 1) * pager.getLimit());
		}

		ItemModel.Result result = getItemModel().search(categoryId, 
			query.getKey(),
			(sorter.getDirection().equals(Sorter.DESC) ? "-" : "+") + sorter.getColumn(),
			p, pager.getLimit());

		pager.setTotal(result.getCount());
		pager.normalize();
		itemList = result.getItems();

		utils().saveLimitParams(pager);
		utils().saveSorterParams(sorter);
		
		return LIST;
	}

	/**
	 * process location
	 * @param categoryId catgoryId
	 */
	protected void processLocation(String categoryId) throws Exception {
		Category c = utils().getCategory(categoryId);
		if (c != null) {
			category = c;
			categoryLocation = new ArrayList<Category>();
			while (true) {
				categoryLocation.add(0, c);
				c = c.getParent();
				if (c == null || c.equals(utils().getRootCategory())) {
					break;
				}
			}
		}
	}
}

