/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action;

import java.util.Calendar;

import nuts.core.lang.NumberUtils;

import org.apache.commons.lang.StringUtils;

import squirrels.ecshop.bean.Cart;
import squirrels.ecshop.bean.CartItem;
import squirrels.ecshop.constant.RC;
import squirrels.ecshop.model.bean.Goods;
import squirrels.ecshop.model.dao.GoodsDAO;


/**
 * CartAction
 */
@SuppressWarnings("serial")
public class CartAction extends AbstractAction {

	private CartItem cartItem = new CartItem();

	/**
	 * @return the cart
	 */
	public CartItem getCi() {
		return cartItem;
	}

	/**
	 * @param cart the cart to set
	 */
	public void setCi(CartItem cart) {
		this.cartItem = cart;
	}

	/**
	 * @return the cart
	 * @throws Exception if an error occurs
	 */
	public Cart getCart() throws Exception {
		Cart cart = utils().getUserCart(getDataAccessSession());
		return cart;
	}

	/**
	 * if cart is empty then add some message
	 * @throws Exception if an error occurs
	 */
	protected void checkCart() throws Exception {
		Cart cart = getCart();
		if (cart.isEmpty()) {
			addActionMessage(getText(RC.MESSAGE_CART_EMPTY));
		}
	}

	/**
	 * list
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String list() throws Exception {
		checkCart();
		return LIST;
	}

	/**
	 * edit
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String edit() throws Exception {
		checkCart();
		return EDIT;
	}

	/**
	 * add
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String add() throws Exception {
		if (StringUtils.isNotEmpty(cartItem.getGoodsId()) 
				&& NumberUtils.isLong(cartItem.getGoodsId())) {
			if (cartItem.getQuantity() == null) {
				cartItem.setQuantity(1);
			}
			Cart cart = getCart();
			for (CartItem ci : cart.getItems()) {
				if (ci.getGoodsId().equals(cartItem.getGoodsId())) {
					ci.setQuantity(ci.getQuantity() + cartItem.getQuantity());
					ci.setAddDate(Calendar.getInstance().getTime());
					cart.getItems().remove(ci);
					cart.getItems().add(0, ci);
					utils().saveCartToCookie(cart);
					return LIST;
				}
			}

			Integer max = getTextAsInt("cart-items-max", 20);
			if (cart.getItems().size() >= max) {
				addActionError(getText("error-cart-overflow", new String[] { String.valueOf(max) }));
				return LIST;
			}

			GoodsDAO dao = new GoodsDAO(getDataAccessSession());
			Goods goods = dao.selectByPrimaryKey(NumberUtils.toLong(cartItem.getGoodsId()));
			if (goods != null) {
				cartItem.setGoods(goods);
				cartItem.setAddDate(Calendar.getInstance().getTime());
				cart.getItems().add(0, cartItem);
				utils().saveCartToCookie(cart);
				return LIST;
			}
			else {
				addActionError(getText("error-goods-id",
						new String[] { String.valueOf(cartItem.getGoodsId())}));
			}
		}
		return list();
	}

	/**
	 * change
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String change() throws Exception {
		Cart cart = getCart();
		if (cart.isEmpty()) {
			addActionMessage(getText(RC.MESSAGE_CART_EMPTY));
			return EDIT;
		}

		if (cartItem.getGoodsId() == null) {
			return EDIT;
		}

		if (cartItem.getQuantity() == null || cartItem.getQuantity() < 1) {
			addActionError(getText("error-goods-quantity"));
			return EDIT;
		}

		for (CartItem c : cart.getItems()) {
			if (c.getGoodsId().equals(cartItem.getGoodsId())) {
				c.setQuantity(cartItem.getQuantity());
				c.setAddDate(Calendar.getInstance().getTime());
				utils().saveCartToCookie(cart);
				break;
			}
		}
		return EDIT;
	}

	/**
	 * delete
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String delete() throws Exception {
		Cart cart = getCart();
		if (cart.isEmpty()) {
			addActionMessage(getText(RC.MESSAGE_CART_EMPTY));
			return EDIT;
		}

		if (cartItem.getGoodsId() == null) {
			return EDIT;
		}

		for (CartItem c : cart.getItems()) {
			if (c.getGoodsId().equals(cartItem.getGoodsId())) {
				cart.getItems().remove(c);
				utils().saveCartToCookie(cart);
				break;
			}
		}

		if (cart.isEmpty()) {
			addActionMessage(getText(RC.MESSAGE_CART_EMPTY));
		}
		return EDIT;
	}
}
