/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action.admin;

import java.io.StringReader;
import java.io.StringWriter;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import nuts.core.collections.CollectionUtils;
import nuts.core.io.IOUtils;
import nuts.core.lang.StringUtils;
import nuts.exts.fileupload.UploadManager;

import org.apache.commons.vfs2.FileObject;

import squirrels.ecshop.action.admin.goods.GoodsAction;
import squirrels.ecshop.bean.ImageInfo;
import squirrels.ecshop.model.bean.Goods;
import squirrels.ecshop.model.dao.UserOrderItemDAO;
import squirrels.ecshop.model.example.GoodsExample;
import squirrels.ecshop.model.example.UserOrderItemExample;
import squirrels.ecshop.model.metadata.GoodsMetaData;
import squirrels.ecshop.util.LuceneUtils;
import freemarker.template.Configuration;
import freemarker.template.Template;


/**
 * GoodsExAction
 */
@SuppressWarnings("serial")
public class GoodsExAction extends GoodsAction {

	/**
	 * Constructor
	 */
	public GoodsExAction() {
	}

	/**
	 * queryListByExample
	 * @param example example
	 */
	@Override
	protected void queryListByExample(GoodsExample example) throws Exception {
		example.content().exclude();
		super.queryListByExample(example);
	}


	/**
	 * @return image list
	 */
	public List<ImageInfo> getImageList() {
		try {
			if (images == null && getData() != null && getData().getId() != null) {
				images = new ArrayList<ImageInfo>();

				FileObject dir = utils().getGoodsImagesPath(getData());
				if (dir.exists()) {
					for (FileObject f : dir.getChildren()) {
						ImageInfo ii = new ImageInfo();
						ii.setName(f.getName().getBaseName());
						ii.setTime(f.getContent().getLastModifiedTime());
						ii.setDisplay(StringUtils.contains(getData().getThumbs(), ii.getName()));
						images.add(ii);
					}
					Collections.sort(images);
				}
			}
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
		return images;
	}

	private void processThumbs(Goods data) throws Exception {
		if (CollectionUtils.isEmpty(images)) {
			data.setThumbs(null);
		}
		else {
			List<String> thumbs = new ArrayList<String>(
					Arrays.asList(StringUtils.split(StringUtils.defaultString(data.getThumbs()))));
			
			for (int i = thumbs.size() - 1; i >=0; i--) {
				boolean exists = false;
				for (ImageInfo ii : images) {
					if (StringUtils.isNotEmpty(ii.getName())
							&& !ImageInfo.DELETE.equalsIgnoreCase(ii.getAction())) {
						if (ii.getName().equals(thumbs.get(i))) {
							exists = true;
							break;
						}
					}
					else if (StringUtils.isNotEmpty(ii.getTemp())
							&& ImageInfo.INSERT.equalsIgnoreCase(ii.getAction())) {
						if (ii.getTemp().equals(thumbs.get(i))) {
							exists = true;
							break;
						}
					}
				}
				if (!exists) {
					thumbs.remove(i);
				}
			}
			
			data.setThumbs(StringUtils.join(thumbs, '\n'));
		}
	}
	
	private void processImages(Goods data) throws Exception {
		if (CollectionUtils.isNotEmpty(images)) {
			FileObject gi = utils().getGoodsImagesPath(data);
			if (!gi.exists()) {
				gi.createFolder();
			}

			for (ImageInfo ii : images) {
				if (ImageInfo.INSERT.equalsIgnoreCase(ii.getAction())
						&& StringUtils.isNotEmpty(ii.getTemp())) {
					FileObject ft = UploadManager.resolveUploadFile(ii.getTemp());

					FileObject fi = gi.resolveFile(ft.getName().getBaseName());

					ft.moveTo(fi);
				}
				else if (ImageInfo.DELETE.equalsIgnoreCase(ii.getAction())
						&& StringUtils.isNotEmpty(ii.getName())) {
					FileObject fi = gi.resolveFile(ii.getName());
					if (fi.exists()) {
						fi.delete();
					}
				}
			}
			images = null;
		}
	}

	/**
	 * content syntax check
	 * @param g goods
	 * @return true if check success
	 */
	protected boolean checkContent(Goods g) {
		g.setDiscount(100 - g.getSalePrice().multiply(new BigDecimal(100)).divide(g.getListPrice()).intValue());
		g.setStockQuantity(g.getSupplyQuantity() - g.getSoldQuantity() - g.getOrderQuantity());

		if (StringUtils.isNotEmpty(g.getContent())) {
			try {
				Template tpl = new Template("content", 
					new StringReader(g.getContent()), 
					new Configuration());
				tpl.process(g, new StringWriter());
			}
			catch (Exception e) {
				addFieldError(getDataFieldName(GoodsMetaData.PN_CONTENT), e.getMessage());
				return false;
			}
		}
		return true;
	}
	

	/**
	 * checkOnInsert
	 * @param data data
	 * @return true if check success
	 * @throws Exception
	 */
	@Override
	protected boolean checkOnInsert(Goods data) throws Exception {
		processThumbs(data);

		boolean c = super.checkOnInsert(data);
		if (!c && isCheckAbortOnError()) {
			return false;
		}
		if (!checkContent(data)) {
			c = false;
		}
		return c;
	}
	
	/**
	 * commitInsert
	 * @param data data
	 * @throws Exception  if an error occurs
	 */
	@Override
	protected void commitInsert(Goods data) throws Exception {
		super.commitInsert(data);

		processImages(data);

		if (!data.beInvalid()) {
			LuceneUtils.updateIndex(data);
		}
	}

	/**
	 * checkOnUpdate
	 * @param data data
	 * @param srcData source Data
	 * @return true if check success
	 * @throws Exception
	 */
	@Override
	protected boolean checkOnUpdate(Goods data, Goods srcData) throws Exception {
		processThumbs(data);

		boolean c = super.checkOnUpdate(data, srcData);
		if (!c && isCheckAbortOnError()) {
			return false;
		}
		if (!checkContent(data)) {
			c = false;
		}
		return c;
	}

	/**
	 * commitUpdate
	 * @param data data
	 * @param srcData source Data
	 */
	@Override
	protected void commitUpdate(Goods data, Goods srcData) throws Exception {
		super.commitUpdate(data, srcData);

		processImages(data);

		LuceneUtils.updateIndex(data);
	}

	/**
	 * checkOnDelete
	 * @param data data
	 * @param srcData source data
	 * @return true - continue, false - skip
	 */
	@Override
	protected boolean checkOnDelete(Goods data, Goods srcData) throws Exception {
		if (!super.checkOnDelete(data, srcData)) {
			return false;
		}
		
		UserOrderItemDAO coiDao = new UserOrderItemDAO(getDataAccessSession());
		UserOrderItemExample coiExp = coiDao.createExample();
		coiExp.goodsId().equalTo(data.getId()).setLimit(1);
		if (coiDao.countByExample(coiExp) > 0) {
			addActionError(getText("error-delete-with-order"));
			return false;
		}

		return true;
	}
	
	/**
	 * commitDelete
	 * @param data data
	 */
	@Override
	protected void commitDelete(Goods data) throws Exception {
		super.commitDelete(data);

		IOUtils.deltree(utils().getGoodsImagesPath(data));

		LuceneUtils.deleteIndex(data);
	}

	/**
	 * checkOnBulkDelete
	 * @param data data
	 * @return true - continue, false - skip
	 */
	@Override
	protected boolean checkOnBulkDelete(List<Goods> dataList) throws Exception {
		if (!super.checkOnBulkDelete(dataList)) {
			return false;
		}

		UserOrderItemDAO coiDao = new UserOrderItemDAO(getDataAccessSession());

		for (Goods g : dataList) {
			UserOrderItemExample coiExp = coiDao.createExample();
			coiExp.goodsId().equalTo(g.getId()).setLimit(1);
			if (coiDao.countByExample(coiExp) > 0) {
				addActionError(getText("error-bdelete-with-order", new String[] { g.getId().toString() }));
			}
		}

		return !hasActionErrors();
	}

	/**
	 * commitBulkDelete
	 * @param dataList dataList
	 */
	@Override
	protected void commitBulkDelete(List<Goods> dataList) throws Exception {
		super.commitBulkDelete(dataList);
		
		for (Goods g : dataList) {
			IOUtils.deltree(utils().getGoodsImagesPath(g));
		}

		LuceneUtils.deleteIndexes(dataList);
	}
	
	/**
	 * bindexExecute
	 * @return SUCCESS
	 * @throws Exception if an error occurs
	 */
	public String bindexExecute() throws Exception {
		if (CollectionUtils.isNotEmpty(getDataList())) {
			LuceneUtils.updateIndexes(getDataList());
			addActionMessage(getMessage(ACTION_SUCCESS_PREFIX + getActionScenario(), 
					new String[] { String.valueOf(getDataList().size()) }));
		}
		setMethodResult(RESULT_SUCCESS);
		return SUCCESS;
	}

	/**
	 * reindex
	 * @return result name
	 * @throws Exception  if an error occurs
	 */
	public String reindex() throws Exception {
		addActionConfirm(getText("confirm-reindex"));
		return SUCCESS;
	}
}
