/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action.admin;

import java.util.Calendar;
import java.util.List;

import squirrels.ecshop.action.admin.userorder.UserOrderAction;
import squirrels.ecshop.constant.VC;
import squirrels.ecshop.model.bean.UserOrder;
import squirrels.ecshop.model.bean.UserOrderItem;
import squirrels.ecshop.model.dao.UserOrderItemDAO;
import squirrels.ecshop.model.example.UserOrderItemExample;



/**
 * UserOrderExAction
 */
@SuppressWarnings("serial")
public class UserOrderExAction extends UserOrderAction {

	private boolean syncItemStatus = true;
	
	/**
	 * Constructor
	 */
	public UserOrderExAction() {
		super();
	}

	/**
	 * @return the syncItemStatus
	 */
	public boolean isSyncItemStatus() {
		return syncItemStatus;
	}

	/**
	 * @param syncItemStatus the syncItemStatus to set
	 */
	public void setSyncItemStatus(boolean syncItemStatus) {
		this.syncItemStatus = syncItemStatus;
	}

	/**
	 * startUpdate
	 * @param data data
	 * @param srcData srcData
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startUpdate(UserOrder data, UserOrder srcData) throws Exception {
		super.startUpdate(data, srcData);

		if (VC.ORDER_STATUS_BEGIN.equals(data.getStatus())) {
			data.setShipDate(null);
			data.setEndDate(null);
		}
		else if (VC.ORDER_STATUS_PREPARE.equals(data.getStatus())) {
			data.setShipDate(null);
			data.setEndDate(null);
		}
		else if (VC.ORDER_STATUS_SHIPPING.equals(data.getStatus())) {
			if (data.getShipDate() == null) {
				if (srcData.getShipDate() != null) {
					data.setShipDate(srcData.getShipDate());
				}
				else {
					data.setShipDate(Calendar.getInstance().getTime());
				}
			}
		}
		else if (VC.ORDER_STATUS_FINISHED.equals(data.getStatus())) {
			if (data.getShipDate() == null) {
				if (srcData.getShipDate() != null) {
					data.setShipDate(srcData.getShipDate());
				}
				else {
					data.setShipDate(Calendar.getInstance().getTime());
				}
			}
			if (data.getEndDate() == null) {
				if (srcData.getEndDate() != null) {
					data.setEndDate(srcData.getEndDate());
				}
				else {
					data.setEndDate(Calendar.getInstance().getTime());
				}
			}
		}
	}

	/**
	 * commitUpdate
	 * @param data data
	 * @param srcData source data
	 * @throws Exception  if an error occurs
	 */
	@Override
	protected void commitUpdate(UserOrder data, UserOrder srcData) throws Exception {
		if (!data.beInvalid()) {
			if (!srcData.beInvalid()) {
				updateStock(data.getId(), false);
				updateItems(data.getId(), false);
			}
			
			if (syncItemStatus) {
				UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());

				UserOrderItemExample example = new UserOrderItemExample();
				example.userOrderId().equalTo(data.getId()).and().status().notEqualTo(data.getStatus());
				
				List<UserOrderItem> list = (List<UserOrderItem>)idao.selectByExample(example);
				for (UserOrderItem coi : list) {
					if (VC.ORDER_STATUS_FINISHED.equals(data.getStatus())) {
						utils().daoUpdateStockQuantity(getDataAccessSession(), coi.getGoodsId(), -coi.getQuantity(), coi.getQuantity(), null);
					}
					else if (VC.ORDER_STATUS_FINISHED.equals(coi.getStatus())) {
						utils().daoUpdateStockQuantity(getDataAccessSession(), coi.getGoodsId(), coi.getQuantity(), -coi.getQuantity(), null);
					}
				}
				
				UserOrderItem coi = new UserOrderItem();
				coi.setStatus(data.getStatus());
				idao.updateByExampleSelective(coi, example);
			}
		}
		else { 
			if (!srcData.beInvalid()) {
				updateStock(data.getId(), true);
				updateItems(data.getId(), true);
			}

			if (syncItemStatus) {
				UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());

				UserOrderItemExample example = new UserOrderItemExample();
				example.userOrderId().equalTo(data.getId()).and().status().notEqualTo(data.getStatus());
				
				UserOrderItem coi = new UserOrderItem();
				coi.setStatus(data.getStatus());
				idao.updateByExampleSelective(coi, example);
			}
		}

		super.commitUpdate(data, srcData);
	}

	private void updateStock(Long coId, boolean delete) throws Exception {
		UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());
		UserOrderItemExample iexp = new UserOrderItemExample();
		iexp.userOrderId().equalTo(coId).and().invalid().notEqualTo(delete);

		List<UserOrderItem> list = (List<UserOrderItem>)idao.selectByExample(iexp);
		for (UserOrderItem coi : list) {
			Integer quantity = delete ? -coi.getQuantity() : coi.getQuantity();
			if (VC.ORDER_STATUS_FINISHED.equals(coi.getStatus())) {
				utils().daoUpdateStockQuantity(getDataAccessSession(), coi.getGoodsId(), null, quantity, null);
			}
			else {
				utils().daoUpdateStockQuantity(getDataAccessSession(), coi.getGoodsId(), quantity, null, null);
			}
		}
	}
	
	private void updateItems(Long coId, boolean delete) throws Exception {
		UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());

		UserOrderItemExample iexp = new UserOrderItemExample();
		iexp.userOrderId().equalTo(coId).and().invalid().notEqualTo(delete);

		UserOrderItem coi = new UserOrderItem();
		coi.setInvalid(delete);
		idao.updateByExampleSelective(coi, iexp);
	}
	
	/**
	 * startDelete
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startDelete(UserOrder data) throws Exception {
		super.startDelete(data);
		deleteOrderItem(data);
	}
	
	private void deleteOrderItem(UserOrder data) throws Exception {
		if (!data.beInvalid()) {
			updateStock(data.getId(), true);
		}

		UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());
		UserOrderItemExample iexp = new UserOrderItemExample();
		iexp.userOrderId().equalTo(data.getId());
		idao.deleteByExample(iexp);
	}

	/**
	 * startBulkDelete
	 * @param dataList data list
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startBulkDelete(List<UserOrder> dataList) throws Exception {
		super.startBulkDelete(dataList);
		
		for (UserOrder co : dataList) {
			deleteOrderItem(co);
		}
	}
}
