/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action.user;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import nuts.aems.model.bean.User;
import nuts.core.lang.StringUtils;

import org.apache.commons.mail.EmailException;

import squirrels.ecshop.bean.Cart;
import squirrels.ecshop.bean.CartItem;
import squirrels.ecshop.constant.RC;
import squirrels.ecshop.constant.TPL;
import squirrels.ecshop.constant.VC;
import squirrels.ecshop.model.bean.UserAddress;
import squirrels.ecshop.model.bean.UserCreditCard;
import squirrels.ecshop.model.bean.UserOrder;
import squirrels.ecshop.model.bean.UserOrderItem;
import squirrels.ecshop.model.dao.UserAddressDAO;
import squirrels.ecshop.model.dao.UserCreditCardDAO;
import squirrels.ecshop.model.dao.UserOrderDAO;
import squirrels.ecshop.model.dao.UserOrderItemDAO;
import squirrels.ecshop.model.example.UserAddressExample;
import squirrels.ecshop.model.example.UserCreditCardExample;


/**
 * OrderAction
 */
@SuppressWarnings("serial")
public class OrderAction extends AbstractUserAction {

	private Cart cart;
	private List<String> goodsIdList;

	private Long userAddressId;
	private UserAddress userAddress;
	private List<UserAddress> userAddressList;
	
	private Long userCreditCardId;
	private UserCreditCard userCreditCard;
	private List<UserCreditCard> userCreditCardList;
	
	private UserOrder userOrder = new UserOrder();
	
	private static final String STEP_GOODS = "goods";
	private static final String STEP_ADDRESS = "address";
	private static final String STEP_DELIVERY = "delivery";
	private static final String STEP_PAYMENT = "payment";
	private static final String STEP_CONFIRM = "confirm";
	private static final String STEP_SUCCESS = "success";
	
	private String step = STEP_GOODS;

	/**
	 * Constructor
	 */
	public OrderAction() {
	}

	/**
	 * @return the cartList
	 * @throws Exception if an error occurs 
	 */
	public Cart getCart() throws Exception {
		if (cart == null) {
			cart = utils().getUserCart(getDataAccessSession());
		}
		return cart;
	}

	/**
	 * @return the userAddressList
	 * @throws Exception if an error occurs 
	 */
	public List<UserAddress> getUserAddressList() throws Exception {
		if (userAddressList == null) {
			UserAddressDAO dao = new UserAddressDAO(getDataAccessSession());
			UserAddressExample example = dao.createExample();
			example.userId().equalTo(utils().getLoginUser().getId())
				.and().invalid().isFalse();
			userAddressList = dao.selectByExample(example);
		}
		return userAddressList;
	}

	/**
	 * @return the userCredirtCardList
	 * @throws Exception if an error occurs 
	 */
	public List<UserCreditCard> getUserCreditCardList() throws Exception {
		if (userCreditCardList == null) {
			UserCreditCardDAO dao = new UserCreditCardDAO(getDataAccessSession());
			UserCreditCardExample example = dao.createExample();
			example.userId().equalTo(utils().getLoginUser().getId())
				.and().invalid().isFalse();
			userCreditCardList = dao.selectByExample(example);
		}
		return userCreditCardList;
	}

	/**
	 * @return the goodsIdList
	 */
	public List<String> getGids() {
		return goodsIdList;
	}

	/**
	 * @param goodsIdList the goodsIdList to set
	 */
	public void setGids(List<String> goodsIdList) {
		this.goodsIdList = goodsIdList;
	}

	/**
	 * @return the userAddress
	 */
	public UserAddress getCa() {
		return userAddress;
	}

	/**
	 * @param userAddress the userAddress to set
	 */
	public void setCa(UserAddress userAddress) {
		this.userAddress = userAddress;
	}

	/**
	 * @return the userAddressId
	 */
	public Long getCaId() {
		return userAddressId;
	}

	/**
	 * @param userAddressId the userAddressId to set
	 */
	public void setCaId(Long userAddressId) {
		this.userAddressId = userAddressId;
	}

	/**
	 * @return the userCreditCardId
	 */
	public Long getCccId() {
		return userCreditCardId;
	}

	/**
	 * @param userCreditCardId the userCreditCardId to set
	 */
	public void setCccId(Long userCreditCardId) {
		this.userCreditCardId = userCreditCardId;
	}

	/**
	 * @return the userCreditCard
	 */
	public UserCreditCard getCcc() {
		return userCreditCard;
	}

	/**
	 * @param userCreditCard the userCreditCard to set
	 */
	public void setCcc(UserCreditCard userCreditCard) {
		this.userCreditCard = userCreditCard;
	}

	/**
	 * @return the userOrder
	 */
	public UserOrder getCo() {
		return userOrder;
	}

	/**
	 * @param userOrder the userOrder to set
	 */
	public void setCo(UserOrder userOrder) {
		this.userOrder = userOrder;
	}

	/**
	 * @return the step
	 */
	public String getStep() {
		return step;
	}

	/**
	 * @param step the step to set
	 */
	public void setStep(String step) {
		this.step = step;
	}

	/**
	 * @see nuts.exts.xwork2.ActionSupport#getInputResultName()
	 */
	@Override
	public String getInputResultName() {
		if (STEP_CONFIRM.equals(step)) {
			return CONFIRM;
		}
		else {
			return INPUT;
		}
	}

	/**
	 * input
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String input() throws Exception {
		Cart cart = getCart();
		if (cart.isEmpty()) {
			addActionError(getText(RC.MESSAGE_CART_EMPTY));
		}

		if (hasErrors()) {
			return INPUT;
		}
		return selectGoods();
	}
	
	/**
	 * selectGoods
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectGoods() throws Exception {
		step = STEP_GOODS;

		if (goodsIdList == null) {
			goodsIdList = new ArrayList<String>();
			Cart cart = getCart();
			for (CartItem ci : cart.getItems()) {
				goodsIdList.add(ci.getGoodsId());
			}
		}

		return INPUT;
	}

	/**
	 * selectGoodsConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectGoodsConfirm() throws Exception {
		return selectAddress();
	}
	
	/**
	 * selectAddress
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectAddress() throws Exception {
		step = STEP_ADDRESS;
		return INPUT;
	}

	/**
	 * selectAddressConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectAddressConfirm() throws Exception {
		for (UserAddress ca : getUserAddressList()) {
			if (userAddressId.equals(ca.getId())) {
				return selectDelivery();
			}
		}

		step = STEP_ADDRESS;
		addActionError(getText("error-address-not-exists"));

		return INPUT;
	}

	/**
	 * inputAddressConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String inputAddressConfirm() throws Exception {
		userAddressId = null;
		return selectDelivery();
	}

	/**
	 * selectDelivery
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectDelivery() throws Exception {
		step = STEP_DELIVERY;

		if (StringUtils.isEmpty(userOrder.getDeliveryKind())) {
			userOrder.setDeliveryKind(getText("delivery-kind-default", ""));
		}
		if (StringUtils.isEmpty(userOrder.getDeliveryTime())) {
			userOrder.setDeliveryTime(VC.DELIVERY_TIME_DEFAULT);
		}
		return INPUT;
	}

	/**
	 * selectDeliveryConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectDeliveryConfirm() throws Exception {
		return selectPayment();
	}

	/**
	 * selectPayment
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectPayment() throws Exception {
		step = STEP_PAYMENT;
		if (StringUtils.isEmpty(userOrder.getPaymentKind())) {
			userOrder.setPaymentKind(VC.PAYMENT_KIND_AFTER);
		}
		return INPUT;
	}

	/**
	 * selectPaymentConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectPaymentConfirm() throws Exception {
		return confirm();
	}

	/**
	 * selectCreditCardConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String selectCreditCardConfirm() throws Exception {
		step = STEP_PAYMENT;
		for (UserCreditCard ccc : getUserCreditCardList()) {
			if (userCreditCardId.equals(ccc.getId())) {
				userOrder.setPaymentKind(VC.PAYMENT_KIND_CREDITCARD);
				return confirm();
			}
		}
		addActionError(getText("error-creditcard-not-exists"));
		return confirm();
	}

	/**
	 * validateInputCreditCardConfirm
	 * @throws Exception if an error occurs
	 */
	public void validateInputCreditCardConfirm() throws Exception {
		if (userCreditCard != null
				&& StringUtils.isNotEmpty(userCreditCard.getCardNo())) {
			User user = utils().getLoginUser();

			UserCreditCardDAO dao = new UserCreditCardDAO(getDataAccessSession());
			UserCreditCardExample example = dao.createExample();
			example.userId().equalTo(user.getId())
				.and().cardNo().equalTo(userCreditCard.getCardNo());
			if (dao.countByExample(example) > 0) {
				addFieldError("ccc.cardNo", getText("error-cardno-duplicate"));
			}
			
		}
	}
	
	/**
	 * inputCreditCardConfirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String inputCreditCardConfirm() throws Exception {
		userOrder.setPaymentKind(VC.PAYMENT_KIND_CREDITCARD);
		return confirm();
	}

	/**
	 * confirm
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String confirm() throws Exception {
		step = STEP_CONFIRM;
		
		if (userAddressId != null) {
			for (UserAddress ca : getUserAddressList()) {
				if (userAddressId.equals(ca.getId())) {
					userAddress = ca;
					break;
				}
			}
		}
		if (userCreditCardId != null) {
			for (UserCreditCard ccc : getUserCreditCardList()) {
				if (userCreditCardId.equals(ccc.getId())) {
					userCreditCard = ccc;
					break;
				}
			}
		}
		return CONFIRM;
	}

	/**
	 * validateExecute
	 * @throws Exception if an error occurs
	 */
	public void validateExecute() throws Exception {
		if (VC.PAYMENT_KIND_CREDITCARD.equals(userOrder.getPaymentKind())) {
			if (userCreditCard == null) {
				addFieldError("ccc.cardNo", "validation-required");
				addFieldError("ccc.cardType", "validation-required");
				addFieldError("ccc.cardName", "validation-required");
				addFieldError("ccc.cardExpires", "validation-required");
				return;
			}
			
			if (StringUtils.isEmpty(userCreditCard.getCardNo())) {
				addFieldError("ccc.cardNo", "validation-required");
			}
			if (StringUtils.isEmpty(userCreditCard.getCardType())) {
				addFieldError("ccc.cardType", "validation-required");
			}
			if (StringUtils.isEmpty(userCreditCard.getCardName())) {
				addFieldError("ccc.cardName", "validation-required");
			}
			if (userCreditCard.getCardExpires() == null) {
				addFieldError("ccc.cardExpires", "validation-required");
			}

			if (userCreditCardId == null) {
				validateInputCreditCardConfirm();
			}
		}
	}

	/**
	 * execute
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String execute() throws Exception {
		Cart cart = getCart();

		List<CartItem> orderCartItems = new ArrayList<CartItem>();
		for (CartItem ci : cart.getItems()) {
			if (goodsIdList.contains(ci.getGoodsId())) {
				orderCartItems.add(ci);
			}
		}
		
		if (orderCartItems.isEmpty()) {
			addActionError(getText("error-goods-not-exists"));
			return INPUT;
		}

		User user = utils().getLoginUser();

		UserOrderDAO dao = new UserOrderDAO(getDataAccessSession());
		UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());

		if (userAddressId == null) {
			UserAddressDAO caDao = new UserAddressDAO(getDataAccessSession());
			userAddress.setId(null);
			userAddress.setUserId(user.getId());
			userAddress.setInvalid(false);
			userAddress.setCusid(user.getId());
			userAddress.setCtime(Calendar.getInstance().getTime());
			userAddress.setUusid(user.getId());
			userAddress.setUtime(Calendar.getInstance().getTime());
			caDao.insert(userAddress);
		}
		if (VC.PAYMENT_KIND_CREDITCARD.equals(userOrder.getPaymentKind())
				&& userCreditCardId == null) {
			UserCreditCardDAO cccDao = new UserCreditCardDAO(getDataAccessSession());
			userCreditCard.setId(null);
			userCreditCard.setUserId(user.getId());
			userCreditCard.setInvalid(false);
			userCreditCard.setCusid(user.getId());
			userCreditCard.setCtime(Calendar.getInstance().getTime());
			userCreditCard.setUusid(user.getId());
			userCreditCard.setUtime(Calendar.getInstance().getTime());
			cccDao.insert(userCreditCard);
		}
		
		BigDecimal orderPrice = new BigDecimal(0);
		BigDecimal shipCost = new BigDecimal(0);
		for (CartItem c : orderCartItems) {
			orderPrice = orderPrice.add(c.getGoods().getSalePrice().multiply(
				new BigDecimal(c.getQuantity())));
			shipCost = shipCost.add(c.getGoods().getShipCost().multiply(
				new BigDecimal(c.getQuantity())));
		}

		UserOrder order = new UserOrder();
		
		order.setId(null);
		order.setCode(utils().genUserOrderCode());
		order.setStatus(VC.ORDER_STATUS_BEGIN);
		order.setUserId(user.getId());
		
		order.setOrderPrice(orderPrice);
		order.setShipCost(shipCost);
		
		order.setToName(userAddress.getName());
		order.setToZipcode(userAddress.getZipcode());
		order.setToAddress(userAddress.getAddress());
		order.setToCompany(userAddress.getCompany());
		order.setToPhone(userAddress.getPhone());
		
		order.setDeliveryKind(userOrder.getDeliveryKind());
		order.setDeliveryDate(userOrder.getDeliveryDate());
		order.setDeliveryTime(userOrder.getDeliveryTime());
		order.setPaymentKind(userOrder.getPaymentKind());
		
		if (VC.PAYMENT_KIND_CREDITCARD.equals(userOrder.getPaymentKind())) {
			order.setCardNo(userCreditCard.getCardNo());
			order.setCardType(userCreditCard.getCardType());
			order.setCardName(userCreditCard.getCardName());
			order.setCardExpires(userCreditCard.getCardExpires());
		}
		
		order.setOrderDate(Calendar.getInstance().getTime());
		order.setMemo(userOrder.getMemo());

		order.setInvalid(false);
		order.setCusid(user.getId());
		order.setCtime(Calendar.getInstance().getTime());
		order.setUusid(user.getId());
		order.setUtime(Calendar.getInstance().getTime());
		dao.insert(order);

		order.setItems(new ArrayList<UserOrderItem>()); 
		for (CartItem ci : cart.getItems()) {
			UserOrderItem coi = new UserOrderItem();
			coi.setId(null);
			coi.setUserId(order.getUserId());
			coi.setUserOrderId(order.getId());
			coi.setGoods(ci.getGoods());
			coi.setGoodsId(ci.getGoods().getId());
			coi.setQuantity(ci.getQuantity());
			coi.setSalePrice(ci.getGoods().getSalePrice());
			coi.setShipCost(ci.getGoods().getShipCost());
			coi.setStatus(VC.ORDER_STATUS_BEGIN);
			coi.setInvalid(false);
			coi.setCusid(user.getId());
			coi.setCtime(Calendar.getInstance().getTime());
			coi.setUusid(user.getId());
			coi.setUtime(Calendar.getInstance().getTime());
			idao.insert(coi);

			utils().daoUpdateGoodsOrderQuantity(getDataAccessSession(), coi.getGoodsId(), coi.getQuantity());
			
			order.getItems().add(coi);
		}

		Map<String, Object> context = new HashMap<String, Object>();
		context.put("action", this);
		context.put("user", user);
		context.put("order", order);
		
		try {
			utils().sendTemplateMail(user, TPL.MAIL_ORDER, context);
		}
		catch (EmailException e) {
			String msg = getText(RC.ERROR_SENDMAIL, new String[] { user.getEmail() });
			if (utils().ignoreEmailException()) {
				addActionWarning(msg);
			}
			else {
				addActionError(msg);
				getDataAccessSession().rollback();
				return INPUT;
			}
		}

		getDataAccessSession().commit();

		addActionMessage(getText(RC.MESSAGE_SUCCESS, new String[] { order.getCode() }));

		cart.getItems().removeAll(orderCartItems);
		utils().saveCartToCookie(cart);
		
		userOrder = order;
		
		step = STEP_SUCCESS;
		return SUCCESS;
	}
}
