/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.util;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.json.JSONArray;

import nuts.aems.model.bean.User;
import nuts.aems.util.BaseActionUtils;
import nuts.core.lang.NumberUtils;
import nuts.core.orm.dao.DataAccessException;
import nuts.core.orm.dao.DataAccessSession;
import nuts.exts.struts2.actions.CommonAction;
import nuts.exts.struts2.util.StrutsContextUtils;
import nuts.exts.xwork2.util.LocalizedTextUtils;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.SimpleEmail;
import org.apache.commons.vfs2.FileObject;

import squirrels.ecshop.Application;
import squirrels.ecshop.bean.Cart;
import squirrels.ecshop.bean.CartCookieItem;
import squirrels.ecshop.bean.CartItem;
import squirrels.ecshop.constant.AC;
import squirrels.ecshop.constant.REQ;
import squirrels.ecshop.model.bean.Category;
import squirrels.ecshop.model.bean.Goods;
import squirrels.ecshop.model.dao.GoodsDAO;
import squirrels.ecshop.model.example.GoodsExample;
import freemarker.template.TemplateException;


/**
 * ActionUtils.
 */
public class ActionUtils extends BaseActionUtils {

	/**
	 * Constructor
	 */
	public ActionUtils(CommonAction action) {
		super(action);
	}

	/**
	 * @param sid super id
	 * @param cid category id
	 * @return true if sid is super of cid
	 */
	public boolean isSuperCategory(String sid, String cid) {
		try {
			Long lsid = Long.parseLong(sid);
			Long lcid = Long.parseLong(cid);
			Category c = getCategory(lcid);
			while (c != null) {
				if (c.getId().equals(lsid)) {
					return true;
				}
				c = c.getParent();
			}
		}
		catch (Exception e) {
			
		}
		return false;
	}
	
	/**
	 * @return category map
	 */
	@SuppressWarnings("unchecked")
	public Map<Long, Category> getCategoryMap() {
		return (Map<Long, Category>)getApplication().get(AC.CATEGORY_MAP);
	}

	/**
	 * @param c category
	 * @return category instance
	 */
	public Category getRootCategory(Category c) {
		if (c != null) {
			while (c.getParent() != null) {
				c = c.getParent();
			}
		}
		return c;
	}

	/**
	 * @param cid category Id
	 * @return category instance
	 */
	public Category getRootCategory(Long cid) {
		Category c = getCategory(cid);
		return getRootCategory(c);
	}

	/**
	 * @param cid category Id
	 * @return category instance
	 */
	public Category getCategory(Long cid) {
		return getCategoryMap().get(cid);
	}

	/**
	 * @param cid category id
	 * @return category image dir
	 * @throws IOException if an IO error occurs
	 */
	public FileObject getCategoryImagesPath(Long cid) throws IOException {
		return Application.getCategoryImagesDir().resolveFile(cid.toString());
	}

	/**
	 * @param c category
	 * @return category image dir
	 * @throws IOException if an IO error occurs
	 */
	public FileObject getCategoryImagesPath(Category c) throws IOException {
		return getCategoryImagesPath(c.getId());
	}
	
	/**
	 * @return category list (top)
	 * @throws Exception if an error occurs
	 */
	@SuppressWarnings("unchecked")
	public List<Category> getCategoryTopList() throws Exception {
		return (List<Category>)getApplication().get(AC.CATEGORY_TOP_LIST);
	}

	private void addCategoryChilds(List<Long> ids, Category c) {
		ids.add(c.getId());
		if (c.getChilds() != null) {
			for (Category cc : c.getChilds()) {
				addCategoryChilds(ids, cc);
			}
		}
	}

	public List<Long> getChildCategoryList(Category c) {
		List<Long> ids = new ArrayList<Long>();
		addCategoryChilds(ids, c);
		return ids;
	}
	
	/**
	 * @param gid goods id
	 * @return goods image dir
	 * @throws IOException if an IO error occurs
	 */
	public FileObject getGoodsImagesPath(Long gid) throws IOException {
		return Application.getGoodsImagesDir().resolveFile(gid.toString());
	}

	/**
	 * @param g goods
	 * @return goods image dir
	 * @throws IOException if an IO error occurs
	 */
	public FileObject getGoodsImagesPath(Goods g) throws IOException {
		return getGoodsImagesPath(g.getId());
	}

	/**
	 * @return Cart
	 * @throws Exception if an error occurs
	 */
	public Cart getUserCart(DataAccessSession das) throws Exception {
		Cart cart = (Cart)getSession().get(REQ.CART);
		if (cart == null) {
			cart = loadCartFromCookie(das);
			getSession().put(REQ.CART, cart);
		}
		return cart;
	}
	
	/**
	 * load cart from cookie
	 * @return Cart
	 * @throws Exception if an error occurs
	 */
	@SuppressWarnings("unchecked")
	public Cart loadCartFromCookie(DataAccessSession das) throws Exception {
		Cart cart = new Cart();

		HttpServletRequest req = StrutsContextUtils.getServletRequest();
		Cookie[] cookies = req.getCookies();
		if (cookies == null) {
			return cart;
		}
		
		Collection<CartCookieItem> cookieCarts = null;
		for (Cookie c : cookies) {
			if ("cart".equals(c.getName())) {
				String cv = c.getValue();
				if (StringUtils.isNotEmpty(cv)) {
					try {
						JSONArray ja = JSONArray.fromObject(cv);
						cookieCarts = (Collection<CartCookieItem>)JSONArray.toCollection(ja, CartCookieItem.class);
					}
					catch (Exception e) {
						log.warn("Failed to restore Cart from Cookie", e);
						return cart;
					}
				}
				break;
			}
		}

		if (cookieCarts != null && !cookieCarts.isEmpty()) {
			List<Long> pids = new ArrayList<Long>();
			for (CartCookieItem cc : cookieCarts) {
				if (StringUtils.isNotEmpty(cc.getG()) && NumberUtils.isLong(cc.getG())) {
					pids.add(NumberUtils.toLong(cc.getG()));
				}
			}

			GoodsDAO dao = new GoodsDAO(das);
			GoodsExample example = dao.createExample();
			example.invalid().isFalse().and().id().in(pids);
			
			List<Goods> gs = dao.selectByExample(example);

			for (Goods g : gs) {
				for (CartCookieItem cc : cookieCarts) {
					if (cc.getG().equals(g.getId().toString())) {
						CartItem ci = new CartItem(cc);
						ci.setGoods(g);
						cart.getItems().add(ci);
						break;
					}
				}
			}
		}
		
		return cart;
	}
	
	/**
	 * Save cart to cookie 
	 * @param cart cart 
	 */
	public void saveCartToCookie(Cart cart) {
		HttpServletRequest req = StrutsContextUtils.getServletRequest();
		HttpServletResponse res = StrutsContextUtils.getServletResponse();

		Cookie cookie = new Cookie("cart", "");
		
		cookie.setPath(req.getContextPath());
		cookie.setMaxAge(LocalizedTextUtils.findDefaultTextAsInt("cart-cookie-expires", getLocale(), -1));

		if (!cart.isEmpty()) {
			JSONArray ja = new JSONArray();
			for (CartItem ci : cart.getItems()) {
				CartCookieItem cci = new CartCookieItem(ci);
				ja.add(cci);
			}
			cookie.setValue(ja.toString());
		}
		
		res.addCookie(cookie);
	}

	/**
	 * create a illegal user id exception
	 * @param id user id
	 * @return exception
	 */
	public Exception IllegalUserIdException(Long id) {
		return new IllegalArgumentException("Incorrect User ID [" + id + "].");
	}
	
	private final SimpleDateFormat COC_DF = new SimpleDateFormat("yyyyMMdd-HHmmss-SSS");

	/**
	 * generate user order unique code
	 * @return code
	 */
	public String genUserOrderCode() {
		Date d = Calendar.getInstance().getTime();
		return COC_DF.format(d) + "-" + d.getTime(); 
	}

	/**
	 * send email
	 * @param user user 
	 * @param name template name
	 * @param context context object
	 * @throws IOException io error 
	 * @throws TemplateException  template error
	 * @throws EmailException  email error
	 */
	public void sendTemplateMail(User user, String name,
			Object context) throws TemplateException, IOException, EmailException {
		SimpleEmail email = new SimpleEmail();

		email.addTo(user.getEmail(), user.getName());
		
		sendTemplateMail(email, name, context);
	}

	/**
	 * send email
	 * @param user user
	 * @param name template name
	 * @throws IOException io error 
	 * @throws TemplateException  template error
	 * @throws EmailException  email error
	 */
	public void sendTemplateMail(User user, String name)
			throws TemplateException, IOException, EmailException {
		SimpleEmail email = new SimpleEmail();

		email.addTo(user.getEmail(), user.getName());
		
		sendTemplateMail(email, name);
	}
	
	/**
	 * calcUserOrderPrice
	 * 
	 * @param userOrderId userOrderId
	 * @return update count
	 * @throws DataAccessException if a data access error occurs
	 */
	public int daoCalcUserOrderPrice(DataAccessSession das, Long userOrderId) throws DataAccessException {
		//TODO
		//return getDataAccessClient().update(ADMIN_CALC_USER_ORDER_PRICE, userOrderId);
		return 0;
	}

	/**
	 * update goods order quantity
	 * 
	 * @param goodsId goodsId
	 * @param orderQuantity orderQuantity
	 * @return update count
	 * @throws DataAccessException if a data access error occurs
	 */
	public int daoUpdateGoodsOrderQuantity(DataAccessSession das, Long goodsId, Integer orderQuantity) throws DataAccessException {
		return daoUpdateStockQuantity(das, goodsId, orderQuantity, null, null);
	}

	/**
	 * update goods supply quantity
	 * 
	 * @param goodsId goodsId
	 * @param supplyQuantity supplyQuantity
	 * @return update count
	 * @throws DataAccessException if a data access error occurs
	 */
	public int daoUpdateGoodsSupplyQuantity(DataAccessSession das, Long goodsId, Integer supplyQuantity) throws DataAccessException {
		return daoUpdateStockQuantity(das, goodsId, null, null, supplyQuantity);
	}

	private static int toInt(Integer v) {
		return v == null ? 0 : v;
	}
	
	/**
	 * update stock quantity
	 * 
	 * @param goodsId goodsId
	 * @param orderQuantity orderQuantity
	 * @param soldQuantity soldQuantity
	 * @param supplyQuantity supplyQuantity
	 * @return update count
	 * @throws DataAccessException if a data access error occurs
	 */
	public int daoUpdateStockQuantity(DataAccessSession das, Long goodsId, Integer orderQuantity,
			Integer soldQuantity, Integer supplyQuantity) throws DataAccessException {
		GoodsDAO gDao = new GoodsDAO(das);
		
		Goods g = gDao.selectByPrimaryKey(goodsId);
		if (g != null) {
			g.setOrderQuantity(g.getOrderQuantity() + toInt(orderQuantity));
			g.setSoldQuantity(g.getSoldQuantity() + toInt(soldQuantity));
			g.setSupplyQuantity(g.getSupplyQuantity() + toInt(supplyQuantity));
			g.setStockQuantity(g.getSupplyQuantity() - g.getSoldQuantity() - g.getOrderQuantity());
			
			return gDao.updateByPrimaryKey(g);
		}
		else {
			return 0;
		}
	}
}
