/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import nuts.core.lang.NumberUtils;
import nuts.core.util.Pager;
import nuts.core.util.SimpleQuery;
import nuts.core.util.Sorter;
import nuts.exts.lucene.LuceneUtils;

import org.apache.commons.lang.StringUtils;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.queryParser.QueryParser;
import org.apache.lucene.search.FieldCacheTermsFilter;
import org.apache.lucene.search.Filter;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.Searcher;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopDocsCollector;
import org.apache.lucene.search.TopFieldCollector;
import org.apache.lucene.search.TopScoreDocCollector;
import org.apache.lucene.util.Version;

import squirrels.ecshop.Application;
import squirrels.ecshop.constant.RC;
import squirrels.ecshop.model.bean.Category;
import squirrels.ecshop.model.bean.Goods;
import squirrels.ecshop.model.dao.GoodsDAO;
import squirrels.ecshop.model.example.GoodsExample;
import squirrels.ecshop.model.metadata.GoodsMetaData;
import freemarker.template.Configuration;
import freemarker.template.Template;

/**
 * Goods Action
 */
@SuppressWarnings("serial")
public class GoodsAction extends AbstractAction {

	protected Pager pager = new Pager();

	protected Sorter sorter = new Sorter();

	protected SimpleQuery query = new SimpleQuery();
	
	protected String categoryId;

	protected Category category;

	protected List<Category> categoryLocation;
	
	protected String goodsId;
	
	protected Goods goods;

	protected String[] goodsThumbs;

	protected List<Goods> goodsList;

	/**
	 * @return the categoryId
	 */
	public String getCategoryId() {
		return categoryId;
	}

	/**
	 * @param categoryId the categoryId to set
	 */
	public void setCategoryId(String categoryId) {
		this.categoryId = categoryId;
	}

	/**
	 * @return the category
	 */
	public Category getCategory() {
		return category;
	}

	/**
	 * @return the categoryLocation
	 */
	public List<Category> getCategoryLocation() {
		return categoryLocation;
	}

	/**
	 * @return the goodsId
	 */
	public String getGoodsId() {
		return goodsId;
	}

	/**
	 * @param goodsId the goodsId to set
	 */
	public void setGoodsId(String goodsId) {
		this.goodsId = goodsId;
	}

	/**
	 * @return the goods
	 */
	public Goods getGoods() {
		return goods;
	}

	/**
	 * @return the goodsThumbs
	 */
	public String[] getGoodsThumbs() {
		return goodsThumbs;
	}

	/**
	 * @return the goodsList
	 */
	public List<Goods> getGoodsList() {
		return goodsList;
	}

	//-----------------------------------------------------------------------
	// short name
	//-----------------------------------------------------------------------
	/**
	 * @return the pager
	 */
	public Pager getPg() {
		return pager;
	}

	/**
	 * @param pager the pager to set
	 */
	public void setPg(Pager pager) {
		this.pager = pager;
	}

	/**
	 * @return the sorter
	 */
	public Sorter getSo() {
		return sorter;
	}

	/**
	 * @param sorter the sorter to set
	 */
	public void setSo(Sorter sorter) {
		this.sorter = sorter;
	}

	/**
	 * @return the query
	 */
	public SimpleQuery getQq() {
		return query;
	}

	/**
	 * @param query the query to set
	 */
	public void setQq(SimpleQuery query) {
		this.query = query;
	}

	/**
	 * @return the goodsId
	 */
	public String getGid() {
		return goodsId;
	}

	/**
	 * @param goodsId the goodsId to set
	 */
	public void setGid(String goodsId) {
		this.goodsId = goodsId;
	}

	/**
	 * @return the categoryId
	 */
	public String getCid() {
		return categoryId;
	}

	/**
	 * @param categoryId the categoryId to set
	 */
	public void setCid(String categoryId) {
		this.categoryId = categoryId;
	}

	/**
	 * list
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String list() throws Exception {
		query.setKey(null);

		utils().loadLimitParams(pager);
		utils().loadSorterParams(sorter);

		GoodsExample example = new GoodsExample();
		
		example.invalid().isFalse();
		
		Long cid = NumberUtils.toLong(categoryId);
		if (cid != null) {
			processLocation(cid);
			if (category != null) {
				List<Long> cids = utils().getChildCategoryList(category);
				example.and().categoryId().in(cids);
			}
		}

		example.getOrders().addOrder(resolveColumnAlias(sorter.getColumn()), sorter.getDirection());

		GoodsDAO dao = new GoodsDAO(getDataAccessSession());
		pager.setTotal(dao.countByExample(example));
		pager.normalize();
		if (pager.getTotal() > 0) {
			example.setStart(pager.getStart());
			example.setLimit(pager.getLimit());
			goodsList = dao.selectByExample(example);
		}
		utils().saveLimitParams(pager);
		utils().saveSorterParams(sorter);

		return LIST;
	}

	/**
	 * search
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String search() throws Exception {
		if (StringUtils.isEmpty(query.getKey())) {
			return list();
		}

		utils().loadLimitParams(pager);
		utils().loadSorterParams(sorter);

		Searcher searcher = new IndexSearcher(Application.getLuceneDirectory()); 
        Analyzer analyzer = new StandardAnalyzer(Version.LUCENE_30);
        QueryParser parser = new QueryParser(Version.LUCENE_30, "name", analyzer);

        Query q = null;
		try {
			q = LuceneUtils.parseSimpleQuery(parser, query.getKey());
		}
		catch (ParseException e) {
			log.warn("query", e);
			pager.setTotal(0);
			return LIST;
		}

		TopDocsCollector collector;
		if (StringUtils.isNotEmpty(sorter.getColumn())) {
			Sort sort = new Sort(new SortField(sorter.getColumn(),
					"salePrice".equals(sorter.getColumn()) ? SortField.FLOAT
							: SortField.STRING_VAL,
					Sorter.DESC.equals(sorter.getDirection())));
			collector = TopFieldCollector.create(sort, pager.getStart() + pager.getLimit(),
			        true, false, false, false);
		}
		else {
			collector = TopScoreDocCollector.create(pager.getStart() + pager.getLimit(), false);
		}
		
		Filter filter = null;
		if (StringUtils.isNotEmpty(query.getTarget()) && !"0".equals(query.getTarget())) {
			filter = new FieldCacheTermsFilter("category", query.getTarget());
		}
		searcher.search(q, filter, collector);
		
		pager.setTotal(collector.getTotalHits());
		pager.normalize();
		if (pager.getTotal() > 0) {
			TopDocs hits = collector.topDocs(pager.getStart(), pager.getLimit());
			goodsList = new ArrayList<Goods>(hits.scoreDocs.length);
			GoodsDAO gdao = new GoodsDAO(getDataAccessSession());

			for (int i = 0; i < hits.scoreDocs.length; i++) {
				Document doc = searcher.doc(hits.scoreDocs[i].doc);
				Goods g = gdao.selectByPrimaryKey(Long.parseLong(doc.get("id")));
				if (g != null) {
					goodsList.add(g);
				}
			}
		}
		
		utils().saveLimitParams(pager);
		utils().saveSorterParams(sorter);

		return LIST;
	}

	/**
	 * get column alias
	 * @param name name
	 * @return column alias
	 */
	private String resolveColumnAlias(String name) {
		return GoodsMetaData.getInstance().getColumnAlias(name);
	}

	/**
	 * view
	 * @return result name
	 * @throws Exception if an error occurs
	 */
	public String view() throws Exception {
		if (StringUtils.isNotEmpty(goodsId) && NumberUtils.isLong(goodsId)) {
			GoodsDAO dao = new GoodsDAO(getDataAccessSession());
			
			goods = dao.selectByPrimaryKey(NumberUtils.toLong(goodsId));
			if (goods != null) {
				categoryId = goods.getCategoryId().toString();
				processLocation(goods.getCategoryId());
				if (StringUtils.isNotEmpty(goods.getContent())) {
					Template tpl = new Template("content", 
						new StringReader(goods.getContent()), 
						new Configuration());
					StringWriter sw = new StringWriter();
					tpl.process(goods, sw);
					
					goods.setContent(sw.toString());
				}

				goodsThumbs = StringUtils.split(goods.getThumbs());
			}
			else {
				addActionError(getText(RC.ERROR_DATA_NOTFOUND));
			}
		}
		return VIEW;
	}

	/**
	 * process location
	 * @param categoryId catgoryId
	 */
	protected void processLocation(Long categoryId) throws Exception {
		Category c = utils().getCategory(categoryId);
		if (c != null) {
			category = c;
			categoryLocation = new ArrayList<Category>();
			do {
				categoryLocation.add(0, c);
				c = c.getParent();
			}
			while (c != null);
		}
	}

}