/*
 * This file is part of Nuts.
 * Copyright (C) 2009 Nuts Develop Team.
 *
 * Nuts is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Nuts is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Nuts. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action.admin;

import java.io.BufferedReader;
import java.io.FileReader;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Random;

import javax.servlet.ServletContext;

import nuts.aems.model.bean.User;
import nuts.core.io.IOUtils;
import nuts.core.lang.StringUtils;
import nuts.core.orm.dao.DataAccessSession;
import nuts.core.orm.dao.DataAccessUtils;
import nuts.core.orm.dao.DataHandler;
import nuts.exts.struts2.actions.CommonWorkAction;
import nuts.exts.struts2.util.StrutsContextUtils;

import org.apache.commons.collections.buffer.CircularFifoBuffer;
import org.apache.lucene.analysis.standard.StandardAnalyzer;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.LogByteSizeMergePolicy;
import org.apache.lucene.index.MergePolicy;
import org.apache.lucene.index.SerialMergeScheduler;
import org.apache.lucene.util.Version;

import squirrels.ecshop.Application;
import squirrels.ecshop.model.bean.Goods;
import squirrels.ecshop.model.dao.GoodsDAO;
import squirrels.ecshop.model.example.GoodsExample;
import squirrels.ecshop.util.ActionUtils;
import squirrels.ecshop.util.LuceneUtils;



/**
 */
@SuppressWarnings("serial")
public class GoodsIndexAction extends CommonWorkAction {

	private static String time;
	private static boolean stop = false;
	private static boolean progress = false;
	private static int count;
	private static String line;
	
	private int start;
	private int limit;
	private int words;
	
	public GoodsIndexAction() {
	}

	/**
	 * @return the time
	 */
	public String getTime() {
		return time;
	}

	/**
	 * @return the stop
	 */
	public boolean isStop() {
		return stop;
	}

	/**
	 * @return the count
	 */
	public int getCount() {
		return count;
	}

	/**
	 * @return the line
	 */
	public String getLine() {
		return line;
	}

	/**
	 * @return the progress
	 */
	public boolean isProgress() {
		return progress;
	}

	/**
	 * @return the start
	 */
	public int getStart() {
		return start;
	}

	/**
	 * @param start the start to set
	 */
	public void setStart(int start) {
		this.start = start;
	}

	/**
	 * @return the limit
	 */
	public int getLimit() {
		return limit;
	}

	/**
	 * @param limit the limit to set
	 */
	public void setLimit(int limit) {
		this.limit = limit;
	}

	/**
	 * @return the words
	 */
	public int getWords() {
		return words;
	}

	/**
	 * @param words the words to set
	 */
	public void setWords(int words) {
		this.words = words;
	}

	/**
	 * @return the utils
	 */
	public ActionUtils utils() {
		return (ActionUtils)super.getUtils();
	}

	public String stop() {
		stop = true;
		time = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").format(Calendar.getInstance().getTime());
		return "xml";
	}
	
	public String peek() {
		time = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS").format(Calendar.getInstance().getTime());
		return "xml";
	}
	
	protected void init() throws Exception {
		super.init();
		stop = false;
	}
	
	public String index() {
		try {
			init();
			if (progress) {
				doProgress();
			}
			else {
				doIndex();
			}
		}
		catch (Throwable e) {
			log.warn("index", e);
			try {
				printError(e.getMessage());
			}
			catch (Exception ex) {
			}
		}
		return NONE;
	}
	
	public String trunc() {
		try {
			init();
			if (progress) {
				doProgress();
			}
			else {
				doTrunc();
			}
		}
		catch (Throwable e) {
			log.warn("trunc", e);
			try {
				printError(e.getMessage());
			}
			catch (Exception ex) {
			}
		}
		return NONE;
	}
	
	public String impex() {
		try {
			init();
			if (progress) {
				doProgress();
			}
			else {
				doImpex();
			}
		}
		catch (Throwable e) {
			log.warn("impex", e);
			try {
				printError(e.getMessage());
			}
			catch (Exception ex) {
			}
		}
		return NONE;
	}

	private void doProgress() throws Exception {
		int cnt = 0;
		while (progress) {
			while (progress && cnt == count) {
				Thread.sleep(500);
			}
			cnt = count;
			printProcess(count, line);
		}
		printSuccess(String.valueOf(count));
	}

	private Date getRandomDate(Random random) {
		try {
			Calendar c = Calendar.getInstance();

			c.setTimeInMillis(random.nextLong());
			c.set(Calendar.YEAR, 2011);

			return c.getTime();
		}
		catch (Exception e) {
			return Calendar.getInstance().getTime();
		}
	}
	
	private CircularFifoBuffer wordList;
	
	private String makeWords(String word) {
		if (words < 2) {
			return word;
		}
		if (wordList == null) {
			wordList = new CircularFifoBuffer(words);
		}

		wordList.add(word);
		
		return StringUtils.join(wordList, " ");
	}
	
	private void doImpex() throws Exception {
		User user = utils().getLoginUser();
		BufferedReader br = null;
		try {
			progress = true;
			count = 0;

			Random random = new Random();
			
			ServletContext sc = StrutsContextUtils.getServletContext();
			br = new BufferedReader(new FileReader(sc.getRealPath("Unabr.dict")));
			
			GoodsDAO s1dao = new GoodsDAO(getDataAccessSession());
			
			count = 0;
			while ((line = br.readLine()) != null) {
				if (stop) {
					break;
				}
				if (count - start > limit && limit > 0) {
					break;
				}
				if (count < start) {
					count++;
					continue;
				}

				Goods s1 = new Goods();
				s1.setName(makeWords(line));
				if (words > 1 && wordList.size() < words) {
					count++;
					continue;
				}

				s1.setCategoryId(101101L);
				s1.setCode(StringUtils.substring(String.valueOf(random.nextDouble()), 0, 10));
				s1.setListPrice(BigDecimal.valueOf(random.nextDouble()));
				s1.setSalePrice(BigDecimal.valueOf(random.nextDouble()));
				s1.setDiscount((int)(s1.getSalePrice().floatValue() / s1.getListPrice().floatValue() * 100));
				s1.setShipCost(BigDecimal.valueOf(random.nextDouble()));
				s1.setPublishDate(getRandomDate(random));
				s1.setInvalid(false);
				s1.setCusid(user.getId());
				s1.setCtime(Calendar.getInstance().getTime());
				s1.setUusid(s1.getCusid());
				s1.setUtime(s1.getCtime());
				s1dao.insert(s1);
				
				String msg = s1.getId() + " - " + s1.getName();
				if (++count % 100 == 0) {
					printProcess(count, msg);
				}
				
				if (count % 100 == 0) {
					getDataAccessSession().commit();
				}
			}
			getDataAccessSession().commit();
			printSuccess(String.valueOf(count));
		}
		finally {
			IOUtils.closeQuietly(br);
			progress = false;
		}
	}

	private void doIndex() throws Exception {
		GoodsDAO adao = new GoodsDAO(getDataAccessSession());
		GoodsExample aexp = adao.createExample();
		aexp.invalid().isFalse();
		if (start > 0) {
			aexp.setStart(start);
		}
		if (limit > 0) {
			aexp.setLimit(limit);
		}

		final IndexWriter writer = new IndexWriter(Application.getLuceneDirectory(),
				new StandardAnalyzer(Version.LUCENE_30), true,
				IndexWriter.MaxFieldLength.LIMITED);

		writer.setMergeScheduler(new SerialMergeScheduler());
		MergePolicy mp = writer.getMergePolicy();
		if (mp instanceof LogByteSizeMergePolicy) {
			((LogByteSizeMergePolicy)mp).setMaxMergeMB(0.9);
		}
		else {
			mp = new LogByteSizeMergePolicy(writer);
			((LogByteSizeMergePolicy)mp).setMaxMergeMB(0.9);
			writer.setMergePolicy(mp);
		}

		try {
			progress = true;
			count = 0;

			int count = adao.selectByExampleWithDataHandler(aexp, new DataHandler<Goods>() {
				int i = 0;
				public boolean handleData(Goods g) throws Exception {
					String msg = g.getId() + " - " + g.getName();
					writer.addDocument(LuceneUtils.document(g));
					if (++i % 100 == 0) {
						printProcess(i, msg);
					}
					if (i % 10000 == 0) {
						writer.optimize();
						writer.commit();
					}
					return true;
				}
			});
			
			writer.optimize();
			writer.commit();

			printSuccess(String.valueOf(count));
		}
		finally {
			progress = false;
			writer.close();
		}
	}

	private void doTrunc() throws Exception {
		final DataAccessSession das = Application.get().openDataAccessSession();
		try {
			progress = true;
			count = 0;

			final GoodsDAO sdao = new GoodsDAO(getDataAccessSession());
			final GoodsDAO ddao = new GoodsDAO(das);
			
			GoodsExample sexp = sdao.createExample(); 
			if (start > 0) {
				sexp.setStart(start);
			}
			if (limit > 0) {
				sexp.setLimit(limit);
			}

			sdao.selectByExampleWithDataHandler(sexp, new DataHandler<Goods>() {
				public boolean handleData(Goods s1) throws Exception {
					if (stop) {
						return false;
					}
					
					ddao.deleteByPrimaryKey(s1);

					String msg = s1.getId() + " - " + s1.getName();
					if (++count % 100 == 0) {
						printProcess(count, msg);
						das.commit();
					}
					
					return true;
				}
			});

			das.commit();

			printSuccess(String.valueOf(count));
		}
		finally {
			progress = false;
			DataAccessUtils.closeQuietly(das);
		}
	}
}
