/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 *
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action.admin;

import squirrels.ecshop.action.admin.goodsreview.GoodsReviewAction;
import squirrels.ecshop.constant.VC;
import squirrels.ecshop.model.bean.GoodsReview;
import squirrels.ecshop.model.bean.UserOrderItem;
import squirrels.ecshop.model.dao.UserOrderItemDAO;
import squirrels.ecshop.model.example.UserOrderItemExample;




/**
 * GoodsReviewExAction
 */
@SuppressWarnings("serial")
public class GoodsReviewExAction extends GoodsReviewAction {

	/**
	 * Constructor
	 */
	public GoodsReviewExAction() {
		super();
	}

	/**
	 * prepareDefaultData
	 * @param data data
	 * @return data
	 */
	@Override
	protected GoodsReview prepareDefaultData(GoodsReview data) throws Exception {
		if (data == null) {
			data = new GoodsReview();
		}
		if (data.getRank() == null) {
			data.setRank(VC.GOODS_REVIEW_RANK_DEFAULT);
		}
		if (data.getVoteOk() == null) {
			data.setVoteOk(0);
		}
		if (data.getVoteNo() == null) {
			data.setVoteNo(0);
		}
		return data;
	}

	/**
	 */
	@Override
	protected boolean checkOnInsert(GoodsReview data) throws Exception {
		boolean c = super.checkOnInsert(data);
		if (!c && isCheckAbortOnError()) {
			return false;
		}
		
		if (userOrderItemId == null) {
			addFieldError("userOrderItemId", getText("validation-required"));
		}
		else {
			UserOrderItemDAO dao = new UserOrderItemDAO(getDataAccessSession());
			UserOrderItem coi = dao.selectByPrimaryKey(userOrderItemId);
			
			if (coi == null) {
				addFieldError("userOrderItemId", getText("error-coi-notexists"));
				return false;
			}
			if (!VC.ORDER_STATUS_FINISHED.equals(coi.getStatus())) {
				addFieldError("userOrderItemId", getText("error-coi-unfinished"));
				return false;
			}
			if (coi.getReviewId() != null) {
				addFieldError("userOrderItemId", getText("error-coi-reviewed"));
				return false;
			}
		}
		return c;
	}

	/**
	 */
	@Override
	protected boolean checkOnUpdate(GoodsReview data, GoodsReview srcData) throws Exception {
		boolean c = super.checkOnUpdate(data, srcData);
		if (!c && isCheckAbortOnError()) {
			return false;
		}
		if (userOrderItemId != null) {
			UserOrderItemDAO dao = new UserOrderItemDAO(getDataAccessSession());
			UserOrderItem coi = dao.selectByPrimaryKey(userOrderItemId);
			
			if (coi == null) {
				addFieldError("userOrderItemId", getText("error-coi-notexists"));
				return false;
			}
			if (!VC.ORDER_STATUS_FINISHED.equals(coi.getStatus())) {
				addFieldError("userOrderItemId", getText("error-coi-unfinished"));
				return false;
			}
		}
		return c;
	}

	/**
	 * commitInsert
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void commitInsert(GoodsReview data) throws Exception {
		updateCOIReviewId(data);
		super.commitInsert(data);
	}

	/**
	 * commitUpdate
	 * @param data data
	 * @param srcData source Data
	 */
	@Override
	protected void commitUpdate(GoodsReview data, GoodsReview srcData) throws Exception {
		if (userOrderItemId != null) {
			updateCOIReviewId(data);
		}
		super.commitUpdate(data, srcData);
	}

	private void updateCOIReviewId(GoodsReview data) throws Exception {
		UserOrderItem coi = new UserOrderItem();
		coi.setId(userOrderItemId);
		coi.setReviewId(data.getId());

		UserOrderItemDAO dao = new UserOrderItemDAO(getDataAccessSession());
		int i = dao.updateByPrimaryKeySelective(coi);
		if (i != 1) {
			throw new RuntimeException("Failed to update USER_ORDER_ITEM.REVIEW_ID. {id: " 
				+ coi.getId() + ", reviewId: " + coi.getReviewId() + "} COUNT: " + i);
		}
	}

	/**
	 * startDelete
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startDelete(GoodsReview data) throws Exception {
		super.startDelete(data);

		deleteCOIReviewId(data);
	}

	private void deleteCOIReviewId(GoodsReview data) throws Exception {
		UserOrderItem coi = new UserOrderItem();

		UserOrderItemDAO idao = new UserOrderItemDAO(getDataAccessSession());
		UserOrderItemExample iexp = idao.createExample();
		iexp.excludeALL().userOrderId().include();
		
		idao.updateByExample(coi, iexp);
	}

}
