/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.action.admin;

import java.text.MessageFormat;
import java.util.List;

import squirrels.ecshop.action.admin.supplement.SupplementAction;
import squirrels.ecshop.model.bean.Supplement;



/**
 * SupplementExAction
 */
@SuppressWarnings("serial")
public class SupplementExAction extends SupplementAction {

	private static final String ERROR_UPDATE_STOCK_SUPPLY_QUANTITY = "Failed to update stock's supply quantity ({0}). Update count: {1}.";

	/**
	 * Constructor
	 */
	public SupplementExAction() {
		super();
	}

	/**
	 * startDelete
	 * @param data data
	 * @throws Exception if an error occurs
	 */
	@Override
	protected void startDelete(Supplement data) throws Exception {
		super.startDelete(data);
		updateGoodsSupply(data);
	}
	
	private void updateGoodsSupply(Supplement data) throws Exception {
		if (!data.beInvalid()) {
			utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), data.getGoodsId(), -data.getSupplyQuantity());
		}
	}

	/**
	 * startBulkDelete
	 * @param dataList data list
	 * @throws Exception if an error occurs
	 */
	protected void startBulkDelete(List<Supplement> dataList) throws Exception {
		super.startBulkDelete(dataList);

		for (Supplement s : dataList) {
			updateGoodsSupply(s);
		}
	}
	
	/**
	 * commitUpdate
	 * @param data data
	 * @param srcData source Data
	 */
	@Override
	protected void commitUpdate(Supplement data, Supplement srcData) throws Exception {
		if (data.beInvalid() && !srcData.beInvalid()) {
			int cnt = utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), srcData.getGoodsId(), -srcData.getSupplyQuantity());
			if (cnt != 1) {
				throw new RuntimeException(MessageFormat.format(
					ERROR_UPDATE_STOCK_SUPPLY_QUANTITY, -srcData.getSupplyQuantity(), cnt));
			}
		}
		else if (!data.beInvalid() && srcData.beInvalid()) {
			int cnt = utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), data.getGoodsId(), data.getSupplyQuantity());
			if (cnt != 1) {
				throw new RuntimeException(MessageFormat.format(
					ERROR_UPDATE_STOCK_SUPPLY_QUANTITY, data.getSupplyQuantity(), cnt));
			}
		}
		else if (!data.beInvalid() && !srcData.beInvalid()) {
			if (!srcData.getGoodsId().equals(data.getGoodsId())) {
				int cnt = utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), srcData.getGoodsId(), -srcData.getSupplyQuantity());
				if (cnt != 1) {
					throw new RuntimeException(MessageFormat.format(
						ERROR_UPDATE_STOCK_SUPPLY_QUANTITY, -srcData.getSupplyQuantity(), cnt));
				}
				cnt = utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), data.getGoodsId(), data.getSupplyQuantity());
				if (cnt != 1) {
					throw new RuntimeException(MessageFormat.format(
						ERROR_UPDATE_STOCK_SUPPLY_QUANTITY, data.getSupplyQuantity(), cnt));
				}
			}
			else if (!srcData.getSupplyQuantity().equals(data.getSupplyQuantity())) {
				int cnt = utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), data.getGoodsId(), 
						data.getSupplyQuantity() - srcData.getSupplyQuantity());
				if (cnt != 1) {
					throw new RuntimeException(MessageFormat.format(
						ERROR_UPDATE_STOCK_SUPPLY_QUANTITY, data.getSupplyQuantity()
								- srcData.getSupplyQuantity(), cnt));
				}
			}
		}

		super.commitUpdate(data, srcData);
	}

	/**
	 * commitInsert
	 * 
	 * @param data data
	 */
	@Override
	protected void commitInsert(Supplement data) throws Exception {
		int cnt = utils().daoUpdateGoodsSupplyQuantity(getDataAccessSession(), data.getGoodsId(), data.getSupplyQuantity());
		if (cnt != 1) {
			throw new RuntimeException(MessageFormat.format(
				ERROR_UPDATE_STOCK_SUPPLY_QUANTITY, data.getSupplyQuantity(), cnt));
		}
		super.commitInsert(data);
	}

}
