/*
 * This file is part of Squirrels.
 * Copyright (C) 2009 Squirrels Develop Team.
 *
 * Squirrels is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License any later version.
 * 
 * Squirrels is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Squirrels. If not, see <http://www.gnu.org/licenses/>.
 */
package squirrels.ecshop.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;

import squirrels.ecshop.Application;
import squirrels.ecshop.constant.AC;
import squirrels.ecshop.model.bean.Category;
import squirrels.ecshop.model.bean.Goods;

/**
 */
public class LuceneUtils {

	private static Log log = LogFactory.getLog(LuceneUtils.class);

	public static void updateIndex(Goods goods) throws Exception {
		List<Goods> as = new ArrayList<Goods>(1);
		as.add(goods);
		updateIndexes(as);
	}

	public static void deleteIndex(Goods goods) throws Exception {
		List<Goods> as = new ArrayList<Goods>(1);
		as.add(goods);
		deleteIndexes(as);
	}

	/**
	 * @param cid category Id
	 * @return category instance
	 */
	public static long getRootCategoryId(Long cid) {
		Map map = (Map)Application.get().getServletContext().getAttribute(AC.CATEGORY_MAP);
		Category c = (Category)map.get(cid);
		if (c != null) {
			while (c.getParent() != null) {
				c = c.getParent();
			}
		}
		return c == null ? cid : c.getId();
	}
	
	public static Document document(Goods goods) throws Exception {
		Document doc = new Document();

		doc.add(new Field("id", goods.getId().toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
		doc.add(new Field("name", goods.getName(), Field.Store.NO, Field.Index.ANALYZED));
		doc.add(new Field("category", String.valueOf(getRootCategoryId(goods.getCategoryId())), Field.Store.YES, Field.Index.NOT_ANALYZED));
		doc.add(new Field("salePrice", String.valueOf(goods.getSalePrice()), Field.Store.NO, Field.Index.NOT_ANALYZED));

		return doc;
	}

	public static Term term(Goods goods) throws Exception {
		Term term = new Term("id", goods.getId().toString());
		return term;
	}

	public static void updateIndexes(List<Goods> goodss) throws Exception {
		IndexWriter writer = Application.getLuceneIndexWriter();
		try {
			for (Goods g : goodss) {
				if (g.beInvalid()) {
					log.debug("deleteIndex: " + g.getId() + " - " + g.getName());
					writer.deleteDocuments(term(g));
				}
				else {
					log.debug("updateIndex: " + g.getId() + " - " + g.getName());
					writer.updateDocument(term(g), document(g));
				}
			}
			writer.optimize();
			writer.commit();
		}
		finally {
			writer.close();
		}
	}

	public static void deleteIndexes(List<Goods> goodss) throws Exception {
		IndexWriter writer = Application.getLuceneIndexWriter();
		try {
			for (Goods g : goodss) {
				log.debug("deleteIndex: " + g.getId() + " - " + g.getName());
				writer.deleteDocuments(term(g));
			}
			writer.optimize();
			writer.commit();
		}
		finally {
			writer.close();
		}
	}
}
