﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using CSCore;

namespace FDK.メディア.サウンド.WASAPI
{
	public class Sound : IDisposable
	{
		public long 長さsample
		{
			get
				=> this._SampleSource.Length;
		}

		public double 長さsec
		{
			get
				=> this._SampleToSec( this.長さsample );
		}

		public long 位置sample
		{
			get
				=> this._SampleSource.Position;

			set
				=> this._SampleSource.Position = value;
		}

		public double 位置sec
		{
			get
				=> this._SampleToSec( this.位置sample );

			set
			{
				long position = this._SecToSample( value );

				if( ( 0 > position ) || ( ( this.長さsample > 0 ) && ( this.長さsample <= position ) ) )		// 長さsample == 0 では例外は出さない
					throw new ArgumentOutOfRangeException();

				this.位置sample = position;
			}
		}

		public bool 再生中である
		{
			get;
			protected set;
		} = false;

		public bool 再生停止中である
		{
			get 
				=> !( this.再生中である );

			protected set 
				=> this.再生中である = !( value );
		}

		public ISampleSource SampleSource
		{
			get
				=> this._SampleSource;
		}

		public IWaveSource WaveSource
		{
			get
				=> this._WaveSource;
		}

		/// <summary>
		///		音量。0.0(無音)～1.0(原音)。
		/// </summary>
		public float Volume
		{
			get
				=> this._Volume;

			set
				=> this._Volume =
					( 0.0f > value ) ? throw new ArgumentOutOfRangeException() :
					( 1.0f < value ) ? throw new ArgumentOutOfRangeException() :
					value;
		}


		/// <summary>
		///		Sound の生成は、コンストラクタではなく <see cref="Device.サウンドを生成する(string)"/> で行うこと。
		///		（Device 内部で持っている Mixer への参照が必要なため。）
		/// </summary>
		/// <param name="path">
		///		サウンドファイルパス
		///	</param>
		/// <param name="mixer">
		///		使用する Mixer。
		///	</param>
		internal Sound( string path, Mixer mixer )
		{
			this._MixerRef = new WeakReference<Mixer>( mixer );
			this._WaveSource = new DecodedWaveSource( path, mixer.WaveFormat );
			this._SampleSource = this._WaveSource.ToSampleSource();
		}

		/// <summary>
		///		Sound の生成は、コンストラクタではなく Device.CreateSound() で行うこと。
		///		（Device 内部で持っている Mixer への参照が必要なため。）
		/// </summary>
		/// <param name="source">
		///		サウンドの IWaveSource インスタンス。
		///	</param>
		/// <param name="mixer">
		///		使用する Mixer。
		///	</param>
		internal Sound( IWaveSource source, Mixer mixer )
		{
			this._MixerRef = new WeakReference<Mixer>( mixer );
			this._WaveSource = source;
			this._SampleSource = this._WaveSource.ToSampleSource();
		}

		/// <summary>
		///		サウンドデータを解放する。
		/// </summary>
		public void Dispose()
		{
			this.Stop();

			Utilities.解放する( ref this._SampleSource );
			Utilities.解放する( ref this._WaveSource );

			this._MixerRef = null;
		}

		/// <summary>
		///		再生を開始する。
		/// </summary>
		/// <param name="再生開始位置sample">
		///		再生開始位置。サンプル単位。（フレーム単位じゃない。）
		/// </param>
		public void Play( long 再生開始位置sample = 0 )
		{
			this.位置sample = 再生開始位置sample;

			if( this._MixerRef.TryGetTarget( out Mixer mixer ) )
			{
				mixer.AddSound( this );
				this.再生中である = true;
			}
		}

		/// <summary>
		///		再生を開始する。
		/// </summary>
		/// <param name="再生開始位置sec">
		///		再生開始位置。秒単位。
		/// </param>
		public void Play( double 再生開始位置sec )
			=> this.Play( this._SecToSample( 再生開始位置sec ) );

		/// <summary>
		///		再生を停止する。
		/// </summary>
		public void Stop()
		{
			if( this._MixerRef.TryGetTarget( out Mixer mixer ) )
			{
				mixer.RemoveSound( this );
			}

			this.再生停止中である = true;
		}


		private IWaveSource _WaveSource = null;

		private ISampleSource _SampleSource = null;

		/// <summary>
		///		この Sound が登録されているミキサーへの弱参照。
		///		循環参照を避けるため、強参照は持たない。
		/// </summary>
		private WeakReference<Mixer> _MixerRef = null;

		private float _Volume = 1.0f;


		private long _SecToSample( double 時間sec )
		{
			var wf = this.SampleSource.WaveFormat;

			long 時間sample = (long) ( 時間sec * wf.SampleRate * wf.Channels + 0.5 ); // +0.5 は四捨五入
			時間sample -= ( 時間sample % wf.Channels );    // チャンネル数の倍数にする。

			return 時間sample;
		}

		private double _SampleToSec( long 時間sample )
		{
			var wf = this.SampleSource.WaveFormat;

			return 時間sample / ( wf.Channels * wf.SampleRate );
		}
	}
}
