﻿using System;
using System.Diagnostics;

namespace FDK.カウンタ
{
	/// <summary>
	/// ある int 型整数を、開始値から終了値まで、一定間隔で単純増加させるカウンタ。
	/// 終了値に達したら開始値に戻る。以降、それを繰り返す。
	/// </summary>
	public class 単純増加後反復カウンタ
	{
		/// <summary>
		/// カウンタを進行し、現在の値を取得する。
		/// </summary>
		public int 現在値
		{
			get
			{
				lock( this.排他利用 )
				{
					this.進行する();
					return this.bs_現在値;
				}
			}
		}
		/// <summary>
		/// カウンタを進行し、現在の値を割合（0.0:開始値 ～ 1.0:終了値）に変換して取得する。
		/// </summary>
		public float 現在値の割合
		{
			get
			{
				lock( this.排他利用 )
				{
					Debug.Assert( 0 != ( this.bs_終了値 - this.bs_開始値 ) );
					this.進行する();
					return (float) ( this.bs_現在値 - this.bs_開始値 ) / (float) ( this.bs_終了値 - this.bs_開始値 );
				}
			}
		}
		public int 開始値
		{
			get
			{
				lock( this.排他利用 )
				{
					return this.bs_開始値;
				}
			}
		}
		public int 終了値
		{
			get
			{
				lock( this.排他利用 )
				{
					return this.bs_終了値;
				}
			}
		}
		/// <summary>
		/// カウンタを進行し、その結果、カウンタの進行がまだ動作中なら true を返す。（終了値に達しているかどうかは別問題。）
		/// </summary>
		public bool 動作中である
		{
			get
			{
				lock( this.排他利用 )
				{
					this.進行する();    // 終了してるかどうか判定する前に、溜まってる進行を全部消化する。
					return this.bs_動作中;
				}
			}
		}
		/// <summary>
		/// カウンタの進行が一時停止されているなら true を返す。（終了値に達しているかどうかは別問題。）
		/// </summary>
		public bool 停止中である
			=> !this.動作中である;

		/// <summary>
		/// コンストラクタ(1) 初期化のみ行い、カウンタは開始しない。
		/// </summary>
		public 単純増加後反復カウンタ()
		{
			this.間隔ms = QPCTimer.未使用;
			this.定間隔進行 = null;

			this.bs_開始値 = 0;
			this.bs_終了値 = 0;
			this.bs_現在値 = 0;
			this.bs_動作中 = false;
		}
		/// <summary>
		/// コンストラクタ(2) 初期化と同時にカウンタを開始する。
		/// </summary>
		public 単純増加後反復カウンタ( int 最初の値, int 最後の値, long 値をひとつ増加させるのにかける時間ms = 1000 ) : this()
		{
			this.開始する( 最初の値, 最後の値, 値をひとつ増加させるのにかける時間ms );
		}

		public void 開始する( int 最初の値, int 最後の値, long 値をひとつ増加させるのにかける時間ms = 1000 )
		{
			lock( this.排他利用 )
			{
				this.間隔ms = 値をひとつ増加させるのにかける時間ms;
				this.定間隔進行 = new 定間隔進行(); // 同時に開始する。
				this.bs_開始値 = 最初の値;
				this.bs_終了値 = System.Math.Max( 最初の値, 最後の値 );    // 逆転禁止。
				this.bs_現在値 = 最初の値;
				this.bs_動作中 = true;
			}
		}
		public void 一時停止する()
		{
			lock( this.排他利用 )
			{
				this.定間隔進行.経過時間の計測を一時停止する();
				this.bs_動作中 = false;
			}
		}
		public void 再開する()
		{
			lock( this.排他利用 )
			{
				this.定間隔進行.経過時間の計測を再開する();
				this.bs_動作中 = true;
			}
		}

		private long 間隔ms = QPCTimer.未使用;
		private 定間隔進行 定間隔進行 = null;
		private readonly object 排他利用 = new object();

		/// <summary>
		/// 前回のこのメソッドの呼び出しからの経過時間をもとに、必要なだけ現在値を増加させる。
		/// カウント値が終了値に達している場合は、開始値に繰り戻す。
		/// </summary>
		private void 進行する()
		{
			Debug.Assert( QPCTimer.未使用 != this.間隔ms );

			lock( this.排他利用 )
			{
				this.定間隔進行?.経過時間の分だけ進行する( this.間隔ms, () => {

					if( this.bs_動作中 )
					{
						this.bs_現在値++;

						// 終了値を超えていれば開始値に戻る。
						if( this.bs_現在値 > this.bs_終了値 )
							this.bs_現在値 = this.bs_開始値;
					}

				} );
			}
		}

		#region " バックストア。"
		//---------------------------------
		private int bs_開始値 = 0;
		private int bs_終了値 = 0;
		private int bs_現在値 = 0;
		private bool bs_動作中 = false;
		//---------------------------------
		#endregion
	}
}
