﻿using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace FDK.メディア
{
	/// <summary>
	/// DirectWrite を使った Direct2D1ビットマップ。
	/// </summary>
	/// <remarks>
	/// 「表示文字列」メンバを設定/更新すれば、次回の描画時にビットマップが生成される。
	/// </remarks>
	public class 文字列画像 : FDK.Activity
	{
		/// <summary>
		/// このメンバを set すれば、次回の進行描画時に画像が更新される。
		/// </summary>
		public string 表示文字列 { get; set; } = null;
		public string フォント名 { get; set; } = "メイリオ";
		public float フォントサイズpt { get; set; } = 20.0f;
		public SharpDX.Direct2D1.InterpolationMode 補正モード { get; set; } = SharpDX.Direct2D1.InterpolationMode.Linear;
		public SharpDX.RectangleF? 転送元矩形dpx { get; set; } = null;
		public bool 加算合成 { get; set; } = false;
		public SharpDX.Size2F レイアウトサイズdpx { get; set; } = new SharpDX.Size2F( -1f, -1f );
		public bool 下詰め { get; set; } = false;

		public 文字列画像()
		{
		}
		public 文字列画像( string 文字列, float フォントサイズpt = 20.0f, string フォント名 = "メイリオ" )
		{
			this.表示文字列 = 文字列;
			this.フォント名 = フォント名;
			this.フォントサイズpt = フォントサイズpt;
		}
		protected override void Onデバイス依存リソースの作成( デバイスリソース dr )
		{
			//FDK.Log.Info( $"{FDK.Utilities.現在のメソッド名}: 文字列ビットマップを生成します。" );

			this.前回の表示文字列 = null;
			if( this.表示文字列.Nullでも空でもない() )
			{
				this.ビットマップを生成する( dr );
				this.前回の表示文字列 = this.表示文字列; // 最初の構築完了。
			}
		}
		protected override void Onデバイス依存リソースの解放( デバイスリソース dr )
		{
			FDK.Utilities.解放する( ref this.黒ブラシ );
			FDK.Utilities.解放する( ref this.白ブラシ );
			FDK.Utilities.解放する( ref this.ビットマップレンダーターゲット );
			FDK.Utilities.解放する( ref this.テキストレイアウト );
			FDK.Utilities.解放する( ref this.テキストフォーマット );

			//FDK.Log.Info( $"{FDK.Utilities.現在のメソッド名}: 文字列ビットマップを解放しました。" );
		}
		public void 進行描画する(
			デバイスリソース dr,
			float 左位置dpx,
			float 上位置dpx,
			float 不透明度0to1 = 1.0f,
			float X方向拡大率 = 1.0f,
			float Y方向拡大率 = 1.0f,
			SharpDX.Matrix? 変換行列3Dpx = null )
		{
			var 変換行列2Dpx = SharpDX.Matrix3x2.Identity
				* SharpDX.Matrix3x2.Scaling( X方向拡大率, Y方向拡大率 )   // スケーリング
				* dr.行列を単位変換するDPXtoPX( SharpDX.Matrix3x2.Translation( 左位置dpx, 上位置dpx ) );  // 平行移動（物理単位）。

			this.進行描画する( dr, 変換行列2Dpx, 変換行列3Dpx, 不透明度0to1 );
		}

		public void 進行描画する(
			デバイスリソース dr,
			SharpDX.Matrix3x2? 変換行列2Dpx = null,
			SharpDX.Matrix? 変換行列3Dpx = null,
			float 不透明度0to1 = 1.0f )
		{
			Debug.Assert( this.活性化している );

			if( this.表示文字列.Nullまたは空である() )
				return;

			// 表示文字列が変更されているなら、ここで表示ビットマップの再構築を行う。
			if( false == string.Equals( this.表示文字列, this.前回の表示文字列 ) )
				this.ビットマップを生成する( dr );

			if( null == this.ビットマップレンダーターゲット )
				return;

			Utilities.D2DBatchDraw( dr.D2DContext1, () => {

				// 変換行列とブレンドモードをD2Dレンダーターゲットに設定する。
				dr.D2DContext1.Transform = 変換行列2Dpx ?? SharpDX.Matrix3x2.Identity;
				dr.D2DContext1.PrimitiveBlend = ( 加算合成 ) ? SharpDX.Direct2D1.PrimitiveBlend.Add : SharpDX.Direct2D1.PrimitiveBlend.SourceOver;

				// D2Dレンダーターゲットに this.Bitmap を描画する。
				using( var bmp = this.ビットマップレンダーターゲット.Bitmap )
				{
					dr.D2DContext1.DrawBitmap(
						bitmap: bmp,
						destinationRectangle: null,
						opacity: 不透明度0to1,
						interpolationMode: this.補正モード,
						sourceRectangle: this.転送元矩形dpx,
						erspectiveTransformRef: 変換行列3Dpx );
				}
			} );
		}

		protected string 前回の表示文字列 = null;
		protected SharpDX.Direct2D1.BitmapRenderTarget ビットマップレンダーターゲット = null;
		protected SharpDX.DirectWrite.TextFormat テキストフォーマット = null;
		protected SharpDX.DirectWrite.TextLayout テキストレイアウト = null;
		protected SharpDX.Direct2D1.SolidColorBrush 白ブラシ = null;
		protected SharpDX.Direct2D1.SolidColorBrush 黒ブラシ = null;

		protected void ビットマップを生成する( デバイスリソース dr )
		{
			this.前回の表示文字列 = this.表示文字列;

			// テキストフォーマット／レイアウトを作成し、表示ビットマップのサイズを計算する。
			if( null == this.テキストフォーマット )
			{
				this.テキストフォーマット = new SharpDX.DirectWrite.TextFormat( dr.DWriteFactory, this.フォント名, this.フォントサイズpt ) {
					TextAlignment = SharpDX.DirectWrite.TextAlignment.Leading,
				};
			}

			if( ( 0.0f >= this.レイアウトサイズdpx.Width ) || ( 0.0f >= this.レイアウトサイズdpx.Height ) )
				this.レイアウトサイズdpx = dr.設計画面サイズdpx;

			this.テキストレイアウト?.Dispose();
			this.テキストレイアウト = new SharpDX.DirectWrite.TextLayout(
				dr.DWriteFactory,
				this.表示文字列,
				this.テキストフォーマット,
				this.レイアウトサイズdpx.Width,
				this.レイアウトサイズdpx.Height );

			var 表示ビットマップのサイズdpx = new SharpDX.Size2F();
			var 上マージン = 0.0f;
			if( this.下詰め )
			{
				表示ビットマップのサイズdpx = new SharpDX.Size2F(
					this.テキストレイアウト.Metrics.WidthIncludingTrailingWhitespace,
					this.レイアウトサイズdpx.Height );       // レイアウトの最大高

				上マージン = this.レイアウトサイズdpx.Height - this.テキストレイアウト.Metrics.Height;
			}
			else
			{
				表示ビットマップのサイズdpx = new SharpDX.Size2F(
					this.テキストレイアウト.Metrics.WidthIncludingTrailingWhitespace,
					this.テキストレイアウト.Metrics.Height );
			}

			// ビットマップレンダーターゲットを生成する。
			using( var target = dr.D2DContext1.Target )	// Target を get すると COM参照カウンタが増えるので注意。
			{
				// D2DContext1.Target が設定済みであること。さもなきゃ例外も出さずに落ちる。
				Debug.Assert( null != target );
			}
			this.ビットマップレンダーターゲット?.Dispose();
			this.ビットマップレンダーターゲット = new SharpDX.Direct2D1.BitmapRenderTarget(
				dr.D2DContext1,
				SharpDX.Direct2D1.CompatibleRenderTargetOptions.None,
				表示ビットマップのサイズdpx );

			// ブラシの作成がまだなら行う。
			if( null == this.白ブラシ )
				this.白ブラシ = new SharpDX.Direct2D1.SolidColorBrush( this.ビットマップレンダーターゲット, SharpDX.Color.LightGray );
			if( null == this.黒ブラシ )
				this.黒ブラシ = new SharpDX.Direct2D1.SolidColorBrush( this.ビットマップレンダーターゲット, SharpDX.Color.Black );

			// ビットマップレンダーターゲットにテキストを描画する。
			Utilities.D2DBatchDraw( this.ビットマップレンダーターゲット, () => {

				this.ビットマップレンダーターゲット.Clear( SharpDX.Color.Transparent );

				this.ビットマップレンダーターゲット.DrawTextLayout(    // ドロップシャドウ
					new SharpDX.Vector2( 1.0f, 上マージン + 1.0f ),
					this.テキストレイアウト,
					this.黒ブラシ,
					SharpDX.Direct2D1.DrawTextOptions.Clip );

				this.ビットマップレンダーターゲット.DrawTextLayout(    // 本体
					new SharpDX.Vector2( 0.0f, 上マージン ),
					this.テキストレイアウト,
					this.白ブラシ,
					SharpDX.Direct2D1.DrawTextOptions.Clip );
			} );
		}
	}
}
