﻿using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace FDK.メディア
{
	public class 画像 : Activity
	{
		public bool 生成成功 => ( null != this.Bitmap );
		public bool 生成失敗 => ( null == this.Bitmap );

		/// <summary>
		/// 画像は、設計単位で作成される。
		/// </summary>
		public SharpDX.Size2F サイズdpx => new SharpDX.Size2F( (float) this.Bitmap.PixelSize.Width, (float) this.Bitmap.PixelSize.Height );

		// あまり頻繁に変更されないであろう描画パラメータは、メソッド引数ではなくプロパティにしておく。
		public SharpDX.Direct2D1.InterpolationMode 補正モード { get; set; } = SharpDX.Direct2D1.InterpolationMode.Linear;
		public bool 加算合成 { get; set; } = false;

		public 画像( string 画像ファイルパス )
		{
			this.画像ファイルパス = FDK.フォルダ.絶対パスに含まれるフォルダ変数を展開して返す( 画像ファイルパス );
		}
		protected override void Onデバイス依存リソースの作成( デバイスリソース dr )
		{
			this.画像を生成する( dr );
		}
		protected override void Onデバイス依存リソースの解放( デバイスリソース dr )
		{
			FDK.Utilities.解放する( ref this.Bitmap );

			string 変数付きファイルパス = フォルダ.絶対パスをフォルダ変数付き絶対パスに変換して返す( this.画像ファイルパス );
			//Log.Info( $"{Utilities.現在のメソッド名}: 画像を解放しました。[{変数付きファイルパス}]" );
		}

		/// <summary>
		/// 画像を描画する。
		/// </summary>
		/// <param name="dr">デバイスリソース。</param>
		/// <param name="左位置dpx">画像の描画先範囲の左上隅X座標。</param>
		/// <param name="上位置dpx">画像の描画先範囲の左上隅Y座標。</param>
		/// <param name="不透明度0to1">不透明度。(0:透明～1:不透明)</param>
		/// <param name="X方向拡大率">画像の横方向の拡大率。</param>
		/// <param name="Y方向拡大率">画像の縦方向の拡大率。</param>
		/// <param name="転送元矩形dpx">画像の転送元範囲。画像は設計単位で作成されている。</param>
		/// <param name="変換行列3Dpx">射影行列。Direct2D の仕様により、設計単位ではなく物理単位で構築すること。</param>
		public virtual void 進行描画する(
			FDK.メディア.デバイスリソース dr,
			float 左位置dpx,
			float 上位置dpx,
			float 不透明度0to1 = 1.0f,
			float X方向拡大率 = 1.0f,
			float Y方向拡大率 = 1.0f,
			SharpDX.RectangleF? 転送元矩形dpx = null,
			SharpDX.Matrix? 変換行列3Dpx = null )
		{
			var 変換行列2Dpx = SharpDX.Matrix3x2.Identity
				* SharpDX.Matrix3x2.Scaling( X方向拡大率, Y方向拡大率 )   // スケーリング。
				* dr.行列を単位変換するDPXtoPX( SharpDX.Matrix3x2.Translation( 左位置dpx, 上位置dpx ) );   // 平行移動（物理単位）。

			this.進行描画する( dr, 変換行列2Dpx, 変換行列3Dpx, 不透明度0to1, 転送元矩形dpx );
		}

		/// <summary>
		/// 画像を描画する。
		/// </summary>
		/// <param name="dr">デバイスリソース。</param>
		/// <param name="変換行列2Dpx">Transform に適用する行列。Direct2D の仕様により、設計単位ではなく物理単位で構築すること。</param>
		/// <param name="変換行列3Dpx">射影行列。Direct2D の仕様により、設計単位ではなく物理単位で構築すること。</param>
		/// <param name="不透明度0to1">不透明度。(0:透明～1:不透明)</param>
		/// <param name="転送元矩形dpx">描画する画像範囲。画像は設計単位で作成されている。</param>
		public virtual void 進行描画する(
			FDK.メディア.デバイスリソース dr,
			SharpDX.Matrix3x2? 変換行列2Dpx = null,
			SharpDX.Matrix? 変換行列3Dpx = null,
			float 不透明度0to1 = 1.0f,
			SharpDX.RectangleF? 転送元矩形dpx = null )
		{
			Debug.Assert( this.活性化している );

			if( null == this.Bitmap )
				return;

			Utilities.D2DBatchDraw( dr.D2DContext1, () => {

				// 変換行列とブレンドモードをD2Dレンダーターゲットに設定する。
				dr.D2DContext1.Transform = 変換行列2Dpx ?? SharpDX.Matrix3x2.Identity;
				dr.D2DContext1.PrimitiveBlend = ( this.加算合成 ) ? SharpDX.Direct2D1.PrimitiveBlend.Add : SharpDX.Direct2D1.PrimitiveBlend.SourceOver;

				// D2Dレンダーターゲットに this.Bitmap を描画する。
				dr.D2DContext1.DrawBitmap(
					bitmap: this.Bitmap,
					destinationRectangle: null,
					opacity: 不透明度0to1,
					interpolationMode: this.補正モード,
					sourceRectangle: 転送元矩形dpx,  // this.Bitmap は設計単位で作成されている。
					erspectiveTransformRef: 変換行列3Dpx ); // null 指定可。
			} );
		}

		protected string 画像ファイルパス = null;
		protected SharpDX.Direct2D1.Bitmap1 Bitmap = null;

		protected void 画像を生成する( FDK.メディア.デバイスリソース dr, SharpDX.Direct2D1.BitmapProperties1 bitmapProperties1 = null )
		{
			var decoder = (SharpDX.WIC.BitmapDecoder) null;
			var sourceFrame = (SharpDX.WIC.BitmapFrameDecode) null;
			var converter = (SharpDX.WIC.FormatConverter) null;

			string 変数付きファイルパス = フォルダ.絶対パスをフォルダ変数付き絶対パスに変換して返す( this.画像ファイルパス );
			try
			{
				#region " 画像ファイルパスの有効性を確認する。"
				//-----------------
				if( string.IsNullOrEmpty( this.画像ファイルパス ) )
				{
					Log.ERROR( $"画像ファイルパスが null または空文字列です。[{変数付きファイルパス}]" );
					return;
				}
				if( false == System.IO.File.Exists( this.画像ファイルパス ) )
				{
					Log.ERROR( $"画像ファイルが存在しません。[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion
				#region " 画像ファイルに対応できるデコーダを見つける。"
				//-----------------
				try
				{
					decoder = new SharpDX.WIC.BitmapDecoder(
						dr.WicImagingFactory2,
						this.画像ファイルパス,
						SharpDX.IO.NativeFileAccess.Read,
						SharpDX.WIC.DecodeOptions.CacheOnLoad );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"画像ファイルに対応するコーデックが見つかりません。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion
				#region " 最初のフレームをデコードし、取得する。"
				//-----------------
				try
				{
					sourceFrame = decoder.GetFrame( 0 );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"画像ファイルの最初のフレームのデコードに失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion
				#region " 32bitPBGRA へのフォーマットコンバータを生成する。"
				//-----------------
				try
				{
					// WICイメージングファクトリから新しいコンバータを生成。
					converter = new SharpDX.WIC.FormatConverter( dr.WicImagingFactory2 );

					// コンバータに変換元フレームや変換後フォーマットなどを設定。
					converter.Initialize(
						sourceRef: sourceFrame,
						dstFormat: SharpDX.WIC.PixelFormat.Format32bppPBGRA,
						dither: SharpDX.WIC.BitmapDitherType.None,
						paletteRef: null,
						alphaThresholdPercent: 0.0,
						paletteTranslate: SharpDX.WIC.BitmapPaletteType.MedianCut );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"32bitPBGRA へのフォーマットコンバータの生成または初期化に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion
				#region " コンバータを使って、フレームを WICビットマップ経由で D2D ビットマップに変換する。"
				//-----------------
				try
				{
					// WIC ビットマップを D2D ビットマップに変換する。
					this.Bitmap?.Dispose();
					this.Bitmap = SharpDX.Direct2D1.Bitmap1.FromWicBitmap(
						dr.D2DContext1,
						converter,
						bitmapProperties1 );
				}
				catch( SharpDX.SharpDXException e )
				{
					Log.ERROR( $"Direct2D1.Bitmap1 への変換に失敗しました。(0x{e.HResult:x8})[{変数付きファイルパス}]" );
					return;
				}
				//-----------------
				#endregion

				//Log.Info( $"{Utilities.現在のメソッド名}: 画像を生成しました。[{変数付きファイルパス}]" );
			}
			finally
			{
				converter?.Dispose();
				sourceFrame?.Dispose();
				decoder?.Dispose();
			}
		}
	}
}
