﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace SSTFormat
{
	public class スコア : IDisposable
	{
		// ヘルパ

		/// <summary>
		/// 指定されたコマンド名が対象文字列内で使用されている場合に、パラメータ部分の文字列を返す。
		/// </summary>
		/// <remarks>
		/// .dtx や box.def 等で使用されている "#＜コマンド名＞[:]＜パラメータ＞[;コメント]" 形式の文字列（対象文字列）について、
		/// 指定されたコマンドを使用する行であるかどうかを判別し、使用する行であるなら、そのパラメータ部分の文字列を引数に格納し、true を返す。
		/// 対象文字列のコマンド名が指定したコマンド名と異なる場合には、パラメータ文字列に null を格納して false を返す。
		/// コマンド名は正しくてもパラメータが存在しない場合には、空文字列("") を格納して true を返す。
		/// </remarks>
		/// <param name="対象文字列">調べる対象の文字列。（例: "#TITLE: 曲名 ;コメント"）</param>
		/// <param name="コマンド名">調べるコマンドの名前（例:"TITLE"）。#は不要、大文字小文字は区別されない。</param>
		/// <returns>パラメータ文字列の取得に成功したら true、異なるコマンドだったなら false。</returns>
		public static bool コマンドのパラメータ文字列部分を返す( string 対象文字列, string コマンド名, out string パラメータ文字列 )
		{
			// コメント部分を除去し、両端をトリムする。なお、全角空白はトリムしない。
			対象文字列 = 対象文字列.Split( ';' )[ 0 ].Trim( ' ', '\t' );

			string 正規表現パターン = $@"^\s*#{コマンド名}(:|\s)+(.*)\s*$";  // \s は空白文字。
			var m = Regex.Match( 対象文字列, 正規表現パターン, RegexOptions.IgnoreCase );

			if( m.Success && ( 3 <= m.Groups.Count ) )
			{
				パラメータ文字列 = m.Groups[ 2 ].Value;
				return true;
			}
			else
			{
				パラメータ文字列 = null;
				return false;
			}
		}

		// 定数プロパティ

		public const double db初期BPM = 120.0;
		public const double db初期小節解像度 = 480.0;
		public const double dbBPM初期値固定での1小節4拍の時間ms = ( 60.0 * 1000 ) / ( スコア.db初期BPM / 4.0 );
		public const double dbBPM初期値固定での1小節4拍の時間sec = 60.0 / ( スコア.db初期BPM / 4.0 );

		/// <summary>
		/// 1ms あたりの設計ピクセル数 [dpx] 。
		/// </summary>
		/// <remarks>
		/// BPM 150 のとき、1小節が 234 dpx になるように調整。
		/// → 60秒で150拍のとき、1小節(4拍)が 234 dpx。
		/// → 60秒の間に、150[拍]÷4[拍]＝37.5[小節]。
		/// → 60秒の間に、37.5[小節]×234[dpx/小節]＝ 8775[dpx]。
		/// → 1ms の間に、8775[dpx]÷60000[ms]＝0.14625[dpx/ms]。割り切れて良かった。
		/// </remarks>
		public const double 基準譜面速度dpxms = 0.14625 * 2.25;  // "* 2.25" は「x1.0はもう少し速くてもいいんではないか？」という感覚的な調整分。

		/// <summary>
		/// 1秒あたりの設計ピクセル数 [dpx] 。
		/// </summary>
		public const double db基準譜面速度dpxsec = 基準譜面速度dpxms * 1000.0;

		public static readonly Dictionary<レーン種別, List<チップ種別>> dicSSTFレーンチップ対応表 = new Dictionary<レーン種別, List<チップ種別>>() {
			#region " *** "
			//-----------------
			{ レーン種別.Bass, new List<チップ種別>() { チップ種別.Bass } },
			{ レーン種別.BPM, new List<チップ種別>() { チップ種別.BPM } },
			{ レーン種別.China, new List<チップ種別>() { チップ種別.China } },
			{ レーン種別.HiHat, new List<チップ種別>() { チップ種別.HiHat_Close, チップ種別.HiHat_Foot, チップ種別.HiHat_HalfOpen, チップ種別.HiHat_Open } },
			{ レーン種別.LeftCrash, new List<チップ種別>() { チップ種別.LeftCrash } },
			{ レーン種別.Ride, new List<チップ種別>() { チップ種別.Ride, チップ種別.Ride_Cup } },
			{ レーン種別.RightCrash, new List<チップ種別>() { チップ種別.RightCrash } },
			{ レーン種別.Snare, new List<チップ種別>() { チップ種別.Snare, チップ種別.Snare_ClosedRim, チップ種別.Snare_Ghost, チップ種別.Snare_OpenRim } },
			{ レーン種別.Song, new List<チップ種別>() { チップ種別.背景動画 } },
			{ レーン種別.Splash, new List<チップ種別>() { チップ種別.Splash } },
			{ レーン種別.Tom1, new List<チップ種別>() { チップ種別.Tom1, チップ種別.Tom1_Rim } },
			{ レーン種別.Tom2, new List<チップ種別>() { チップ種別.Tom2, チップ種別.Tom2_Rim } },
			{ レーン種別.Tom3, new List<チップ種別>() { チップ種別.Tom3, チップ種別.Tom3_Rim } },
			//-----------------
			#endregion
		};

		// 背景動画のデフォルト拡張子
		public static readonly List<string> listデフォルト拡張子 = new List<string>() {
			".avi", ".flv", ".mp4", ".wmv", ".mpg", ".mpeg"
		};

		// プロパティ；読み込み時または編集時に設定される

		public string 背景動画ファイル名 = "";

		public class CHeader
		{
			public string str曲名 = "(no title)";
			public string str説明文 = "";
		}
		public CHeader Header = new CHeader();

		public List<チップ> listチップ
		{
			get;
			protected set;
		}
		public List<double> list小節長倍率
		{
			get;
			protected set;
		}
		public int 最大小節番号
		{
			get
			{
				int n最大小節番号 = 0;

				foreach( チップ chip in this.listチップ )
				{
					if( chip.小節番号 > n最大小節番号 )
						n最大小節番号 = chip.小節番号;
				}

				return n最大小節番号;
			}
		}
		public Dictionary<int, string> dicメモ = new Dictionary<int, string>();

		// メソッド

		public スコア()
		{
			this.listチップ = new List<チップ>();
			this.list小節長倍率 = new List<double>();
		}
		public スコア( string str曲データファイル名, bool 左ライド = false, bool 左チャイナ = false, bool 左スプラッシュ = true )
			: this()
		{
			this.t曲データファイルを読み込む( str曲データファイル名, 左ライド, 左チャイナ, 左スプラッシュ );
		}
		public void Dispose()
		{
		}

		/// <summary>
		/// 指定された曲データファイルを読み込む。
		/// </summary>
		/// <remarks>
		/// 失敗すれば何らかの例外を発出する。
		/// </remarks>
		public void t曲データファイルを読み込む( string str曲データファイル名, bool 左ライド = false, bool 左チャイナ = false, bool 左スプラッシュ = true )
		{
			#region " 初期化する。"
			//-----------------
			this.list小節長倍率 = new List<double>();
			this.dicメモ = new Dictionary<int, string>();
			//-----------------
			#endregion
			#region " 曲データファイルを読み込む。"
			//-----------------
			var sr = new StreamReader( str曲データファイル名, Encoding.UTF8 );
			try
			{
				int n行番号 = 0;
				int n現在の小節番号 = 0;
				int n現在の小節解像度 = 384;
				//double db現在の小節長倍率 = 1.0;
				チップ種別 e現在のチップ = チップ種別.Unknown;

				while( !sr.EndOfStream )
				{
					// 1行ずつ読み込む。
					n行番号++;
					string str行 = sr.ReadLine();

					// 前処理。

					#region " 改行は ';' に、TABは空白文字にそれぞれ変換し、先頭末尾の空白を削除する。"
					//-----------------
					str行 = str行.Replace( Environment.NewLine, ";" );
					str行 = str行.Replace( '\t', ' ' );
					str行 = str行.Trim();
					//-----------------
					#endregion
					#region " 行中の '#' 以降はコメントとして除外する。また、コメントだけの行はスキップする。"
					//-----------------
					int n区切り位置 = 0;
					for( n区切り位置 = 0; n区切り位置 < str行.Length; n区切り位置++ )
					{
						if( str行[ n区切り位置 ] == '#' )
							break;
					}
					if( n区切り位置 == 0 )
						continue;       // コメントだけの行はスキップ。
					if( n区切り位置 < str行.Length )
					{
						str行 = str行.Substring( 0, n区切り位置 - 1 );
						str行 = str行.Trim();
					}
					//-----------------
					#endregion
					#region " 空行ならこの行はスキップする。"
					//-----------------
					if( string.IsNullOrEmpty( str行 ) )
						continue;
					//-----------------
					#endregion

					// ヘッダコマンド処理。

					#region " ヘッダコマンドの処理を行う。"
					//-----------------
					if( str行.StartsWith( "Title", StringComparison.OrdinalIgnoreCase ) )
					{
						#region " Title コマンド "
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							FDK.Log.ERROR( $"Title の書式が不正です。スキップします。[{n行番号}行目]" );
							continue;
						}

						this.Header.str曲名 = items[ 1 ].Trim();
						//-----------------
						#endregion

						continue;
					}
					if( str行.StartsWith( "Description", StringComparison.OrdinalIgnoreCase ) )
					{
						#region " Description コマンド "
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							FDK.Log.ERROR( $"Description の書式が不正です。スキップします。[{n行番号}行目]" );
							continue;
						}

						// ２文字のリテラル "\n" は改行に復号。
						this.Header.str説明文 = items[ 1 ].Trim().Replace( @"\n", Environment.NewLine );
						//-----------------
						#endregion

						continue;
					}
					//-----------------
					#endregion

					// メモ（小節単位）処理。

					#region " メモ（小節単位）の処理を行う。"
					//-----------------
					if( str行.StartsWith( "PartMemo", StringComparison.OrdinalIgnoreCase ) )
					{
						#region " '=' 以前を除去する。"
						//-----------------
						int n等号位置 = str行.IndexOf( '=' );
						if( n等号位置 <= 0 )
						{
							FDK.Log.ERROR( $"PartMemo の書式が不正です。スキップします。[{n行番号}]行目]" );
							continue;
						}
						str行 = str行.Substring( n等号位置 + 1 ).Trim();
						if( string.IsNullOrEmpty( str行 ) )
						{
							FDK.Log.ERROR( $"PartMemo の書式が不正です。スキップします。[{n行番号}]行目]" );
							continue;
						}
						//-----------------
						#endregion
						#region " カンマ位置を取得する。"
						//-----------------
						int nカンマ位置 = str行.IndexOf( ',' );
						if( nカンマ位置 <= 0 )
						{
							FDK.Log.ERROR( $"PartMemo の書式が不正です。スキップします。[{n行番号}]行目]" );
							continue;
						}
						//-----------------
						#endregion
						#region " 小節番号を取得する。"
						//-----------------
						string str小節番号 = str行.Substring( 0, nカンマ位置 );
						int n小節番号;
						if( !int.TryParse( str小節番号, out n小節番号 ) || n小節番号 < 0 )
						{
							FDK.Log.ERROR( $"PartMemo の小節番号が不正です。スキップします。[{n行番号}]行目]" );
							continue;
						}
						//-----------------
						#endregion
						#region " メモを取得する。"
						//-----------------
						string strメモ = str行.Substring( nカンマ位置 + 1 );

						// ２文字のリテラル文字列 "\n" は改行に復号。
						strメモ = strメモ.Replace( @"\n", Environment.NewLine );
						//-----------------
						#endregion
						#region " メモが空文字列でないなら dicメモ に登録すると同時に、チップとしても追加する。"
						//-----------------
						if( !string.IsNullOrEmpty( strメモ ) )
						{
							this.dicメモ.Add( n小節番号, strメモ );

							this.listチップ.Add(
								new チップ() {
									チップ種別 = チップ種別.小節メモ,
									小節番号 = n小節番号,
									小節内位置 = 0,
									小節解像度 = 1,
								} );
						}
						//-----------------
						#endregion

						continue;
					}
					//-----------------
					#endregion

					// 上記行頭コマンド以外は、チップ記述行だと見なす。

					#region " チップ記述コマンドの処理を行う。"
					//-----------------

					// 行を区切り文字でトークンに分割。
					string[] tokens = str行.Split( new char[] { ';', ':' } );

					// すべてのトークンについて……
					foreach( string token in tokens )
					{
						// トークンを分割。

						#region " トークンを区切り文字 '=' で strコマンド と strパラメータ に分割し、それぞれの先頭末尾の空白を削除する。"
						//-----------------
						string[] items = token.Split( '=' );

						if( items.Length != 2 )
						{
							if( token.Trim().Length == 0 )  // 空文字列（行末など）は不正じゃない。
								continue;

							FDK.Log.ERROR( $"コマンドとパラメータの記述書式が不正です。このコマンドをスキップします。[{n行番号}行目]" );
							continue;
						}

						string strコマンド = items[ 0 ].Trim();
						string strパラメータ = items[ 1 ].Trim();
						//-----------------
						#endregion

						// コマンド別に処理。

						if( strコマンド.Equals( "Part", StringComparison.OrdinalIgnoreCase ) )
						{
							#region " Part（小節番号指定）コマンド "
							//-----------------

							#region " 小節番号を取得・設定。"
							//-----------------
							string str小節番号 = this.t指定された文字列の先頭から数字文字列を取り出す( ref strパラメータ );
							if( string.IsNullOrEmpty( str小節番号 ) )
							{
								FDK.Log.ERROR( $"Part（小節番号）コマンドに小節番号の記述がありません。このコマンドをスキップします。[{n行番号}行目]" );
								continue;
							}

							int n小節番号 = 0;
							if( !int.TryParse( str小節番号, out n小節番号 ) )
							{
								FDK.Log.ERROR( $"Part（小節番号）コマンドの小節番号が不正です。このコマンドをスキップします。[{n行番号}行目]" );
								continue;
							}
							if( n小節番号 < 0 )
							{
								FDK.Log.ERROR( $"Part（小節番号）コマンドの小節番号が負数です。このコマンドをスキップします。[{n行番号}行目]" );
								continue;
							}

							n現在の小節番号 = n小節番号;
							//-----------------
							#endregion
							#region " Part 属性があれば取得する。"
							//-----------------

							while( strパラメータ.Length > 0 )
							{
								// 属性ID を取得。

								char c属性ID = char.ToLower( strパラメータ[ 0 ] );


								// Part 属性があれば取得する。

								if( c属性ID == 's' )
								{
									#region " 小節長倍率(>0) → list小節長倍率 "
									//-----------------

									strパラメータ = strパラメータ.Substring( 1 ).Trim();

									string str小節長倍率 = this.t指定された文字列の先頭から数字文字列を取り出す( ref strパラメータ );
									strパラメータ = strパラメータ.Trim();

									if( string.IsNullOrEmpty( str小節長倍率 ) )
									{
										FDK.Log.ERROR( $"Part（小節番号）コマンドに小節長倍率の記述がありません。この属性をスキップします。[{n行番号}行目]" );
										continue;
									}

									double db小節長倍率 = 1.0;

									if( !double.TryParse( str小節長倍率, out db小節長倍率 ) )
									{
										FDK.Log.ERROR( $"Part（小節番号）コマンドの小節長倍率が不正です。この属性をスキップします。[{n行番号}行目]" );
										continue;
									}

									if( db小節長倍率 <= 0.0 )
									{
										FDK.Log.ERROR( $"Part（小節番号）コマンドの小節長倍率が 0.0 または負数です。この属性をスキップします。[{n行番号}行目]" );
										continue;
									}


									// 小節長倍率辞書に追加 or 上書き更新。

									this.小節長倍率を設定する( n現在の小節番号, db小節長倍率 );

									//-----------------
									#endregion

									continue;
								}
							}

							//-----------------
							#endregion

							//-----------------
							#endregion

							continue;
						}
						if( strコマンド.Equals( "Lane", StringComparison.OrdinalIgnoreCase ) )
						{
							#region " Lane（レーン指定）コマンド（チップ種別の仮決め）"
							//-----------------
							if( strパラメータ.Equals( "LeftCrash", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.LeftCrash;

							else if( strパラメータ.Equals( "Ride", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Ride;

							else if( strパラメータ.Equals( "China", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.China;

							else if( strパラメータ.Equals( "Splash", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Splash;

							else if( strパラメータ.Equals( "HiHat", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.HiHat_Close;

							else if( strパラメータ.Equals( "Snare", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Snare;

							else if( strパラメータ.Equals( "Bass", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Bass;

							else if( strパラメータ.Equals( "Tom1", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Tom1;

							else if( strパラメータ.Equals( "Tom2", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Tom2;

							else if( strパラメータ.Equals( "Tom3", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.Tom3;

							else if( strパラメータ.Equals( "RightCrash", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.RightCrash;

							else if( strパラメータ.Equals( "BPM", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.BPM;

							else if( strパラメータ.Equals( "Song", StringComparison.OrdinalIgnoreCase ) )
								e現在のチップ = チップ種別.背景動画;
							else
								FDK.Log.ERROR( $"Lane（レーン指定）コマンドのパラメータ記述 '{strパラメータ}' が不正です。このコマンドをスキップします。[{n行番号}行目]" );
							//-----------------
							#endregion

							continue;
						}
						if( strコマンド.Equals( "Resolution", StringComparison.OrdinalIgnoreCase ) )
						{
							#region " Resolution（小節解像度指定）コマンド "
							//-----------------
							int n解像度 = 1;
							if( !int.TryParse( strパラメータ, out n解像度 ) )
							{
								FDK.Log.ERROR( $"Resolution（小節解像度指定）コマンドの解像度が不正です。このコマンドをスキップします。[{n行番号}行目]" );
								continue;
							}
							if( n解像度 < 1 )
							{
								FDK.Log.ERROR( $"Resolution（小節解像度指定）コマンドの解像度は 1 以上でなければなりません。このコマンドをスキップします。[{n行番号}行目]" );
								continue;
							}
							n現在の小節解像度 = n解像度;
							//-----------------
							#endregion

							continue;
						}
						if( strコマンド.Equals( "Chips", StringComparison.OrdinalIgnoreCase ) )
						{
							#region " Chips（チップ指定）コマンド "
							//-----------------

							// パラメータを区切り文字 ',' でチップトークンに分割。
							string[] chipTokens = strパラメータ.Split( ',' );

							// すべてのチップトークンについて……
							for( int i = 0; i < chipTokens.Length; i++ )
							{
								chipTokens[ i ].Trim();

								#region " 空文字はスキップ。"
								//-----------------
								if( chipTokens[ i ].Length == 0 )
									continue;
								//-----------------
								#endregion
								#region " チップを生成する。"
								//-----------------
								var chip = new チップ() {
									小節番号 = n現在の小節番号,
									チップ種別 = e現在のチップ,
									小節解像度 = n現在の小節解像度,
									音量 = 4,
								};
								if( chip.チップ種別 == チップ種別.China ) chip.チップ内文字列 = "C N";
								if( chip.チップ種別 == チップ種別.Splash ) chip.チップ内文字列 = "S P";

								if( ( e現在のチップ == チップ種別.BPM ) ||
									( e現在のチップ == チップ種別.背景動画 ) ||
									( e現在のチップ == チップ種別.小節メモ ) ||
									( e現在のチップ == チップ種別.Unknown ) )
								{
									chip.可視 = false;
								}
								//-----------------
								#endregion

								#region " チップ位置を取得する。"
								//-----------------
								int nチップ位置 = 0;

								string str位置番号 = this.t指定された文字列の先頭から数字文字列を取り出す( ref chipTokens[ i ] );
								chipTokens[ i ].Trim();

								#region " 文法チェック。"
								//-----------------
								if( string.IsNullOrEmpty( str位置番号 ) )
								{
									FDK.Log.ERROR( $"チップの位置指定の記述がありません。このチップをスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
									continue;
								}
								//-----------------
								#endregion
								#region " 位置を取得。"
								//-----------------
								if( !int.TryParse( str位置番号, out nチップ位置 ) )
								{
									FDK.Log.ERROR( $"チップの位置指定の記述が不正です。このチップをスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
									continue;
								}
								//-----------------
								#endregion
								#region " 値域チェック。"
								//-----------------
								if( ( nチップ位置 < 0 ) || ( nチップ位置 >= n現在の小節解像度 ) )
								{
									FDK.Log.ERROR( $"チップの位置が負数であるか解像度(Resolution)以上の値になっています。このチップをスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
									continue;
								}
								//-----------------
								#endregion

								chip.小節内位置 = nチップ位置;
								//-----------------
								#endregion
								#region " 共通属性・レーン別属性があれば取得する。"
								//-----------------
								while( chipTokens[ i ].Length > 0 )
								{
									// 属性ID を取得。
									char c属性ID = char.ToLower( chipTokens[ i ][ 0 ] );

									// 共通属性があれば取得。
									if( c属性ID == 'v' )
									{
										#region " 音量 "
										//-----------------
										chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();

										string str音量 = this.t指定された文字列の先頭から数字文字列を取り出す( ref chipTokens[ i ] );
										chipTokens[ i ].Trim();

										int nチップ音量 = 0;

										#region " 文法チェック。"
										//-----------------
										if( string.IsNullOrEmpty( str音量 ) )
										{
											FDK.Log.ERROR( $"チップの音量指定の記述がありません。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											continue;
										}
										//-----------------
										#endregion
										#region " チップ音量の取得。"
										//-----------------
										if( !int.TryParse( str音量, out nチップ音量 ) )
										{
											FDK.Log.ERROR( $"チップの音量指定の記述が不正です。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											continue;
										}
										//-----------------
										#endregion
										#region " 値域チェック。"
										//-----------------
										if( nチップ音量 < 1 || nチップ音量 > チップ.最大音量 )
										{
											FDK.Log.ERROR( $"チップの音量が適正範囲（1～{チップ.最大音量}）を超えています。このチップをスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											continue;
										}
										//-----------------
										#endregion

										chip.音量 = nチップ音量;
										//-----------------
										#endregion

										continue;
									}


									// レーン別属性があれば取得。

									switch( e現在のチップ )
									{
										#region " case LeftCymbal "
										//-----------------
										case チップ種別.LeftCrash:

											#region " 未知の属性 "
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region " case Ride "
										//-----------------
										case チップ種別.Ride:
										case チップ種別.Ride_Cup:

											if( c属性ID == 'c' )
											{
												#region " Ride.カップ "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Ride_Cup;
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case China "
										//-----------------
										case チップ種別.China:

											#region " 未知の属性 "
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region " case Splash "
										//-----------------
										case チップ種別.Splash:

											#region " 未知の属性 "
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region " case HiHat "
										//-----------------
										case チップ種別.HiHat_Close:
										case チップ種別.HiHat_HalfOpen:
										case チップ種別.HiHat_Open:
										case チップ種別.HiHat_Foot:

											if( c属性ID == 'o' )
											{
												#region " HiHat.オープン "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.HiHat_Open;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'h' )
											{
												#region " HiHat.ハーフオープン "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.HiHat_HalfOpen;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'c' )
											{
												#region " HiHat.クローズ "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.HiHat_Close;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'f' )
											{
												#region " HiHat.フットスプラッシュ "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.HiHat_Foot;
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case Snare "
										//-----------------
										case チップ種別.Snare:
										case チップ種別.Snare_ClosedRim:
										case チップ種別.Snare_OpenRim:
										case チップ種別.Snare_Ghost:

											if( c属性ID == 'o' )
											{
												#region " Snare.オープンリム "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Snare_OpenRim;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'c' )
											{
												#region " Snare.クローズドリム "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Snare_ClosedRim;
												//-----------------
												#endregion
											}
											else if( c属性ID == 'g' )
											{
												#region " Snare.ゴースト "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Snare_Ghost;
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case Bass "
										//-----------------
										case チップ種別.Bass:

											#region " 未知の属性 "
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region " case Tom1 "
										//-----------------
										case チップ種別.Tom1:
										case チップ種別.Tom1_Rim:

											if( c属性ID == 'r' )
											{
												#region " Tom1.リム "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Tom1_Rim;
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case Tom2 "
										//-----------------
										case チップ種別.Tom2:
										case チップ種別.Tom2_Rim:

											if( c属性ID == 'r' )
											{
												#region " Tom2.リム "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Tom2_Rim;
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case Tom3 "
										//-----------------
										case チップ種別.Tom3:
										case チップ種別.Tom3_Rim:

											if( c属性ID == 'r' )
											{
												#region " Tom3.リム "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												chip.チップ種別 = チップ種別.Tom3_Rim;
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case RightCymbal "
										//-----------------
										case チップ種別.RightCrash:

											#region " 未知の属性 "
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											//-----------------
											#endregion

											continue;

										//-----------------
										#endregion
										#region " case BPM "
										//-----------------
										case チップ種別.BPM:

											if( c属性ID == 'b' )
											{
												#region " BPM値 "
												//-----------------

												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();

												string strBPM = this.t指定された文字列の先頭から数字文字列を取り出す( ref chipTokens[ i ] );
												chipTokens[ i ].Trim();

												if( string.IsNullOrEmpty( strBPM ) )
												{
													FDK.Log.ERROR( $"BPM数値の記述がありません。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
													continue;
												}

												double dbBPM = 0.0;

												if( !double.TryParse( strBPM, out dbBPM ) || dbBPM <= 0.0 )
												{
													FDK.Log.ERROR( $"BPM数値の記述が不正です。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
													continue;
												}

												chip.BPM = dbBPM;
												chip.チップ内文字列 = dbBPM.ToString( "###.##" );
												//-----------------
												#endregion
											}
											else
											{
												#region " 未知の属性 "
												//-----------------
												chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
												FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
												//-----------------
												#endregion
											}
											continue;

										//-----------------
										#endregion
										#region " case Song "
										//-----------------
										case チップ種別.背景動画:

											#region " 未知の属性 "
											//-----------------
											chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
											FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
											//-----------------
											#endregion

											continue;

											//-----------------
											#endregion
									}

									#region " 未知の属性 "
									//-----------------
									chipTokens[ i ] = chipTokens[ i ].Substring( 1 ).Trim();
									FDK.Log.ERROR( $"未対応の属性「{c属性ID}」が指定されています。この属性をスキップします。[{n行番号}行目; {i + 1}個目のチップ]" );
									//-----------------
									#endregion
								}
								//-----------------
								#endregion

								// チップをリストに追加。
								this.listチップ.Add( chip );
							}
							//-----------------
							#endregion

							continue;
						}

						FDK.Log.ERROR( $"不正なコマンド「{strコマンド}」が存在します。[{n行番号}行目]" );
					}
					//-----------------
					#endregion
				}
			}
			finally
			{
				sr.Close();
			}
			//-----------------
			#endregion
			#region " 拍線の追加。小節線を先に追加すると小節が１つ増えるので、先に拍線から追加する。"
			//-----------------

			// 「this.n最大小節番号」はチップ数に依存して変化するので、for 文には組み込まないこと。
			int n最大小節番号 = this.最大小節番号;

			for( int i = 0; i <= n最大小節番号; i++ )
			{
				double db小節長倍率 = this.小節長倍率を取得する( i );
				for( int n = 1; n * 0.25 < db小節長倍率; n++ )
				{
					this.listチップ.Add(
						new チップ() {
							小節番号 = i,
							チップ種別 = チップ種別.拍線,
							小節内位置 = (int) ( ( n * 0.25 ) * 100 ),
							小節解像度 = (int) ( db小節長倍率 * 100 ),
						} );
				}
			}
			//-----------------
			#endregion
			#region " 小節線の追加。"
			//-----------------
			n最大小節番号 = this.最大小節番号;

			for( int i = 0; i <= n最大小節番号 + 1; i++ )
			{
				this.listチップ.Add(
					new チップ() {
						小節番号 = i,
						チップ種別 = チップ種別.小節線,
						小節内位置 = 0,
						小節解像度 = 1,
					} );
			}
			//-----------------
			#endregion

			this.listチップ.Sort();

			#region " 全チップの発声/描画時刻と譜面内位置を計算する。"
			//-----------------

			// 1. BPMチップを無視し(初期BPMで固定)、dic小節長倍率, Cチップ.n小節解像度, Cチップ.n小節内位置 から両者を計算する。
			//    以下、listチップが小節番号順にソートされているという前提で。

			double dbチップが存在する小節の先頭時刻ms = 0.0;
			int n現在の小節の番号 = 0;

			foreach( チップ chip in this.listチップ )
			{
				#region " チップの小節番号が現在の小節番号よりも大きい場合、チップが存在する小節に至るまで、「dbチップが存在する小節の先頭時刻ms」を更新する。"
				//-----------------
				while( n現在の小節の番号 < chip.小節番号 )
				{
					double db現在の小節の小節長倍率 = this.小節長倍率を取得する( n現在の小節の番号 );
					dbチップが存在する小節の先頭時刻ms += dbBPM初期値固定での1小節4拍の時間ms * db現在の小節の小節長倍率;

					n現在の小節の番号++;    // n現在の小節番号 が chip.n小節番号 に追いつくまでループする。
				}
				//-----------------
				#endregion
				#region " チップの発声/描画時刻を求める。"
				//-----------------
				double dbチップが存在する小節の小節長倍率 = this.小節長倍率を取得する( n現在の小節の番号 );

				chip.発声時刻ms =
					chip.描画時刻ms =
						(long) ( dbチップが存在する小節の先頭時刻ms + ( dbBPM初期値固定での1小節4拍の時間ms * dbチップが存在する小節の小節長倍率 * chip.小節内位置 ) / chip.小節解像度 );
				//-----------------
				#endregion
			}

			// 2. BPMチップを考慮しながら調整する。（譜面内位置grid はBPMの影響を受けないので無視）

			double db現在のBPM = スコア.db初期BPM;
			int nチップ数 = this.listチップ.Count;
			for( int i = 0; i < nチップ数; i++ )
			{
				// BPM チップ以外は無視。
				var BPMチップ = this.listチップ[ i ];
				if( BPMチップ.チップ種別 != チップ種別.BPM )
					continue;

				// BPMチップより後続の全チップの n発声/描画時刻ms を、新旧BPMの比率（加速率）で修正する。
				double db加速率 = BPMチップ.BPM / db現在のBPM; // BPMチップ.dbBPM > 0.0 であることは読み込み時に保証済み。
				for( int j = i + 1; j < nチップ数; j++ )
				{
					this.listチップ[ j ].発声時刻ms =
						this.listチップ[ j ].描画時刻ms =
							(long) ( BPMチップ.発声時刻ms + ( ( this.listチップ[ j ].発声時刻ms - BPMチップ.発声時刻ms ) / db加速率 ) );
				}

				db現在のBPM = BPMチップ.BPM;
			}
			//-----------------
			#endregion
		}
		public void t曲データファイルを読み込む_ヘッダだけ( string str曲データファイル名 )
		{
			this.list小節長倍率 = new List<double>();

			#region " 曲データファイルを読み込む。"
			//-----------------
			var sr = new StreamReader( str曲データファイル名, Encoding.UTF8 );
			try
			{
				int n行番号 = 0;
				//int n現在の小節番号 = 0;
				//int n現在の小節解像度 = 384;
				//double db現在の小節長倍率 = 1.0;
				//チップ種別 e現在のチップ = チップ種別.Unknown;

				while( !sr.EndOfStream )
				{
					// 1行ずつ読み込む。

					n行番号++;

					string str行 = sr.ReadLine();

					// 前処理。

					#region " 文字列に前処理を行う。"
					//-----------------

					// 改行は ';' に、TABは空白文字にそれぞれ変換し、先頭末尾の空白を削除する。

					str行 = str行.Replace( Environment.NewLine, ";" );
					str行 = str行.Replace( '\t', ' ' );
					str行 = str行.Trim();


					// 行中の '#' 以降はコメントとして除外する。また、コメントだけの行はスキップする。

					int n区切り位置 = 0;

					for( n区切り位置 = 0; n区切り位置 < str行.Length; n区切り位置++ )
					{
						if( str行[ n区切り位置 ] == '#' )
							break;
					}

					if( n区切り位置 == 0 )
						continue;       // コメントだけの行はスキップ。

					if( n区切り位置 < str行.Length )
					{
						str行 = str行.Substring( 0, n区切り位置 - 1 );
						str行 = str行.Trim();
					}


					// 空行ならこの行はスキップする。

					if( string.IsNullOrEmpty( str行 ) )
						continue;

					//-----------------
					#endregion

					// ヘッダコマンド処理。

					#region " ヘッダコマンドの処理を行う。"
					//-----------------

					if( str行.StartsWith( "Title", StringComparison.OrdinalIgnoreCase ) )
					{
						#region " Title コマンド "
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							FDK.Log.ERROR( $"Title の書式が不正です。スキップします。[{n行番号}行目]" );
							continue;
						}

						this.Header.str曲名 = items[ 1 ].Trim();
						//-----------------
						#endregion

						continue;
					}
					if( str行.StartsWith( "Description", StringComparison.OrdinalIgnoreCase ) )
					{
						#region " Description コマンド "
						//-----------------
						string[] items = str行.Split( '=' );

						if( items.Length != 2 )
						{
							FDK.Log.ERROR( $"Description の書式が不正です。スキップします。[{n行番号}行目]" );
							continue;
						}

						// ２文字のリテラル "\n" は改行に復号。
						this.Header.str説明文 = items[ 1 ].Trim().Replace( @"\n", Environment.NewLine );
						//-----------------
						#endregion

						continue;
					}
					//-----------------
					#endregion

					// 上記行頭コマンド以外は無視。
				}
			}
			finally
			{
				sr.Close();
			}
			//-----------------
			#endregion
		}
		/// <summary>
		/// 現在の Cスコア の内容をデータファイル（*.sstf）に書き出す。
		/// </summary>
		/// <remarks>
		/// 小節線、拍線、Unknown チップは出力しない。
		/// 失敗すれば何らかの例外を発出する。
		/// </remarks>
		public void t曲データファイルを書き出す( string str曲データファイル名, string strヘッダ行 )
		{
			var sw = new StreamWriter( str曲データファイル名, false, Encoding.UTF8 );
			try
			{
				// ヘッダ行の出力

				sw.WriteLine( $"{strヘッダ行}" );    // strヘッダ行に"{...}"が入ってても大丈夫なようにstring.Format()で囲む。
				sw.WriteLine( "" );

				// ヘッダコマンド行の出力

				sw.WriteLine( "Title={0}", ( string.IsNullOrEmpty( this.Header.str曲名 ) ) ? "(no title)" : this.Header.str曲名 );
				if( !string.IsNullOrEmpty( this.Header.str説明文 ) )
				{
					// 改行コードは、２文字のリテラル "\n" に置換。
					sw.WriteLine( "Description=" + this.Header.str説明文.Replace( Environment.NewLine, @"\n" ) );
				}
				sw.WriteLine( "" );

				// 全チップの出力

				#region " 全チップの最終小節番号を取得する。"
				//-----------------
				int n最終小節番号 = 0;
				foreach( var cc in this.listチップ )
				{
					if( cc.小節番号 > n最終小節番号 )
						n最終小節番号 = cc.小節番号;
				}
				//-----------------
				#endregion

				for( int n小節番号 = 0; n小節番号 <= n最終小節番号; n小節番号++ )
				{
					#region " dicレーン別チップリストの初期化。"
					//-----------------
					Dictionary<レーン種別, List<チップ>> dicレーン別チップリスト = new Dictionary<レーン種別, List<チップ>>();

					foreach( var lane in Enum.GetValues( typeof( レーン種別 ) ) )
						dicレーン別チップリスト[ (レーン種別) lane ] = new List<チップ>();
					//-----------------
					#endregion
					#region " dicレーン別チップリストの構築； n小節番号 の小節に存在するチップのみをレーン別に振り分けて格納する。"
					//-----------------
					foreach( var cc in this.listチップ )
					{
						if( cc.小節番号 > n小節番号 )
							break;      // listチップは昇順に並んでいるので、これ以上検索しても無駄。

						if( cc.チップ種別 == チップ種別.小節線 ||
							cc.チップ種別 == チップ種別.拍線 ||
							cc.チップ種別 == チップ種別.小節メモ ||
							cc.チップ種別 == チップ種別.Unknown )
						{
							continue;   // これらは出力しないので無視。
						}

						if( cc.小節番号 == n小節番号 )
						{
							var lane = レーン種別.Bass;   // 対応するレーンがなかったら Bass でも返しておく。

							foreach( var kvp in dicSSTFレーンチップ対応表 )
							{
								if( kvp.Value.Contains( cc.チップ種別 ) )
								{
									lane = kvp.Key;
									break;
								}
							}

							dicレーン別チップリスト[ lane ].Add( cc );
						}
					}
					//-----------------
					#endregion

					#region " Part行 出力。"
					//-----------------
					sw.Write( "Part = {0}", n小節番号.ToString() );

					if( this.list小節長倍率[ n小節番号 ] != 1.0 )
						sw.Write( "s" + this.list小節長倍率[ n小節番号 ].ToString() );

					sw.WriteLine( ";" );
					//-----------------
					#endregion
					#region " Lane, Resolution, Chip 行 出力。"
					//-----------------
					foreach( var laneObj in Enum.GetValues( typeof( レーン種別 ) ) )
					{
						var lane = (レーン種別) laneObj;

						if( dicレーン別チップリスト[ lane ].Count > 0 )
						{
							sw.Write( "Lane={0}; ", lane.ToString() );

							#region " 新しい解像度を求める。"
							//-----------------
							int n新しい解像度 = 1;
							foreach( var cc in dicレーン別チップリスト[ lane ] )
								n新しい解像度 = FDK.Utilities.最小公倍数を返す( n新しい解像度, cc.小節解像度 );
							//-----------------
							#endregion
							#region " dicレーン別チップリスト[ lane ] 要素の n小節解像度 と n小節内位置 を n新しい解像度 に合わせて修正する。 "
							//-----------------
							foreach( var cc in dicレーン別チップリスト[ lane ] )
							{
								int n倍率 = n新しい解像度 / cc.小節解像度;      // n新しい解像度は n小節解像度 の最小公倍数なので常に割り切れる

								cc.小節解像度 *= n倍率;
								cc.小節内位置 *= n倍率;
							}
							//-----------------
							#endregion

							sw.Write( "Resolution = {0}; ", n新しい解像度 );
							sw.Write( "Chips = " );

							for( int i = 0; i < dicレーン別チップリスト[ lane ].Count; i++ )
							{
								チップ cc = dicレーン別チップリスト[ lane ][ i ];

								// 位置を出力。
								sw.Write( cc.小節内位置.ToString() );

								// 属性を出力（あれば）。

								#region " (1) 共通属性 "
								//-----------------
								if( cc.音量 < チップ.最大音量 )
									sw.Write( "v" + cc.音量.ToString() );
								//-----------------
								#endregion
								#region " (2) 専用属性 "
								//-----------------
								switch( cc.チップ種別 )
								{
									case チップ種別.Ride_Cup:
										sw.Write( 'c' );
										break;

									case チップ種別.HiHat_Open:
										sw.Write( 'o' );
										break;

									case チップ種別.HiHat_HalfOpen:
										sw.Write( 'h' );
										break;

									case チップ種別.HiHat_Foot:
										sw.Write( 'f' );
										break;

									case チップ種別.Snare_OpenRim:
										sw.Write( 'o' );
										break;

									case チップ種別.Snare_ClosedRim:
										sw.Write( 'c' );
										break;

									case チップ種別.Snare_Ghost:
										sw.Write( 'g' );
										break;

									case チップ種別.Tom1_Rim:
										sw.Write( 'r' );
										break;

									case チップ種別.Tom2_Rim:
										sw.Write( 'r' );
										break;

									case チップ種別.Tom3_Rim:
										sw.Write( 'r' );
										break;

									case チップ種別.BPM:
										sw.Write( "b" + cc.BPM.ToString() );
										break;
								}
								//-----------------
								#endregion

								// 区切り文字 または 終端文字 を出力
								sw.Write( ( i == dicレーン別チップリスト[ lane ].Count - 1 ) ? ";" : "," );
							}

							sw.WriteLine( "" ); // 改行
						}
					}
					//-----------------
					#endregion

					sw.WriteLine( "" ); // 次の Part 前に１行開ける
				}

				// メモ（小節単位）の出力

				#region " dicメモ を小節番号で昇順にソートし直した dic昇順メモ を作成する。"
				//-----------------
				var dic昇順メモ = new Dictionary<int, string>();
				int n最大小節番号 = this.最大小節番号;

				for( int i = 0; i <= n最大小節番号; i++ )
				{
					if( this.dicメモ.ContainsKey( i ) )
						dic昇順メモ.Add( i, this.dicメモ[ i ] );
				}
				//-----------------
				#endregion
				#region " dic昇順メモを出力する。"
				//-----------------
				foreach( var kvp in dic昇順メモ )
				{
					int n小節番号 = kvp.Key;

					// 改行コードは、２文字のリ照られる "\n" に置換。
					string strメモ = kvp.Value.Replace( Environment.NewLine, @"\n" );

					sw.WriteLine( "PartMemo={0},{1}", n小節番号, strメモ );
				}
				sw.WriteLine( "" );
				//-----------------
				#endregion
			}
			finally
			{
				sw.Close();
			}
		}
		/// <summary>
		/// 指定された Config.Speed を考慮し、指定された時間[ms]の間に流れるピクセル数[dpx]を算出して返す。</para>
		/// </summary>
		[Obsolete]
		public int n指定された時間msに対応する符号付きピクセル数を返す( double speed, long n指定時間ms )
		{
			return (int) ( n指定時間ms * スコア.基準譜面速度dpxms * speed );
		}
		/// <summary>
		/// 指定された Config.Speed を考慮し、指定された時間[秒]の間に流れるピクセル数[dpx]を算出して返す。
		/// </summary>
		public double n指定された時間secに対応する符号付きピクセル数を返す( double speed, double 指定時間sec )
		{
			return ( 指定時間sec * スコア.db基準譜面速度dpxsec * speed );
		}
		public double 小節長倍率を取得する( int n小節番号 )
		{
			// list小節長倍率 が短ければ増設する。
			if( n小節番号 >= this.list小節長倍率.Count )
			{
				int n不足数 = n小節番号 - this.list小節長倍率.Count + 1;
				for( int i = 0; i < n不足数; i++ )
					this.list小節長倍率.Add( 1.0 );
			}

			// 小節番号に対応する倍率を返す。
			return this.list小節長倍率[ n小節番号 ];
		}
		public void 小節長倍率を設定する( int n小節番号, double db倍率 )
		{
			// list小節長倍率 が短ければ増設する。
			if( n小節番号 >= this.list小節長倍率.Count )
			{
				int n不足数 = n小節番号 - this.list小節長倍率.Count + 1;
				for( int i = 0; i < n不足数; i++ )
					this.list小節長倍率.Add( 1.0 );
			}

			// 小節番号に対応付けて倍率を登録する。
			this.list小節長倍率[ n小節番号 ] = db倍率;
		}

		/// <summary>
		/// 取出文字列の先頭にある数字（小数点も有効）の連続した部分を取り出して、戻り値として返す。
		/// また、取出文字列から取り出した数字文字列部分を除去した文字列を再度格納する。
		/// </summary>
		protected string t指定された文字列の先頭から数字文字列を取り出す( ref string str取出文字列 )
		{
			int n桁数 = 0;
			while( ( n桁数 < str取出文字列.Length ) && ( char.IsDigit( str取出文字列[ n桁数 ] ) || str取出文字列[ n桁数 ] == '.' ) )
				n桁数++;

			if( n桁数 == 0 )
				return "";

			string str数字文字列 = str取出文字列.Substring( 0, n桁数 );
			str取出文字列 = ( n桁数 == str取出文字列.Length ) ? "" : str取出文字列.Substring( n桁数 );

			return str数字文字列;
		}
	}
}
