﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.ServiceModel;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using SharpDX;
using FDK;
using FDK.メディア;
using FDK.同期;
using FDK.カウンタ;
using SSTFormat.v2;
using SST.入力;
using SST.設定;
using SST.Viewer;
using SST.曲;
using SST.ステージ;
using SST.ステージ.起動;
using SST.ステージ.タイトル;
using SST.ステージ.ユーザ;
using SST.ステージ.選曲;
using SST.ステージ.曲読込;
using SST.ステージ.演奏;
using SST.ステージ.クリア;
using SST.ステージ.結果;

namespace SST
{
	[ServiceBehavior( InstanceContextMode = InstanceContextMode.Single )]   // サービスインターフェースをシングルスレッドで呼び出す。
	class App : ApplicationForm, IStrokeStyleTService, IDisposable
	{
		public static bool ビュアーモードである
		{
			get;
			set;
		} = false;
		public static bool ビュアーモードではない
		{
			get
				=> !( App.ビュアーモードである );

			set
				=> App.ビュアーモードである = !( value );
		}

		#region //////// グローバルリソース(static) ////////

		/// <remarks>
		///		SharpDX.Mathematics パッケージを参照し、かつ SharpDX 名前空間を using しておくと、
		///		SharpDX で定義する追加の拡張メソッド（NextFloatなど）を使えるようになる。
		/// </remarks>
		public static Random 乱数
		{
			get;
			protected set;
		} = null;

		public static システム設定 システム設定
		{
			get;
			protected set;
		} = null;

		public static 入力管理 入力管理
		{
			get;
			protected set;
		} = null;

		public static FDK.メディア.サウンド.WASAPI.Device サウンドデバイス
		{
			get;
			protected set;
		} = null;

		public static FDK.メディア.サウンド.WASAPI.SoundTimer サウンドタイマ
		{
			get;
			protected set;
		} = null;

		/// <remarks>
		///		デバイス依存リソースあり。
		/// </remarks>
		public static ユーザ管理 ユーザ管理
		{
			get;
			protected set;
		} = null;

		/// <remarks>
		///		デバイス依存リソースあり。
		/// </remarks>
		public static ステージ管理 ステージ管理
		{
			get;
			protected set;
		} = null;

		public static スコア 演奏スコア
		{
			get;
			set;
		} = null;

		public static ViewerMessageQueue ビュアーメッセージキュー
		{
			get;
			protected set;
		}

		public static ViewerMessage 最後に取得したビュアーメッセージ
		{
			get;
			protected set;
		} = null;

		public static MusicNode ビュアー用ノード
		{
			get;
			protected set;
		}

		#endregion


		/// <summary>
		///		static コンストラクタ。
		///		インスタンスのコンストラクタに先駆けて必要となる処理を行う。
		/// </summary>
		static App()
		{
			SST.IO.Folder.初期化する();
		}

		/// <summary>
		///		初期化処理。
		/// </summary>
		public App( string[] args )
			: base( 設計画面サイズ: new SizeF( 1920f, 1080f ), 物理画面サイズ: new SizeF( 960f, 540f ) )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				// 高解像度タイマを使えないならエラー。
				if( !( Stopwatch.IsHighResolution ) )
					throw new Exception( "このシステムは、高解像度タイマをサポートしていません。" );

				//Sleep 精度を上げる。
				timeBeginPeriod( 1 );

				#region " ビュアーモードかどうかを確認する。"
				//----------------
				foreach( var arg in args )
				{
					if( ( "-v" == arg.ToLower() ) || ( "-viewer" == arg.ToLower() ) )
					{
						App.ビュアーモードである = true;
						break;
					}
				}
				//----------------
				#endregion

				// グローバルリソースを初期化する (1)フォーム初期化前

				App.乱数 = new Random( DateTime.Now.Millisecond );

				Log.Header( "システム設定を復元します。" );
				App.システム設定 = システム設定.復元する();

				if( App.ビュアーモードである )
					App.システム設定.全画面モードである = false;   // ビュアーモードでは常にウィンドウモード。

				App.ビュアーメッセージキュー = new ViewerMessageQueue();
				App.最後に取得したビュアーメッセージ = null;
				App.ビュアー用ノード = null;

				#region " メインフォームを初期化する。"
				//----------------
				this.Text = $"{Application.ProductName} {Application.ProductVersion}";
				if( App.ビュアーモードである )
					this.Text += " (Viewer)";

				// ユーザはフォームサイズを変更できない。
				//this.AllowUserResizing = false;
				//----------------
				#endregion

				// グローバルリソースを初期化する (2)フォーム初期化後

				Log.Header( "入力を初期化します。" );
				App.入力管理 = new 入力管理( this.Handle, 32 );

				Log.Header( "サウンドを初期化します。" );
				App.サウンドデバイス = new FDK.メディア.サウンド.WASAPI.Device( CSCore.CoreAudioAPI.AudioClientShareMode.Shared );
				App.サウンドタイマ = new FDK.メディア.サウンド.WASAPI.SoundTimer( App.サウンドデバイス );

				Log.Header( "ユーザを初期化します。" );
				App.ユーザ管理 = ユーザ管理.復元する( App.グラフィックデバイス ); // この中で活性化も行われる。
				App.ユーザ管理.最初のユーザを選択する();

				Log.Header( "ステージを初期化します。" );
				App.ステージ管理 = new ステージ管理();

				App.演奏スコア = null;


				this._活性化する();

				#region " 最初のステージへ遷移する。"
				//----------------
				if( App.ビュアーモードではない )
				{
					// (A) 通常モード
					App.ステージ管理.ステージを遷移する( App.グラフィックデバイス, App.ステージ管理.最初のステージ名 );
				}
				else
				{
					// (B) ビュアーモード
					App.ステージ管理.ステージを遷移する( App.グラフィックデバイス, nameof( ユーザ選択ステージ ) );
				}
				//----------------
				#endregion
			}
		}

		/// <summary>
		///		終了処理。
		/// </summary>
		public new void Dispose()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				this._非活性化する();

				App.ビュアー用ノード = null;
				App.最後に取得したビュアーメッセージ = null;
				App.ビュアーメッセージキュー = null;

				App.演奏スコア?.Dispose();
				App.演奏スコア = null;

				App.ステージ管理.Dispose( App.グラフィックデバイス );
				App.ステージ管理 = null;

				//App.ユーザ管理.保存する();	--> 必要時に更新する。
				App.ユーザ管理 = null;

				//App.システム設定.保存する();	--> 必要時に更新する。
				App.システム設定?.Dispose();
				App.システム設定 = null;

				App.サウンドタイマ?.Dispose();
				App.サウンドタイマ = null;

				App.サウンドデバイス?.Dispose();
				App.サウンドデバイス = null;

				App.入力管理?.キーバインディングを保存する();     // 同上。
				App.入力管理?.Dispose();
				App.入力管理 = null;

				base.Dispose();
			}
		}

		/// <summary>
		///		メインループ。
		/// </summary>
		public sealed override void Run()
		{
			Debug.Assert( ( null != App.ステージ管理.現在のステージ ), "最初のステージが設定されていません。" );

			SharpDX.Windows.RenderLoop.Run( this, () => {

				switch( this._AppStatus )
				{
					case AppStatus.開始:

						// 高速進行タスク起動。
						this._高速進行ステータス = new TriStateEvent( TriStateEvent.状態種別.OFF );
						Task.Factory.StartNew( this._高速進行タスクエントリ );

						// 描画タスク開始。
						this._AppStatus = AppStatus.実行中;
						break;

					case AppStatus.実行中:
						this._進行と描画を行う();
						break;

					case AppStatus.終了:
						Thread.Sleep( 500 );    // 終了待機中。
						break;
				}

			} );
		}

		/// <summary>
		///		アプリフォームが読み込まれた。
		/// </summary>
		protected override void OnLoad( EventArgs e )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				lock( this._高速進行と描画の同期 )
				{
					if( App.システム設定.全画面モードである )
						this.全画面モード = true;
				}
			}
		}

		/// <summary>
		///		アプリフォームが閉じられる。
		/// </summary>
		protected override void OnClosing( System.ComponentModel.CancelEventArgs e )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				lock( this._高速進行と描画の同期 )
				{
					// 通常は進行タスクから終了するが、Alt+F4でここに来た場合はそれが行われてないので、行う。
					if( this._AppStatus != AppStatus.終了 )
					{
						this._アプリを終了する();
					}

					base.OnClosing( e );
				}
			}
		}

		protected override void スワップチェーンに依存するグラフィックリソースを作成する( SharpDX.Direct3D11.Device d3dDevice )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				Debug.Assert( null != d3dDevice );
			}
		}

		protected override void スワップチェーンに依存するグラフィックリソースを解放する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
			}
		}

		#region " IStrokeStyleT の実装 "
		//----------------
		// このアセンブリ（exe）は、WCF で IStrokeStyleTService を公開する。（基本的に、SSTFViewer 向け。）
		// ・このサービスインターフェースは、シングルスレッド（GUIスレッド）で同期実行される。（Appクラスの ServiceBehavior属性を参照。）
		// ・このサービスホストはシングルトンであり、すべてのクライアントセッションは同一（単一）のサービスインスタンスへ接続される。（Program.Main() を参照。）

		/// <summary>
		///		曲を読み込み、演奏を開始する。
		///		ビュアーモードのときのみ有効。
		/// </summary>
		/// <param name="path">曲ファイルパス</param>
		/// <param name="startPart">演奏開始小節番号(0～)</param>
		/// <param name="drumsSound">ドラムチップ音を発声させるなら true。</param>
		public void ViewerPlay( string path, int startPart = 0, bool drumsSound = true )
		{
			if( App.ビュアーモードではない )
				return;

			App.ビュアーメッセージキュー.格納する( new ViewerMessage() {
				種別 = ViewerMessage.指示種別.演奏開始,
				演奏対象曲のファイルパス = path,
				演奏を開始する小節番号 = startPart,
				ドラムチップのヒット時に発声する = drumsSound,
			} );
		}

		/// <summary>
		///		現在の演奏を停止する。
		///		ビュアーモードのときのみ有効。
		/// </summary>
		public void ViewerStop()
		{
			if( App.ビュアーモードではない )
				return;

			App.ビュアーメッセージキュー.格納する( new ViewerMessage() {
				種別 = ViewerMessage.指示種別.演奏停止,
			} );
		}

		/// <summary>
		///		サウンドデバイスの発声遅延[ms]を返す。
		/// </summary>
		/// <returns>遅延量[ms]</returns>
		public float GetSoundDelay()
		{
			if( App.ビュアーモードではない )
				return 0f;

			return (float) ( App.サウンドデバイス?.遅延sec ?? 0.0 ) * 1000.0f;
		}
		//----------------
		#endregion


		private enum AppStatus { 開始, 実行中, 終了 };

		/// <summary>
		///		アプリケーションの状態。
		/// </summary>
		private AppStatus _AppStatus = AppStatus.開始;


		// ※ Form イベントの override メソッドは描画スレッドで実行されるため、処理中に進行タスクが呼び出されると困る場合には、進行タスクとの lock を忘れないこと。
		private readonly object _高速進行と描画の同期 = new object();

		/// <summary>
		///		進行タスクの状態。
		///		OFF:タスク起動前、ON:タスク実行中、OFF:タスク終了済み
		/// </summary>
		private TriStateEvent _高速進行ステータス;


		/// <summary>
		///		高速進行ループの処理内容。
		/// </summary>
		private void _高速進行タスクエントリ()
		{
			Log.現在のスレッドに名前をつける( "高速進行" );
			Log.Header( "高速進行タスクを開始します。" );

			this._高速進行ステータス.現在の状態 = TriStateEvent.状態種別.ON;

			while( true )
			{
				lock( this._高速進行と描画の同期 )
				{
					if( this._高速進行ステータス.現在の状態 != TriStateEvent.状態種別.ON )	// lock してる間に状態が変わることがあるので注意。
						break;

					//App.入力管理.すべての入力デバイスをポーリングする();
					// --> 入力ポーリングの挙動はステージごとに異なるので、それぞれのステージ内で行う。

					App.ステージ管理.現在のステージ.高速進行する();
				}

				Thread.Sleep( 1 );  // ウェイト。
			}

			this._高速進行ステータス.現在の状態 = TriStateEvent.状態種別.無効;

			Log.Header( "高速進行タスクを終了しました。" );
		}

		/// <summary>
		///		進行描画ループの処理内容。
		/// </summary>
		private void _進行と描画を行う()
		{
			var gd = App.グラフィックデバイス;
			bool vsync = false;

			lock( this._高速進行と描画の同期 )
			{
				if( this._AppStatus != AppStatus.実行中 )  // 上記lock中に終了されている場合があればそれをはじく。
					return;

				gd.D3DDeviceを取得する( ( d3dDevice ) => {

					#region " (1) D3Dレンダリングの前処理を行う。"
					//----------------
					// 既定のD3Dレンダーターゲットビューを黒でクリアする。
					d3dDevice.ImmediateContext.ClearRenderTargetView( gd.D3DRenderTargetView, Color4.Black );

					// 深度バッファを 1.0f でクリアする。
					d3dDevice.ImmediateContext.ClearDepthStencilView(
							gd.D3DDepthStencilView,
							SharpDX.Direct3D11.DepthStencilClearFlags.Depth,
							depth: 1.0f,
							stencil: 0 );
					//----------------
					#endregion

					gd.Animation.進行する();

					#region " (2) 現在のステージの進行描画を行う。"
					//----------------
					App.ステージ管理.現在のステージ.進行描画する( gd );
					//----------------
					#endregion

					gd.UIFramework.Render( gd );
					
					#region " (3) ステージの状態をチェックし、遷移処理を行う。（必要に応じてビュアーメッセージキューの処理も行う。）"
					//----------------
					switch( App.ステージ管理.現在のステージ )
					{
						case 起動ステージ stage:
							#region " 完了 → タイトルステージへ "
							//----------------
							if( stage.現在のフェーズ == 起動ステージ.フェーズ.完了 )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( タイトルステージ ) );
							}
							//----------------
							#endregion
							break;

						case タイトルステージ stage:
							#region " 確定 → ユーザステージへ "
							//----------------
							if( stage.現在のフェーズ == タイトルステージ.フェーズ.確定 )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( ユーザ選択ステージ ) );
							}
							//----------------
							#endregion
							#region " キャンセル → アプリを終了する。"
							//----------------
							else if( stage.現在のフェーズ == タイトルステージ.フェーズ.キャンセル )
							{
								App.ステージ管理.ステージを遷移する( gd, null );
								this._アプリを終了する();
							}
							//----------------
							#endregion
							break;

						case ユーザ選択ステージ stage:
							#region " キャンセル → タイトルステージへ "
							//----------------
							if( stage.現在のフェーズ == ユーザ選択ステージ.フェーズ.キャンセル )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( タイトルステージ ) );
							}
							//----------------
							#endregion
							#region " 完了 → 通常モード時は選曲ステージ、ビュアーモード時は演奏ステージへ "
							//----------------
							else if( stage.現在のフェーズ == ユーザ選択ステージ.フェーズ.完了 )
							{
								if( App.ビュアーモードではない )
								{
									App.ステージ管理.ステージを遷移する( gd, nameof( 選曲ステージ ) );
								}
								else
								{
									App.ステージ管理.ステージを遷移する( gd, nameof( 演奏ステージ ) );
								}
							}
							//----------------
							#endregion
							break;

						case 選曲ステージ stage:
							#region " キャンセル → ユーザステージへ "
							//----------------
							if( stage.現在のフェーズ == 選曲ステージ.フェーズ.キャンセル )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( ユーザ選択ステージ ) );
							}
							//----------------
							#endregion
							#region " 曲決定 → 曲読込ステージへ "
							//----------------
							else if( stage.現在のフェーズ == 選曲ステージ.フェーズ.曲決定 )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( 曲読込ステージ ) );
							}
							//----------------
							#endregion
							break;

						case 曲読込ステージ stage:
							#region " 完了 → 演奏ステージへ "
							//----------------
							if( stage.現在のフェーズ == 曲読込ステージ.フェーズ.完了 )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( 演奏ステージ ) );
							}
							//----------------
							#endregion
							break;

						case 演奏ステージ stage:
							if( App.ビュアーモードではない )
							{
								// (A) 通常モード時
								#region " Failed, キャンセル → 選曲ステージへ。"
								//--------------------
								if( ( stage.現在のフェーズ == 演奏ステージ.フェーズ.キャンセル ) ||
									( stage.現在のフェーズ == 演奏ステージ.フェーズ.Failed ) )
								{
									App.ステージ管理.ステージを遷移する( gd, nameof( 選曲ステージ ) );
								}
								//--------------------
								#endregion
								#region " クリア → クリアステージへ。"
								//--------------------
								else if( stage.現在のフェーズ == 演奏ステージ.フェーズ.クリア )
								{
									App.ステージ管理.ステージを遷移する( gd, nameof( クリアステージ ) );
								}
								//--------------------
								#endregion
							}
							else
							{
								// (B) ビュアーモード時
								#region " ビュアーメッセージが届いていれば、処理する。"
								//----------------
								var msg
									= App.最後に取得したビュアーメッセージ
									= App.ビュアーメッセージキュー.取得する();

								if( null != msg )
								{
									Log.Info( $"{msg.ToString()}" );

									switch( msg.種別 )
									{
										case ViewerMessage.指示種別.指示なし:
											break;

										case ViewerMessage.指示種別.演奏開始:
											if( stage.現在のフェーズ == 演奏ステージ.フェーズ.演奏中 ||
												stage.現在のフェーズ == 演奏ステージ.フェーズ.クリア時フェードアウト ||
												stage.現在のフェーズ == 演奏ステージ.フェーズ.ビュアーメッセージ待機 )
											{
												stage.非活性化する( gd );

												stage.BGMを停止する();

												if( ( null != App.ビュアー用ノード ) && App.ビュアー用ノード.活性化している )
													App.ビュアー用ノード.非活性化する( gd );

												App.ビュアー用ノード = new MusicNode( msg.演奏対象曲のファイルパス, null );

												// 曲読込ステージへ。
												App.ステージ管理.ステージを遷移する( gd, nameof( 曲読込ステージ ) );
											}
											break;

										case ViewerMessage.指示種別.演奏停止:
											if( stage.現在のフェーズ == 演奏ステージ.フェーズ.演奏中 ||
												stage.現在のフェーズ == 演奏ステージ.フェーズ.クリア時フェードアウト ||
												stage.現在のフェーズ == 演奏ステージ.フェーズ.ビュアーメッセージ待機 )  // 待機中もまだBGMが鳴ってるかもしれない
											{
												stage.演奏を停止する();
												stage.現在のフェーズ = 演奏ステージ.フェーズ.ビュアーメッセージ待機;
											}
											break;

										default:
											Log.ERROR( "未対応のViewerMessageです。" );
											break;
									}
								}
								//----------------
								#endregion
								#region " クリア, Failed, キャンセル → メッセージ待機へ。"
								//----------------
								if( stage.現在のフェーズ == 演奏ステージ.フェーズ.クリア ||
									stage.現在のフェーズ == 演奏ステージ.フェーズ.Failed ||
									stage.現在のフェーズ == 演奏ステージ.フェーズ.キャンセル )
								{
									// フェーズのみ変更し、BGM は止めない。
									stage.現在のフェーズ = 演奏ステージ.フェーズ.ビュアーメッセージ待機;
								}
								//----------------
								#endregion
							}
							break;

						case クリアステージ stage:
							#region " 完了 → 結果ステージへ "
							//----------------
							if( stage.現在のフェーズ == クリアステージ.フェーズ.完了 )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( 結果ステージ ) );
							}
							//----------------
							#endregion
							break;

						case 結果ステージ stage:
							#region " 完了 → 選曲ステージへ。"
							//--------------------
							if( stage.現在のフェーズ == 結果ステージ.フェーズ.完了 )
							{
								App.ステージ管理.ステージを遷移する( gd, nameof( 選曲ステージ ) );
							}
							//--------------------
							#endregion
							break;
					}
					//----------------
					#endregion

					#region " (4) D3Dコマンドをフラッシュする。 "
					//----------------
					if( App.システム設定.垂直帰線待ちを行う )
					{
						// ID3D11DeviceContext::Flush
						// <https://msdn.microsoft.com/ja-jp/library/windows/desktop/ff476425(v=vs.85).aspx>
						// > We recommend that you use Flush when the CPU waits for an arbitrary amount of time
						// > (such as when you call the Sleep function). 
						// > CPUが（Sleep関数の呼び出しなどで）任意の時間 wait するのであれば、Flush の使用を推奨します。
						d3dDevice.ImmediateContext.Flush();
					}
					//----------------
					#endregion
				} );

				vsync = App.システム設定.垂直帰線待ちを行う;
			}

			#region " (5) スワップチェーンを表示する。"
			//----------------
			gd.SwapChain.Present( ( vsync ) ? 1 : 0, SharpDX.DXGI.PresentFlags.None );
			//----------------
			#endregion
		}

		/// <summary>
		///		グローバルリソースのうち、グラフィックリソースを持つものについて、活性化がまだなら活性化する。
		/// </summary>
		private void _活性化する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				if( App.ユーザ管理.活性化していない )
					App.ユーザ管理.活性化する( App.グラフィックデバイス );

				if( App.ステージ管理.現在のステージ?.活性化していない ?? false )
					App.ステージ管理.現在のステージ?.活性化する( App.グラフィックデバイス );
			}
		}

		/// <summary>
		///		グローバルリソースのうち、グラフィックリソースを持つものについて、活性化中なら非活性化する。
		/// </summary>
		private void _非活性化する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				if( App.ステージ管理.現在のステージ?.活性化している ?? false )
					App.ステージ管理.現在のステージ?.非活性化する( App.グラフィックデバイス );

				if( App.ユーザ管理.活性化している )
					App.ユーザ管理.非活性化する( App.グラフィックデバイス );
			}
		}

		/// <summary>
		///		進行タスクを終了し、ウィンドウを閉じ、アプリを終了する。
		/// </summary>
		private void _アプリを終了する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				if( this._AppStatus != AppStatus.終了 )
				{
					this._高速進行ステータス.現在の状態 = TriStateEvent.状態種別.OFF;
					this._AppStatus = AppStatus.終了;

					// _AppStatus を変更したあとに、GUI スレッドで非同期実行を指示する。
					this.BeginInvoke( new Action( () => { this.Close(); } ) );
				}
			}
		}

		protected override void OnKeyDown( KeyEventArgs e )
		{
			if( e.KeyCode == Keys.F11 )
				this.全画面モード = !( this.全画面モード );
		}

		#region " Win32 "
		//----------------
		[System.Runtime.InteropServices.DllImport( "winmm.dll", EntryPoint = "timeBeginPeriod" )]
		public static extern uint timeBeginPeriod( uint uMilliseconds );

		[System.Runtime.InteropServices.DllImport( "winmm.dll", EntryPoint = "timeEndPeriod" )]
		public static extern uint timeEndPeriod( uint uMilliseconds );
		//----------------
		#endregion
	}
}
