﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.Serialization;
using FDK;

namespace SST.設定
{
	/// <summary>
	///		システム設定。
	///		全SSTユーザに対して共通。
	/// </summary>
	[DataContract( Name = "Configurations", Namespace = "" )]
	[KnownType( typeof( SharpDX.Size2 ) )]
	class システム設定 : IDisposable, IExtensibleDataObject
	{
		/// <summary>
		///		メインウィンドウ（全画面モード時は画面）のサイズ。ピクセル単位。
		/// </summary>
		/// <remarks>
		///		プログラム側は、<see cref="物理画面サイズ"/>とは独立した<see cref="FDK.メディア.デバイスリソース.設計画面サイズ"/>を基にサイズや座標を決定しているので、
		///		<see cref="物理画面サイズ"/>はユーザにより自由なサイズに設定することができる。
		/// </remarks>
		[DataMember]
		public SharpDX.Size2 物理画面サイズ { get; set; }

		/// <summary>
		///		スワップチェーンの表示切替をディスプレイの垂直帰線中に行う場合は true。
		/// </summary>
		/// <remarks>
		///		false にすると、ティアリングや、ビデオカードの不要な発熱が生じる可能性がある。
		///		なお、演奏時の入出力や発声など高精度なタイミング制御が必要なアクションは常に垂直帰線を待たずに行われるため、ここで設定される値とは無関係。
		/// </remarks>
		[DataMember]
		public bool 垂直帰線待ちを行う { get; set; }

		/// <summary>
		///		初期の表示モード。
		///		true なら全画面モードで、false ならウィンドウモード。
		/// </summary>
		[DataMember]
		public bool 全画面モードである { get; set; }

		/// <remarks>
		///		ビュアーメッセージを受理したときに更新される。
		/// </remarks>
		public bool Autoチップのドラム音を再生する { get; set; }

		/// <summary>
		///		コンストラクタ。
		/// </summary>
		public システム設定()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				this._既定値で初期化する( new StreamingContext() );
			}
		}

		public void Dispose()
		{
			// 特になし
		}

		/// <summary>
		///		ファイルに保存する。
		/// </summary>
		public void 保存する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				var path = SST.IO.Folder.絶対パスに含まれるフォルダ変数を展開して返す( _Configurationファイルパス );
				FDKUtilities.保存する( this, path );
			}
		}

		/// <summary>
		///		ファイルから復元する。
		/// </summary>
		public static システム設定 復元する()
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				var path = SST.IO.Folder.絶対パスに含まれるフォルダ変数を展開して返す( _Configurationファイルパス );
				var config = FDKUtilities.復元または新規作成する<システム設定>( path );

				// ファイルに反映されていないメンバはいつまでも反映されないので、ここで一度、明示的に保存することにする。
				config.保存する();

				return config;
			}
		}


		/// <summary>
		///		システム設定を保存するファイルのパス。
		/// </summary>
		private static readonly string _Configurationファイルパス = @"$(AppData)\Configurations.json";

		/// <summary>
		///		コンストラクタ時、または逆シリアル化時のメンバの既定値を設定する。
		/// </summary>
		/// <param name="sc">未使用。</param>
		/// <remarks>
		///		.NET 既定の初期値だと支障のある（逆シリアル化対象の）メンバがあれば、ここで初期化しておくこと。
		/// </remarks>
		[OnDeserializing]
		private void _既定値で初期化する( StreamingContext context )
		{
			using( Log.Block( FDKUtilities.現在のメソッド名 ) )
			{
				this.物理画面サイズ = new SharpDX.Size2( 1280, 720 );	// 1920x1080 の縦横三分の二
				this.垂直帰線待ちを行う = true;
				this.全画面モードである = false;
				this.Autoチップのドラム音を再生する = true;
			}
		}

		#region " IExtensibleDataObject の実装 "
		//----------------
		private ExtensionDataObject _ExData;

		public virtual ExtensionDataObject ExtensionData
		{
			get
				=> this._ExData;

			set
				=> this._ExData = value;
		}
		//----------------
		#endregion
	}
}
