/*
  This program is a conversion from original Swiss Ephemeris Software.
*/
/* Copyright (C) 1997, 1998 Astrodienst AG, Switzerland.  All rights reserved.
  
  This file is part of Swiss Ephemeris Free Edition.
  
  Swiss Ephemeris is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Swiss Ephemeris Public License
  ("SEPL" or the "License") for full details.
  
  Every copy of Swiss Ephemeris must include a copy of the License,
  normally in a plain ASCII text file named LICENSE.  The License grants you
  the right to copy, modify and redistribute Swiss Ephemeris, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notices and this notice be preserved on
  all copies.

  For uses of the Swiss Ephemeris which do not fall under the definitions
  laid down in the Public License, the Swiss Ephemeris Professional Edition
  must be purchased by the developer before he/she distributes any of his
  software or makes available any product or service built upon the use of
  the Swiss Ephemeris.

  Authors of the Swiss Ephemeris: Dieter Koch and Alois Treindl

  The authors of Swiss Ephemeris have no control or influence over any of
  the derived works, i.e. over software or services created by other
  programmers which use Swiss Ephemeris functions.

  The names of the authors or of the copyright holder (Astrodienst) must not
  be used for promoting any software, product or service which uses or contains
  the Swiss Ephemeris. This copyright notice is the ONLY place where the
  names of the authors can legally appear, except in cases where they have
  given special permission in writing.

  The trademarks 'Swiss Ephemeris' and 'Swiss Ephemeris inside' may be used
  for promoting such software, products or services.
*/

package jp.co.isic.SwissEphemeris;

//import java.lang.*;
//import java.io.*;
//import SwissEphemeris.*;

public class SwephLib {

  static double tid_acc = SwissEphemeris.SE_TIDAL_DEFAULT;

  static double TWOPI = (Math.PI * 2.0);

  /* Reduce x modulo 2*PI
   */
  static double swi_mod2PI(double x)
  {
    double y;
    y = Math.IEEEremainder(x, TWOPI);
    if( y < 0.0 ) y += TWOPI;
    return(y);
  }

  static void swi_cross_prod(double a[], int aIndex, double b[], int bIndex,
			     double x[])
  {
    x[0] = a[aIndex+1]*b[bIndex+2] - a[aIndex+2]*b[bIndex+1];
    x[1] = a[aIndex+2]*b[bIndex+0] - a[aIndex+0]*b[bIndex+2];
    x[2] = a[aIndex+0]*b[bIndex+1] - a[aIndex+1]*b[bIndex+0];
  }

  /*
   * conversion between ecliptical and equatorial polar coordinates.
   * for users of SWISSEPH, not used by our routines.
   * for ecl. to equ.  eps must be negative.
   * for equ. to ecl.  eps must be positive.
   * xpo, xpn are arrays of 3 doubles containing position.
   * attention: input must be in degrees!
   */
  static void swe_cotrans(double xpo[], double xpn[], double eps)
  {
    int i;
    double x[], e = eps * SwissEphemeris.DEGTORAD;
    x = new double[6];
    for(i = 0; i <= 1; i++)
      x[i] = xpo[i];
    x[0] *= SwissEphemeris.DEGTORAD;
    x[1] *= SwissEphemeris.DEGTORAD;
    x[2] = 1;
    for(i = 3; i <= 5; i++)
      x[i] = 0;
    swi_polcart(x, 0, x, 0);
    swi_coortrf(x, 0, x, 0, e);
    swi_cartpol(x, 0, x, 0);
    xpn[0] = x[0] * SwissEphemeris.RADTODEG;
    xpn[1] = x[1] * SwissEphemeris.RADTODEG;
    xpn[2] = xpo[2];
  }

  /* DeltaT = Ephemeris Time - Universal Time, in days.
   * 
   * 1620 - today + a couple of years:
   * ---------------------------------
   * The tabulated values of deltaT, in hundredths of a second,
   * were taken from The Astronomical Almanac 1997, page K8.  The program
   * adjusts for a value of secular tidal acceleration ndot = -25.7376.
   * arcsec per century squared, the value used in JPL's DE403 ephemeris.
   * ELP2000 (and DE200) used the value -23.8946.
   * To change ndot, one can
   * either redefine SE_TIDAL_DEFAULT in swephexp.h
   * or use the routine swe_set_tid_acc() before calling Swiss 
   * Ephemeris.
   * Bessel's interpolation formula is implemented to obtain fourth 
   * order interpolated values at intermediate times.
   *
   * -500 - 1620:
   * ---------------------------------
   * For dates between -500 and 1600, the table given by 
   * Stephenson (1997; p. 515) is used, with linear interpolation.
   * This table is based on an assumed value of ndot = -26.
   * The program adjusts for ndot = -25.7376.
   * For 1600 - 1620, a linear interpolation between the last value
   * of the latter and the first value of the former table is made.
   *
   * before -500:
   * ---------------------------------
   * For times before -600, a formula of Stephenson & Morrison (1995) 
   * (S. Stephenson 1997; p. 508) is used: 
   * dt = 35 * t * t - 20 sec, where t is centuries from 1735 AD.
   * For -600 to -500, a transition from this formula to the Stephenson
   * table has been implemented in order to avoid a jump.
   *
   * future:
   * ---------------------------------
   * For the time after the last tabulated value, we use the formula
   * of Stephenson (1997; p. 507), with a modification that avoids a jump
   * at the end of the tabulated period. A linear term is added that
   * makes a slow transition from the table to the formula over a period
   * of 100 years. (Need not be updated, when table will be enlarged.)
   *
   * References:
   *
   * Stephenson, F. R., and L. V. Morrison, "Long-term changes
   * in the rotation of the Earth: 700 B.C. to A.D. 1980,"
   * Philosophical Transactions of the Royal Society of London
   * Series A 313, 47-70 (1984)
   *
   * Borkowski, K. M., "ELP2000-85 and the Dynamical Time
   * - Universal Time relation," Astronomy and Astrophysics
   * 205, L8-L10 (1988)
   * Borkowski's formula is derived from partly doubtful eclipses 
   * going back to 2137 BC and uses lunar position based on tidal 
   * coefficient of -23.9 arcsec/cy^2.
   *
   * Chapront-Touze, Michelle, and Jean Chapront, _Lunar Tables
   * and Programs from 4000 B.C. to A.D. 8000_, Willmann-Bell 1991
   * Their table agrees with the one here, but the entries are
   * rounded to the nearest whole second.
   *
   * Stephenson, F. R., and M. A. Houlden, _Atlas of Historical
   * Eclipse Maps_, Cambridge U. Press (1986)
   *
   * Stephenson, F.R. & Morrison, L.V., "Long-Term Fluctuations in 
   * the Earth's Rotation: 700 BC to AD 1990", Philosophical 
   * Transactions of the Royal Society of London, 
   * Ser. A, 351 (1995), 165-202. 
   *
   * Stephenson, F. Richard, _Historical Eclipses and Earth's 
   * Rotation_, Cambridge U. Press (1997)
   * 
   * Table from AA for 1620 through today
   * Note, Stephenson and Morrison's table starts at the year 1630.
   * The Chapronts' table does not agree with the Almanac prior to 1630.
   * The actual accuracy decreases rapidly prior to 1780.
   *
   * Jean Meeus, Astronomical Algorithms, 2nd edition, 1998.
   * 
   * For a comprehensive collection of publications and formulae, see:
   * http://www.phys.uu.nl/~vgent/astro/deltatime.htm
   *
   * Last update of table dt[]: Dieter Koch, 9 Apr. 2002.
   * ATTENTION: Whenever updating this table, do not forget to adjust
   * the macros TABEND and TABSIZ !
   */
  static final int TABSTART = 1620;
  static final int TABEND = 2014;
  static final int TABSIZ = (TABEND-TABSTART+1);
  /* we make the table greater for additional values read from external file */
  static final int TABSIZ_SPACE = (TABSIZ+50);
  static short dt[] = {
    /* 1620.0 thru 1659.0 */
    12400, 11900, 11500, 11000, 10600, 10200, 9800, 9500, 9100, 8800,
    8500, 8200, 7900, 7700, 7400, 7200, 7000, 6700, 6500, 6300,
    6200, 6000, 5800, 5700, 5500, 5400, 5300, 5100, 5000, 4900,
    4800, 4700, 4600, 4500, 4400, 4300, 4200, 4100, 4000, 3800,
    /* 1660.0 thru 1699.0 */
    3700, 3600, 3500, 3400, 3300, 3200, 3100, 3000, 2800, 2700,
    2600, 2500, 2400, 2300, 2200, 2100, 2000, 1900, 1800, 1700,
    1600, 1500, 1400, 1400, 1300, 1200, 1200, 1100, 1100, 1000,
    1000, 1000, 900, 900, 900, 900, 900, 900, 900, 900,
    /* 1700.0 thru 1739.0 */
    900, 900, 900, 900, 900, 900, 900, 900, 1000, 1000,
    1000, 1000, 1000, 1000, 1000, 1000, 1000, 1100, 1100, 1100,
    1100, 1100, 1100, 1100, 1100, 1100, 1100, 1100, 1100, 1100,
    1100, 1100, 1100, 1100, 1200, 1200, 1200, 1200, 1200, 1200,
    /* 1740.0 thru 1779.0 */
    1200, 1200, 1200, 1200, 1300, 1300, 1300, 1300, 1300, 1300,
    1300, 1400, 1400, 1400, 1400, 1400, 1400, 1400, 1500, 1500,
    1500, 1500, 1500, 1500, 1500, 1600, 1600, 1600, 1600, 1600,
    1600, 1600, 1600, 1600, 1600, 1700, 1700, 1700, 1700, 1700,
    /* 1780.0 thru 1799.0 */
    1700, 1700, 1700, 1700, 1700, 1700, 1700, 1700, 1700, 1700,
    1700, 1700, 1600, 1600, 1600, 1600, 1500, 1500, 1400, 1400,
    /* 1800.0 thru 1819.0 */
    1370, 1340, 1310, 1290, 1270, 1260, 1250, 1250, 1250, 1250,
    1250, 1250, 1250, 1250, 1250, 1250, 1250, 1240, 1230, 1220,
    /* 1820.0 thru 1859.0 */
    1200, 1170, 1140, 1110, 1060, 1020, 960, 910, 860, 800,
    750, 700, 660, 630, 600, 580, 570, 560, 560, 560,
    570, 580, 590, 610, 620, 630, 650, 660, 680, 690,
    710, 720, 730, 740, 750, 760, 770, 770, 780, 780,
    /* 1860.0 thru 1899.0 */
    788, 782, 754, 697, 640, 602, 541, 410, 292, 182,
    161, 10, -102, -128, -269, -324, -364, -454, -471, -511,
    -540, -542, -520, -546, -546, -579, -563, -564, -580, -566,
    -587, -601, -619, -664, -644, -647, -609, -576, -466, -374,
    /* 1900.0 thru 1939.0 */
    -272, -154, -2, 124, 264, 386, 537, 614, 775, 913,
    1046, 1153, 1336, 1465, 1601, 1720, 1824, 1906, 2025, 2095,
    2116, 2225, 2241, 2303, 2349, 2362, 2386, 2449, 2434, 2408,
    2402, 2400, 2387, 2395, 2386, 2393, 2373, 2392, 2396, 2402,
    /* 1940.0 thru 1979.0 */
    2433, 2483, 2530, 2570, 2624, 2677, 2728, 2778, 2825, 2871,
    2915, 2957, 2997, 3036, 3072, 3107, 3135, 3168, 3218, 3268,
    3315, 3359, 3400, 3447, 3503, 3573, 3654, 3743, 3829, 3920,
    4018, 4117, 4223, 4337, 4449, 4548, 4646, 4752, 4853, 4959,
    /* 1980.0 thru 1999.0 */
    5054, 5138, 5217, 5296, 5379, 5434, 5487, 5532, 5582, 5630,
    5686, 5757, 5831, 5912, 5998, 6078, 6163, 6230, 6297, 6347,
    /* 2000.0 thru 2002.0 */
    6383, 6409, 6430, 
    /* Extrapolated values, 2003 - 2014 */
    6447, 6580, 6600, 6700, 6800, 6800, 6900,
    7000, 7000, 7100, 7200, 7300,
  };
  /* Table for -500 through 1600, from Stephenson & Morrison (1995).
   *
   * The first value for -550 has been added from Borkowski
   * in order to make this table fit with the Borkowski formula
   * for times before -550.
   */
  static final int TAB2_START = (-500);
  static final int TAB2_END = 1600;
  static final int TAB2_SIZ = 43;
  static short dt2[] = {
    /* -500  -450  -400  -350  -300  -250  -200  -150  -100   -50*/
    16800,16000,15300,14600,14000,13400,12800,12200,11600,11100,
    /*    0    50   100   150   200   250   300   350   400   450*/
    10600,10100, 9600, 9100, 8600, 8200, 7700, 7200, 6700, 6200,
    /*  500   550   600   650   700   750   800   850   900   950*/
    5700, 5200, 4700, 4300, 3800, 3400, 3000, 2600, 2200, 1900,
    /* 1000  1050  1100  1150  1200  1250  1300  1350  1400  1450*/
    1600, 1350, 1100,  900,  750,  600,  470,  380,  300,  230,
    /* 1500  1550  1600 */
    180,  140,  110,
  };

  /* Precession of the equinox and ecliptic
   * from epoch Julian date J to or from J2000.0
   *
   * Program by Steve Moshier.
   * Changes in program structure by Dieter Koch.
   *
   * #define PREC_WILLIAMS_1994 1
   * James G. Williams, "Contributions to the Earth's obliquity rate,
   * precession, and nutation,"  Astron. J. 108, 711-724 (1994).
   *
   * #define PREC_SIMON_1994 0
   * J. L. Simon, P. Bretagnon, J. Chapront, M. Chapront-Touze', G. Francou,
   * and J. Laskar, "Numerical Expressions for precession formulae and
   * mean elements for the Moon and the planets," Astronomy and Astrophysics
   * 282, 663-683 (1994).  
   *
   * #define PREC_IAU_1976 0
   * IAU Coefficients are from:
   * J. H. Lieske, T. Lederle, W. Fricke, and B. Morando,
   * "Expressions for the Precession Quantities Based upon the IAU
   * (1976) System of Astronomical Constants,"  Astronomy and
   * Astrophysics 58, 1-16 (1977).
   *
   * #define PREC_LASKAR_1986 0
   * Newer formulas that cover a much longer time span are from:
   * J. Laskar, "Secular terms of classical planetary theories
   * using the results of general theory," Astronomy and Astrophysics
   * 157, 59070 (1986).
   *
   * See also:
   * P. Bretagnon and G. Francou, "Planetary theories in rectangular
   * and spherical variables. VSOP87 solutions," Astronomy and
   * Astrophysics 202, 309-315 (1988).
   *
   * Laskar's expansions are said by Bretagnon and Francou
   * to have "a precision of about 1" over 10000 years before
   * and after J2000.0 in so far as the precession constants p^0_A
   * and epsilon^0_A are perfectly known."
   *
   * Bretagnon and Francou's expansions for the node and inclination
   * of the ecliptic were derived from Laskar's data but were truncated
   * after the term in T**6. I have recomputed these expansions from
   * Laskar's data, retaining powers up to T**10 in the result.
   *
   * The following table indicates the differences between the result
   * of the IAU formula and Laskar's formula using four different test
   * vectors, checking at J2000 plus and minus the indicated number
   * of years.
   *
   *   Years       Arc
   * from J2000  Seconds
   * ----------  -------
   *        0	  0
   *      100	.006	
   *      200     .006
   *      500     .015
   *     1000     .28
   *     2000    6.4
   *     3000   38.
   *    10000 9400.
   */
  /* In WILLIAMS and SIMON, Laskar's terms of order higher than t^4
     have been retained, because Simon et al mention that the solution
     is the same except for the lower order terms.  */

  /*#if PREC_WILLIAMS_1994 */
  static double pAcof[] = {
    -8.66e-10, -4.759e-8, 2.424e-7, 1.3095e-5, 1.7451e-4, -1.8055e-3,
    -0.235316, 0.076, 110.5407, 50287.70000 };
  static double nodecof[] = {
    6.6402e-16, -2.69151e-15, -1.547021e-12, 7.521313e-12, 1.9e-10, 
    -3.54e-9, -1.8103e-7,  1.26e-7,  7.436169e-5,
    -0.04207794833,  3.052115282424};
  static double inclcof[] = {
    1.2147e-16, 7.3759e-17, -8.26287e-14, 2.503410e-13, 2.4650839e-11, 
    -5.4000441e-11, 1.32115526e-9, -6.012e-7, -1.62442e-5,
    0.00227850649, 0.0 };
  /*#endif*/

  /*#if PREC_SIMON_1994*/
  /* Precession coefficients from Simon et al: */
  /*
  static double pAcof[] = {
    -8.66e-10, -4.759e-8, 2.424e-7, 1.3095e-5, 1.7451e-4, -1.8055e-3,
    -0.235316, 0.07732, 111.2022, 50288.200 };
  static double nodecof[] = {
    6.6402e-16, -2.69151e-15, -1.547021e-12, 7.521313e-12, 1.9e-10, 
    -3.54e-9, -1.8103e-7, 2.579e-8, 7.4379679e-5,
    -0.0420782900, 3.0521126906};
  static double inclcof[] = {
    1.2147e-16, 7.3759e-17, -8.26287e-14, 2.503410e-13, 2.4650839e-11, 
    -5.4000441e-11, 1.32115526e-9, -5.99908e-7, -1.624383e-5,
    0.002278492868, 0.0 };
  */
  /*#endif*/

  /*#if PREC_LASKAR_1986*/
  /* Precession coefficients taken from Laskar's paper: */
  /*
  static double pAcof[] = {
    -8.66e-10, -4.759e-8, 2.424e-7, 1.3095e-5, 1.7451e-4, -1.8055e-3,
    -0.235316, 0.07732, 111.1971, 50290.966 };
  */
  /* Node and inclination of the earth's orbit computed from
   * Laskar's data as done in Bretagnon and Francou's paper.
   * Units are radians.
   */
  /*
  static double nodecof[] = {
    6.6402e-16, -2.69151e-15, -1.547021e-12, 7.521313e-12, 6.3190131e-10, 
    -3.48388152e-9, -1.813065896e-7, 2.75036225e-8, 7.4394531426e-5,
    -0.042078604317, 3.052112654975 };
  static double inclcof[] = {
    1.2147e-16, 7.3759e-17, -8.26287e-14, 2.503410e-13, 2.4650839e-11, 
    -5.4000441e-11, 1.32115526e-9, -5.998737027e-7, -1.6242797091e-5,
    0.002278495537, 0.0 };
  */
  /*#endif*/

  /* Subroutine arguments:
   *
   * R = rectangular equatorial coordinate vector to be precessed.
   *     The result is written back into the input vector.
   * J = Julian date
   * direction =
   *      Precess from J to J2000: direction = 1
   *      Precess from J2000 to J: direction = -1
   * Note that if you want to precess from J1 to J2, you would
   * first go from J1 to J2000, then call the program again
   * to go from J2000 to J2.
   */
  static int swi_precess(double R[], int RIndex, double J, int direction )
  {
//    double sinth, costh, sinZ, cosZ, sinz, cosz;
    double eps, sineps, coseps;
    double A, B, T,  z, pA, W;//,Z, TH;
    double x[];
    x = new double[3];
    double p[];
    int i;
    if( J == SwissEphemeris.J2000 ) 
      return(0);
    /* Each precession angle is specified by a polynomial in
     * T = Julian centuries from J2000.0.  See AA page B18.
     */
    T = (J - SwissEphemeris.J2000)/36525.0;
    /* Implementation by elementary rotations using Laskar's expansions.
     * First rotate about the x axis from the initial equator
     * to the ecliptic. (The input is equatorial.)
     */
    if( direction == 1 ) 
      eps = swi_epsiln(J); /* To J2000 */
    else 
      eps = swi_epsiln(SwissEphemeris.J2000); /* From J2000 */
    sineps = Math.sin(eps);
    coseps = Math.cos(eps);
    x[0] = R[RIndex+0];
    z = coseps*R[RIndex+1] + sineps*R[RIndex+2];
    x[2] = -sineps*R[RIndex+1] + coseps*R[RIndex+2];
    x[1] = z;
    /* Precession in longitude */
    T /= 10.0; /* thousands of years */
    p = pAcof;
    /*pA = *p++;*/
    pA = p[0];
    for( i=0; i<9; i++ ) 
      /*pA = pA * T + *p++;*/
      pA = pA * T + p[1+i];
    pA *= SwissEphemeris.DEGTORAD/3600 * T;
    /* Node of the moving ecliptic on the J2000 ecliptic.
     */
    p = nodecof;
    /*W = *p++;*/
    W = p[0];
    for( i=0; i<10; i++ ) 
      /*W = W * T + *p++;*/
      W = W * T + p[1+i];
    /* Rotate about z axis to the node.
     */
    if( direction == 1 ) 
      z = W + pA;
    else 
      z = W;
    B = Math.cos(z);
    A = Math.sin(z);
    z = B * x[0] + A * x[1];
    x[1] = -A * x[0] + B * x[1];
    x[0] = z;
    /* Rotate about new x axis by the inclination of the moving
     * ecliptic on the J2000 ecliptic.
     */
    p = inclcof;
    /*z = *p++;*/
    z = p[0];
    for( i=0; i<10; i++ ) 
      /*z = z * T + *p++;*/
      z = z * T + p[1+i];
    if( direction == 1 ) 
      z = -z;
    B = Math.cos(z);
    A = Math.sin(z);
    z = B * x[1] + A * x[2];
    x[2] = -A * x[1] + B * x[2];
    x[1] = z;
    /* Rotate about new z axis back from the node.
     */
    if( direction == 1 ) 
      z = -W;
    else 
      z = -W - pA;
    B = Math.cos(z);
    A = Math.sin(z);
    z = B * x[0] + A * x[1];
    x[1] = -A * x[0] + B * x[1];
    x[0] = z;
    /* Rotate about x axis to final equator.
     */
    if( direction == 1 ) 
      eps = swi_epsiln(SwissEphemeris.J2000);
    else 
      eps = swi_epsiln(J);
    sineps = Math.sin(eps);
    coseps = Math.cos(eps);
    z = coseps * x[1] - sineps * x[2];
    x[2] = sineps * x[1] + coseps * x[2];
    x[1] = z;
    for( i=0; i<3; i++ ) 
      R[RIndex+i] = x[i];
    return(0);
  }

  static int swi_nutation(double J, double nutlo[])
  {
    final int ENDMARK = -99;

    short nt[] = {
      /* LS and OC are units of 0.0001"
       *LS2 and OC2 are units of 0.00001"
       *MM,MS,FF,DD,OM, LS, LS2,OC, OC2 */
      0, 0, 0, 0, 2, 2062, 2,-895, 5,
      -2, 0, 2, 0, 1, 46, 0,-24, 0,
      2, 0,-2, 0, 0, 11, 0, 0, 0,
      -2, 0, 2, 0, 2,-3, 0, 1, 0,
      1,-1, 0,-1, 0,-3, 0, 0, 0,
      0,-2, 2,-2, 1,-2, 0, 1, 0,
      2, 0,-2, 0, 1, 1, 0, 0, 0,
      0, 0, 2,-2, 2,-13187,-16, 5736,-31,
      0, 1, 0, 0, 0, 1426,-34, 54,-1,
      0, 1, 2,-2, 2,-517, 12, 224,-6,
      0,-1, 2,-2, 2, 217,-5,-95, 3,
      0, 0, 2,-2, 1, 129, 1,-70, 0,
      2, 0, 0,-2, 0, 48, 0, 1, 0,
      0, 0, 2,-2, 0,-22, 0, 0, 0,
      0, 2, 0, 0, 0, 17,-1, 0, 0,
      0, 1, 0, 0, 1,-15, 0, 9, 0,
      0, 2, 2,-2, 2,-16, 1, 7, 0,
      0,-1, 0, 0, 1,-12, 0, 6, 0,
      -2, 0, 0, 2, 1,-6, 0, 3, 0,
      0,-1, 2,-2, 1,-5, 0, 3, 0,
      2, 0, 0,-2, 1, 4, 0,-2, 0,
      0, 1, 2,-2, 1, 4, 0,-2, 0,
      1, 0, 0,-1, 0,-4, 0, 0, 0,
      2, 1, 0,-2, 0, 1, 0, 0, 0,
      0, 0,-2, 2, 1, 1, 0, 0, 0,
      0, 1,-2, 2, 0,-1, 0, 0, 0,
      0, 1, 0, 0, 2, 1, 0, 0, 0,
      -1, 0, 0, 1, 1, 1, 0, 0, 0,
      0, 1, 2,-2, 0,-1, 0, 0, 0,
      0, 0, 2, 0, 2,-2274,-2, 977,-5,
      1, 0, 0, 0, 0, 712, 1,-7, 0,
      0, 0, 2, 0, 1,-386,-4, 200, 0,
      1, 0, 2, 0, 2,-301, 0, 129,-1,
      1, 0, 0,-2, 0,-158, 0,-1, 0,
      -1, 0, 2, 0, 2, 123, 0,-53, 0,
      0, 0, 0, 2, 0, 63, 0,-2, 0,
      1, 0, 0, 0, 1, 63, 1,-33, 0,
      -1, 0, 0, 0, 1,-58,-1, 32, 0,
      -1, 0, 2, 2, 2,-59, 0, 26, 0,
      1, 0, 2, 0, 1,-51, 0, 27, 0,
      0, 0, 2, 2, 2,-38, 0, 16, 0,
      2, 0, 0, 0, 0, 29, 0,-1, 0,
      1, 0, 2,-2, 2, 29, 0,-12, 0,
      2, 0, 2, 0, 2,-31, 0, 13, 0,
      0, 0, 2, 0, 0, 26, 0,-1, 0,
      -1, 0, 2, 0, 1, 21, 0,-10, 0,
      -1, 0, 0, 2, 1, 16, 0,-8, 0,
      1, 0, 0,-2, 1,-13, 0, 7, 0,
      -1, 0, 2, 2, 1,-10, 0, 5, 0,
      1, 1, 0,-2, 0,-7, 0, 0, 0,
      0, 1, 2, 0, 2, 7, 0,-3, 0,
      0,-1, 2, 0, 2,-7, 0, 3, 0,
      1, 0, 2, 2, 2,-8, 0, 3, 0,
      1, 0, 0, 2, 0, 6, 0, 0, 0,
      2, 0, 2,-2, 2, 6, 0,-3, 0,
      0, 0, 0, 2, 1,-6, 0, 3, 0,
      0, 0, 2, 2, 1,-7, 0, 3, 0,
      1, 0, 2,-2, 1, 6, 0,-3, 0,
      0, 0, 0,-2, 1,-5, 0, 3, 0,
      1,-1, 0, 0, 0, 5, 0, 0, 0,
      2, 0, 2, 0, 1,-5, 0, 3, 0, 
      0, 1, 0,-2, 0,-4, 0, 0, 0,
      1, 0,-2, 0, 0, 4, 0, 0, 0,
      0, 0, 0, 1, 0,-4, 0, 0, 0,
      1, 1, 0, 0, 0,-3, 0, 0, 0,
      1, 0, 2, 0, 0, 3, 0, 0, 0,
      1,-1, 2, 0, 2,-3, 0, 1, 0,
      -1,-1, 2, 2, 2,-3, 0, 1, 0,
      -2, 0, 0, 0, 1,-2, 0, 1, 0,
      3, 0, 2, 0, 2,-3, 0, 1, 0,
      0,-1, 2, 2, 2,-3, 0, 1, 0,
      1, 1, 2, 0, 2, 2, 0,-1, 0,
      -1, 0, 2,-2, 1,-2, 0, 1, 0,
      2, 0, 0, 0, 1, 2, 0,-1, 0,
      1, 0, 0, 0, 2,-2, 0, 1, 0,
      3, 0, 0, 0, 0, 2, 0, 0, 0,
      0, 0, 2, 1, 2, 2, 0,-1, 0,
      -1, 0, 0, 0, 2, 1, 0,-1, 0,
      1, 0, 0,-4, 0,-1, 0, 0, 0,
      -2, 0, 2, 2, 2, 1, 0,-1, 0,
      -1, 0, 2, 4, 2,-2, 0, 1, 0,
      2, 0, 0,-4, 0,-1, 0, 0, 0,
      1, 1, 2,-2, 2, 1, 0,-1, 0,
      1, 0, 2, 2, 1,-1, 0, 1, 0,
      -2, 0, 2, 4, 2,-1, 0, 1, 0,
      -1, 0, 4, 0, 2, 1, 0, 0, 0,
      1,-1, 0,-2, 0, 1, 0, 0, 0,
      2, 0, 2,-2, 1, 1, 0,-1, 0,
      2, 0, 2, 2, 2,-1, 0, 0, 0,
      1, 0, 0, 2, 1,-1, 0, 0, 0,
      0, 0, 4,-2, 2, 1, 0, 0, 0,
      3, 0, 2,-2, 2, 1, 0, 0, 0,
      1, 0, 2,-2, 0,-1, 0, 0, 0,
      0, 1, 2, 0, 1, 1, 0, 0, 0,
      -1,-1, 0, 2, 1, 1, 0, 0, 0,
      0, 0,-2, 0, 1,-1, 0, 0, 0,
      0, 0, 2,-1, 2,-1, 0, 0, 0,
      0, 1, 0, 2, 0,-1, 0, 0, 0,
      1, 0,-2,-2, 0,-1, 0, 0, 0,
      0,-1, 2, 0, 1,-1, 0, 0, 0,
      1, 1, 0,-2, 1,-1, 0, 0, 0,
      1, 0,-2, 2, 0,-1, 0, 0, 0,
      2, 0, 0, 2, 0, 1, 0, 0, 0,
      0, 0, 2, 4, 2,-1, 0, 0, 0,
      0, 1, 0, 1, 0, 1, 0, 0, 0,
      /* corrections to IAU 1980 nutation series by Herring 1987
 *             in 0.00001" !!!
 *              LS      OC      */
      101, 0, 0, 0, 1,-725, 0, 213, 0,
      101, 1, 0, 0, 0, 523, 0, 208, 0,
      101, 0, 2,-2, 2, 102, 0, -41, 0,
      101, 0, 2, 0, 2, -81, 0,  32, 0,
      /*              LC      OS !!!  */
      102, 0, 0, 0, 1, 417, 0, 224, 0,
      102, 1, 0, 0, 0,  61, 0, -24, 0,
      102, 0, 2,-2, 2,-118, 0, -47, 0,
      ENDMARK,
    };

    /* arrays to hold sines and cosines of multiple angles */
    double ss[][];
    ss = new double[5][8];
    for (int i = 0; i < 5; i++) {
      ss[i] = new double[8];
    }
    double cc[][];
    cc = new double[5][8];
    for (int i = 0; i < 5; i++) {
      cc[i] = new double[8];
    }
    double arg;
    double args[];
    args = new double[5];
    double f, g, T, T2;
    double MM, MS, FF, DD, OM;
    double cu, su, cv, sv, sw, s;
    double C, D;
    int i, j, k, k1, m, n;
    int ns[];
    ns = new int[5];
    /*short *p;*/
    int ntIndex;
    /* Julian centuries from 2000 January 1.5,
     * barycentric dynamical time
     */
    T = (J - 2451545.0) / 36525.0;
    T2 = T * T;
    /* Fundamental arguments in the FK5 reference system.
     * The coefficients, originally given to 0.001",
     * are converted here to degrees.
   */
  /* longitude of the mean ascending node of the lunar orbit
   * on the ecliptic, measured from the mean equinox of date
   */
    OM = -6962890.539 * T + 450160.280 + (0.008 * T + 7.455) * T2;
    OM = swe_degnorm(OM/3600) * SwissEphemeris.DEGTORAD;
    /* mean longitude of the Sun minus the
     * mean longitude of the Sun's perigee
     */
    MS = 129596581.224 * T + 1287099.804 - (0.012 * T + 0.577) * T2;
    MS = swe_degnorm(MS/3600) * SwissEphemeris.DEGTORAD;
    /* mean longitude of the Moon minus the
     * mean longitude of the Moon's perigee
     */
    MM = 1717915922.633 * T + 485866.733 + (0.064 * T + 31.310) * T2;
    MM = swe_degnorm(MM/3600) * SwissEphemeris.DEGTORAD;
    /* mean longitude of the Moon minus the
     * mean longitude of the Moon's node
     */
    FF = 1739527263.137 * T + 335778.877 + (0.011 * T - 13.257) * T2;
    FF = swe_degnorm(FF/3600) * SwissEphemeris.DEGTORAD;
    /* mean elongation of the Moon from the Sun.
     */
    DD = 1602961601.328 * T + 1072261.307 + (0.019 * T - 6.891) * T2;
    DD = swe_degnorm(DD/3600) * SwissEphemeris.DEGTORAD;
    args[0] = MM;
    ns[0] = 3;
    args[1] = MS;
    ns[1] = 2;
    args[2] = FF;
    ns[2] = 4;
    args[3] = DD;
    ns[3] = 4;
    args[4] = OM;
    ns[4] = 2;
    /* Calculate sin( i*MM ), etc. for needed multiple angles
     */
    for (k = 0; k <= 4; k++) {
      arg = args[k];
      n = ns[k];
      su = Math.sin(arg);
      cu = Math.cos(arg);
      ss[k][0] = su;			/* sin(L) */
      cc[k][0] = cu;			/* cos(L) */
      sv = 2.0*su*cu;
      cv = cu*cu - su*su;
      ss[k][1] = sv;			/* sin(2L) */
      cc[k][1] = cv;
      for( i=2; i<n; i++ ) {
	s =  su*cv + cu*sv;
	cv = cu*cv - su*sv;
	sv = s;
	ss[k][i] = sv;		/* sin( i+1 L ) */
	cc[k][i] = cv;
      }
    }
    /* first terms, not in table: */
    C = (-0.01742*T - 17.1996)*ss[4][0];	/* sin(OM) */
    D = ( 0.00089*T +  9.2025)*cc[4][0];	/* cos(OM) */
    for(ntIndex = 0; nt[ntIndex] != ENDMARK; ntIndex += 9) {
      /* argument of sine and cosine */
      k1 = 0;
      cv = 0.0;
      sv = 0.0;
      for( m=0; m<5; m++ ) {
	j = nt[ntIndex+m];
	if (j > 100) 
	  j = 0; /* p[0] is a flag */
	if( j != 0 ) {
	  k = j;
	  if( j < 0 ) 
	    k = -k;
	  su = ss[m][k-1]; /* sin(k*angle) */
	  if( j < 0 ) 
	    su = -su;
	  cu = cc[m][k-1];
	  if( k1 == 0 ) { /* set first angle */
	    sv = su;
	    cv = cu;
	    k1 = 1;
	  }
	  else {		/* combine angles */
	    sw = su*cv + cu*sv;
	    cv = cu*cv - su*sv;
	    sv = sw;
	  }
	}
      }
      /* longitude coefficient, in 0.0001" */
      f  = nt[ntIndex+5] * 0.0001;
      if( nt[ntIndex+6] != 0 ) 
	f += 0.00001 * T * nt[ntIndex+6];
      /* obliquity coefficient, in 0.0001" */
      g = nt[ntIndex+7] * 0.0001;
      if( nt[ntIndex+8] != 0 ) 
	g += 0.00001 * T * nt[ntIndex+8];
      if (nt[ntIndex] >= 100) { 	/* coefficients in 0.00001" */
	f *= 0.1;
	g *= 0.1;
      }
      /* accumulate the terms */
      if (nt[ntIndex] != 102) {
	C += f * sv;
	D += g * cv;
      }
      else { 		/* cos for nutl and sin for nuto */
	C += f * cv;
	D += g * sv;
      }
      /*
	if (i >= 105) {
	printf("%4.10f, %4.10f\n",f*sv,g*cv);
	}
      */
    }
    /*
      printf("%4.10f, %4.10f, %4.10f, %4.10f\n",MS*RADTODEG,FF*RADTODEG,DD*RADTODEG,OM*RADTODEG);
  printf( "nutation: in longitude %.9f\", in obliquity %.9f\"\n", C, D );
  */
  /* Save answers, expressed in radians */
    nutlo[0] = SwissEphemeris.DEGTORAD * C / 3600.0;
    nutlo[1] = SwissEphemeris.DEGTORAD * D / 3600.0;
    return(0);
  }

  /* returns DeltaT (ET - UT) in days
   * double tjd 	= 	julian day in UT
   */
  static double swe_deltat(double tjd)
  {
    class GotoException extends Exception {
    };
    double ans = 0, ans2, ans3;
    double p, B, B2, Y, dd;
    int d[];
    d = new int[6];
    int i, iy, k;
    /* read additional values from swedelta.txt */
    int tabsiz = init_dt();
    int tabend = TABSTART + tabsiz - 1;
    Y = 2000.0 + (tjd - SwissEphemeris.J2000)/365.25;
    /* before -500:
     * formula by Stephenson (1997; p. 508) but adjusted to fit the starting
     * point of table dt2 (Stephenson 1997). */
    if( Y < TAB2_START ) {
      B = (Y - 1735) * 0.01;
      ans = -20 + 35 * B * B;
      ans = adjust_for_tidacc(ans, Y);
      /* transition from formula to table over 100 years */
      if (Y >= TAB2_START - 100) {
	/* starting value of table dt2: */
	ans2 = adjust_for_tidacc(dt2[0], TAB2_START);
	/* value of formula at epoch TAB2_START */
	B = (TAB2_START - 1735) * 0.01;
	ans3 = -20 + 35 * B * B;
	ans3 = adjust_for_tidacc(ans3, Y);
	dd = ans3 - ans2;
	B = (Y - (TAB2_START - 100)) * 0.01;
	/* fit to starting point of table dt2. */
	ans = ans - dd * B;
      }
    }
    /* between -500 and 1600: 
     * linear interpolation between values of table dt2 (Stephenson 1997) */
    if (Y >= TAB2_START && Y < TAB2_END) { 
      p = Math.floor(Y);
      iy = (int) ((p - TAB2_START) / 50.0);
      dd = (Y - (TAB2_START + 50 * iy)) / 50.0;
      ans = dt2[iy] + (dt2[iy+1] - dt2[iy]) * dd;
      /* correction for tidal acceleration used by our ephemeris */
      ans = adjust_for_tidacc(ans, Y);
    }
    /* between 1600 and 1620:
     * linear interpolation between 
     * end of table dt2 and start of table dt */
    if (Y >= TAB2_END && Y < TABSTART) { 
      B = TABSTART - TAB2_END;
      iy = (TAB2_END - TAB2_START) / 50;
      dd = (Y - TAB2_END) / B;
      ans = dt2[iy] + dd * (dt[0] / 100.0 - dt2[iy]);
      ans = adjust_for_tidacc(ans, Y);
    }
    /* 1620 - today + a few years (tabend):
     * Besselian interpolation from tabulated values in table dt.
     * See AA page K11.
     */
    try {
      if (Y >= TABSTART && Y <= tabend) {
	/* Index into the table.
	 */
	p = Math.floor(Y);
	iy = (int) (p - TABSTART);
	/* Zeroth order estimate is value at start of year
	 */
	ans = dt[iy];
	k = iy + 1;
	if( k >= tabsiz )
	  /*goto done; /* No data, can't go on. */
	  throw new GotoException();
	/* The fraction of tabulation interval
	 */
	p = Y - p;
	/* First order interpolated value
	 */
	ans += p*(dt[k] - dt[iy]);
	if( (iy-1 < 0) || (iy+2 >= tabsiz) )
	  /*goto done; /* can't do second differences */
	  throw new GotoException();
	/* Make table of first differences
	 */
	k = iy - 2;
	for( i=0; i<5; i++ ) {
	  if( (k < 0) || (k+1 >= tabsiz) ) 
	    d[i] = 0;
	  else
	    d[i] = dt[k+1] - dt[k];
	  k += 1;
	}
	/* Compute second differences
	 */
	for( i=0; i<4; i++ )
	  d[i] = d[i+1] - d[i];
	B = 0.25*p*(p-1.0);
	ans += B*(d[1] + d[2]);
	if( iy+2 >= tabsiz )
	  /*goto done;*/
	  throw new GotoException();
	/* Compute third differences
	 */
	for( i=0; i<3; i++ )
	  d[i] = d[i+1] - d[i];
	B = 2.0*B/3.0;
	ans += (p-0.5)*B*d[1];
	if( (iy-2 < 0) || (iy+3 > tabsiz) )
	  /*goto done;*/
	  throw new GotoException();
	/* Compute fourth differences
	 */
	for( i=0; i<2; i++ )
	  d[i] = d[i+1] - d[i];
	B = 0.125*B*(p+1.0)*(p-2.0);
	ans += B*(d[0] + d[1]);
      }
    }
    catch (GotoException e) {
    }
    finally {
      ans *= 0.01;
      ans = adjust_for_tidacc(ans, Y);
    }
    /* today - : 
     * Formula Stephenson (1997; p. 507),
     * with modification to avoid jump at end of AA table,
     * similar to what Meeus 1998 had suggested.
     * Slow transition within 100 years.
     */
    if (Y > tabend) {
      B = 0.01 * (Y - 1820);
      ans = -20 + 31 * B * B;
      /* slow transition from tabulated values to Stephenson formula: */
      if (Y <= tabend+100) {
	B2 = 0.01 * (tabend - 1820);
	ans2 = -20 + 31 * B2 * B2;
	ans3 = dt[tabsiz-1] * 0.01;
	dd = (ans2 - ans3);
	ans += dd * (Y - (tabend + 100)) * 0.01;
      }
    }
    return ans / 86400.0;
  }

  /**
   * <code>init_dt</code> method
   * read delta t values from external file.
   * record structure: year(whitespace)delta_t in 0.01 sec.
   *
   * @return an <code>int</code> value
   */
  static int init_dt()
  {
    return TABSIZ; 
  }

  /**
   * <code>adjust_for_tidacc</code> method
   * Astronomical Almanac table is corrected by adding the expression
   *     -0.000091 (ndot + 26)(year-1955)^2  seconds
   * to entries prior to 1955 (AA page K8), where ndot is the secular
   * tidal term in the mean motion of the Moon.
   *
   * Entries after 1955 are referred to atomic time standards and
   * are not affected by errors in Lunar or planetary theory.
   *
   * @param ans a <code>double</code> value
   * @param Y a <code>double</code> value
   * @return a <code>double</code> value
   */
  static double adjust_for_tidacc(double ans, double Y)
  {
    double B;
    if( Y < 1955.0 ) {
      B = (Y - 1955.0);
      ans += -0.000091 * (tid_acc + 26.0) * B * B;
    }
    return ans;
  }

  public int swe_time_equ(double tjd, double te, String serr) {
    return(0);
  }
  
  /**
   * <code>swe_sidtime0</code> method
   * Apparent Sidereal Time at Greenwich with equation of the equinoxes
   * AA page B6
   *
   * returns sidereal time in hours.
   *
   * Caution. At epoch J2000.0, the 16 decimal precision
   * of IEEE double precision numbers
   * limits time resolution measured by Julian date
   * to approximately 24 microseconds.
   * 
   * program returns sidereal hours since sidereal midnight 
   * tjd 		julian day UT
   * eps 		obliquity of ecliptic, degrees 
   * nut 		nutation, degrees 
   *
   * @param tjd_ut a <code>double</code> value
   * @param eps a <code>double</code> value
   * @param nut a <code>double</code> value
   * @return a <code>double</code> value
   */
  public static double swe_sidtime0(double tjd_ut, double eps, double nut) {
    double jd0;    	/* Julian day at midnight Universal Time */
    double secs;   	/* Time of day, UT seconds since UT midnight */
    double eqeq, jd, T0, msday;
    double gmst;
    /* Julian day at given UT */
    jd = tjd_ut;
    jd0 = Math.floor(jd);
    secs = tjd_ut - jd0;
    if( secs < 0.5 ) {
      jd0 -= 0.5;
      secs += 0.5;
    } else {
      jd0 += 0.5;
      secs -= 0.5;
    }
    secs *= 86400.0;
    /* Same but at 0h Universal Time of date */
    T0 = (jd0 - SwissEphemeris.J2000)/36525.0;
    eqeq = 240.0 * nut * Math.cos(eps * SwissEphemeris.DEGTORAD);
    /* Greenwich Mean Sidereal Time at 0h UT of date */
    gmst = (( -6.2e-6*T0 + 9.3104e-2)*T0 + 8640184.812866)*T0 + 24110.54841;
    /* mean solar days per sidereal day at date T0, = 1.00273790934 in 1986 */
    msday =
      1.0 + ((-1.86e-5*T0 + 0.186208)*T0 + 8640184.812866)/(86400.*36525.);
    /* Local apparent sidereal time at given UT at Greenwich */
    gmst = gmst + msday*secs + eqeq  /* + 240.0*tlong */;
    /* Sidereal seconds modulo 1 sidereal day */
    gmst = gmst - 86400.0 * Math.floor( gmst/86400.0 );
    /* return in hours */
    gmst /= 3600;
    return gmst;
  }

  public double swe_sidtime(double tjd_ut) {
    return(0);
  }
  
  public void swe_cotrans(double xpo, double xpn, double eps) {
  }
  
  public void swe_cotrans_sp(double xpo, double xpn, double eps) {
  }
  
  public double swe_get_tid_acc() {
    return(0);
  }
  
  public void swe_set_tid_acc(double t_acc) {
  }
  
  /**
   * <code>swe_degnorm</code> method
   * Reduce x modulo 360 degrees
   *
   * @param x a <code>double</code> value
   * @return a <code>double</code> value
   */
  static public double swe_degnorm(double x) {
    double y;
    y = Math.IEEEremainder(x, 360.0);
    if (Math.abs(y) < 1e-13) y = 0;	/* Alois fix 11-dec-1999 */
    if( y < 0.0 ) y += 360.0;
    return(y);
  }
  
  /**
   * <code>swe_radnorm</code> method
   * Reduce x modulo TWOPI degrees
   *
   * @param x a <code>double</code> value
   * @return a <code>double</code> value
   */
  static public double swe_radnorm(double x) {
    double y;
    y = Math.IEEEremainder(x, TWOPI);
    if (Math.abs(y) < 1e-13) y = 0;	/* Alois fix 11-dec-1999 */
    if( y < 0.0 ) y += TWOPI;
    return(y);
  }
  
  public double swe_rad_midp(double x1, double x0) {
    return(0);
  }
  
  public double swe_deg_midp(double x1, double x0) {
    return(0);
  }

  public void swe_split_deg(double ddeg, int roundflag,
			    int ideg, int imin, int isec,
			    double dsecfr, int isgn) {
  }

  /*
   * conversion between ecliptical and equatorial cartesian coordinates
   * for ecl. to equ.  eps must be negative
   * for equ. to ecl.  eps must be positive
   */
  static void swi_coortrf(double xpo[], int xpoIndex,
		   double xpn[], int xpnIndex, double eps) 
  {
    double sineps, coseps;
    double x[];
    x = new double[3];
    sineps = Math.sin(eps);
    coseps = Math.cos(eps);
    x[0] = xpo[xpoIndex+0];
    x[1] = xpo[xpoIndex+1] * coseps + xpo[xpoIndex+2] * sineps;
    x[2] = -xpo[xpoIndex+1] * sineps + xpo[xpoIndex+2] * coseps;
    xpn[xpnIndex+0] = x[0];
    xpn[xpnIndex+1] = x[1]; 
    xpn[xpnIndex+2] = x[2];
  }

  /*
   * conversion between ecliptical and equatorial cartesian coordinates
   * sineps            sin(eps)
   * coseps            cos(eps)
   * for ecl. to equ.  sineps must be -sin(eps)
   */
  static void swi_coortrf2(double xpo[], int xpoIndex,
		    double xpn[], int xpnIndex,
		    double sineps, double coseps) 
  {
    double x[];
    x = new double[3];
    x[0] = xpo[xpoIndex+0];
    x[1] = xpo[xpoIndex+1] * coseps + xpo[xpoIndex+2] * sineps;
    x[2] = -xpo[xpoIndex+1] * sineps + xpo[xpoIndex+2] * coseps;
    xpn[xpnIndex+0] = x[0];
    xpn[xpnIndex+1] = x[1]; 
    xpn[xpnIndex+2] = x[2];
  }

  /* conversion of cartesian (x[3]) to polar coordinates (l[3]).
   * x = l is allowed.
   * if |x| = 0, then lon, lat and rad := 0.
   */
  static void swi_cartpol(double x[], int xIndex, double l[], int lIndex) 
  {
    double rxy;
    double ll[];
    ll = new double[3];
    if (x[xIndex+0] == 0 && x[xIndex+1] == 0 && x[xIndex+2] == 0) {
      l[lIndex+0] = l[lIndex+1] = l[lIndex+2] = 0;
      return;
    }
    rxy = x[xIndex+0]*x[xIndex+0] + x[xIndex+1]*x[xIndex+1];
    ll[2] = Math.sqrt(rxy + x[xIndex+2]*x[xIndex+2]);
    rxy = Math.sqrt(rxy);
    ll[0] = Math.atan2(x[xIndex+1], x[xIndex+0]);
    if (ll[0] < 0.0) ll[0] += TWOPI;
    ll[1] = Math.atan(x[xIndex+2] / rxy);
    l[lIndex+0] = ll[0];
    l[lIndex+1] = ll[1];
    l[lIndex+2] = ll[2];
  }

  /* conversion from polar (l[3]) to cartesian coordinates (x[3]).
   * x = l is allowed.
   */
  static void swi_polcart(double l[], int lIndex, double x[], int xIndex)
  {
    double xx[];
    xx = new double[3];
    double cosl1;
    cosl1 = Math.cos(l[lIndex+1]);
    xx[0] = l[lIndex+2] * cosl1 * Math.cos(l[lIndex+0]);
    xx[1] = l[lIndex+2] * cosl1 * Math.sin(l[lIndex+0]);
    xx[2] = l[lIndex+2] * Math.sin(l[lIndex+1]);
    x[xIndex+0] = xx[0];
    x[xIndex+1] = xx[1];
    x[xIndex+2] = xx[2];
  }

  /* conversion of position and speed. 
   * from cartesian (x[6]) to polar coordinates (l[6]). 
   * x = l is allowed.
   * if position is 0, function returns direction of 
   * motion.
   */
  static void swi_cartpol_sp(double x[], int xIndex, double l[], int lIndex)
  {
    double xx[], ll[];
    xx = new double[6];
    ll = new double[6];
    double rxy, coslon, sinlon, coslat, sinlat;
    /* zero position */
    if (x[xIndex+0] == 0 && x[xIndex+1] == 0 && x[xIndex+2] == 0) {
      l[lIndex+0] = l[lIndex+1] = l[lIndex+3] = l[lIndex+4] = 0;
      l[lIndex+5] = Math.sqrt(SwissEphemeris.square_sum(x, 3));
      swi_cartpol(x, 3, l, 0);
      l[lIndex+2] = 0;
      return;
    }
    /* zero speed */
    if (x[xIndex+3] == 0 && x[xIndex+4] == 0 && x[xIndex+5] == 0) {
      l[lIndex+3] = l[lIndex+4] = l[lIndex+5] = 0;
      swi_cartpol(x, 0, l, 0);
      return;
    }
    /* position */
    rxy = x[xIndex+0]*x[xIndex+0] + x[xIndex+1]*x[xIndex+1];
    ll[2] = Math.sqrt(rxy + x[xIndex+2]*x[xIndex+2]);
    rxy = Math.sqrt(rxy);
    ll[0] = Math.atan2(x[xIndex+1], x[xIndex+0]);
    if (ll[0] < 0.0) ll[0] += TWOPI;
    ll[1] = Math.atan(x[xIndex+2] / rxy);
    /* speed: 
     * 1. rotate coordinate system by longitude of position about z-axis, 
     *    so that new x-axis = position radius projected onto x-y-plane.
     *    in the new coordinate system 
     *    vy'/r = dlong/dt, where r = Math.sqrt(x^2 +y^2).
     * 2. rotate coordinate system by latitude about new y-axis.
     *    vz"/r = dlat/dt, where r = position radius.
     *    vx" = dr/dt
     */
    coslon = x[xIndex+0] / rxy; 		/* cos(l[0]); */
    sinlon = x[xIndex+1] / rxy; 		/* sin(l[0]); */
    coslat = rxy / ll[2];  	/* cos(l[1]); */
    sinlat = x[xIndex+2] / ll[2];	/* sin(ll[1]); */
    xx[3] = x[xIndex+3] * coslon + x[xIndex+4] * sinlon;
    xx[4] = -x[xIndex+3] * sinlon + x[xIndex+4] * coslon;
    l[lIndex+3] = xx[4] / rxy;  		/* speed in longitude */
    xx[4] = -sinlat * xx[3] + coslat * x[xIndex+5];
    xx[5] =  coslat * xx[3] + sinlat * x[xIndex+5];
    l[lIndex+4] = xx[4] / ll[2];  	/* speed in latitude */
    l[lIndex+5] = xx[5];  		/* speed in radius */
    l[lIndex+0] = ll[0];			/* return position */
    l[lIndex+1] = ll[1];
    l[lIndex+2] = ll[2];
  }

  /* conversion of position and speed 
   * from polar (l[6]) to cartesian coordinates (x[6]) 
   * x = l is allowed
   * explanation s. swi_cartpol_sp()
   */
  static void swi_polcart_sp(double l[], int lIndex, double x[], int xIndex)
  {
    double sinlon, coslon, sinlat, coslat;
    double xx[], rxy, rxyz;
    xx = new double[6];
    /* zero speed */
    if (l[lIndex+3] == 0 && l[lIndex+4] == 0 && l[lIndex+5] == 0) {
      x[xIndex+3] = x[xIndex+4] = x[xIndex+5] = 0;
      swi_polcart(l, 0, x, 0);
      return;
    }
    /* position */
    coslon = Math.cos(l[lIndex+0]);
    sinlon = Math.sin(l[lIndex+0]);
    coslat = Math.cos(l[lIndex+1]);
    sinlat = Math.sin(l[lIndex+1]);
    xx[0] = l[lIndex+2] * coslat * coslon;
    xx[1] = l[lIndex+2] * coslat * sinlon;
    xx[2] = l[lIndex+2] * sinlat;
    /* speed; explanation s. swi_cartpol_sp(),
       same method the other way round*/
    rxyz = l[lIndex+2];
    rxy = Math.sqrt(xx[0] * xx[0] + xx[1] * xx[1]);
    xx[5] = l[lIndex+5];
    xx[4] = l[lIndex+4] * rxyz;
    x[xIndex+5] = sinlat * xx[5] + coslat * xx[4];	/* speed z */
    xx[3] = coslat * xx[5] - sinlat * xx[4];
    xx[4] = l[lIndex+3] * rxy;
    x[xIndex+3] = coslon * xx[3] - sinlon * xx[4];	/* speed x */
    x[xIndex+4] = sinlon * xx[3] + coslon * xx[4];	/* speed y */
    x[xIndex+0] = xx[0];				/* return position */
    x[xIndex+1] = xx[1];
    x[xIndex+2] = xx[2];
  }

  static double swi_dot_prod_unit(double x[], double y[])
  {
    double dop = x[0]*y[0]+x[1]*y[1]+x[2]*y[2];
    double e1 = Math.sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2]);
    double e2 = Math.sqrt(y[0]*y[0]+y[1]*y[1]+y[2]*y[2]);
    dop /= e1;
    dop /= e2;
    if (dop > 1)
      dop = 1;
    if (dop < -1)
      dop = -1;
    return dop;
  }

  /* Obliquity of the ecliptic at Julian date J
   *
   * IAU Coefficients are from:
   * J. H. Lieske, T. Lederle, W. Fricke, and B. Morando,
   * "Expressions for the Precession Quantities Based upon the IAU
   * (1976) System of Astronomical Constants,"  Astronomy and Astrophysics
   * 58, 1-16 (1977).
   *
   * Before or after 200 years from J2000, the formula used is from:
   * J. Laskar, "Secular terms of classical planetary theories
   * using the results of general theory," Astronomy and Astrophysics
   * 157, 59070 (1986).
   *
   *  See precess and page B18 of the Astronomical Almanac.
   */
  static double swi_epsiln(double J) 
  {
    double T, eps;
    T = (J - 2451545.0)/36525.0;
    /* This expansion is from the AA.
     * Note the official 1976 IAU number is 23d 26' 21.448", but
     * the JPL numerical integration found 21.4119".
     */
    if( Math.abs(T) < 2.0 )
      eps =
	(((1.813e-3*T-5.9e-4)*T-46.8150)*T+84381.448)
	*SwissEphemeris.DEGTORAD/3600;
    else {
      /* This expansion is from Laskar, cited above.
       * Bretagnon and Simon say, in Planetary Programs and Tables, that it
       * is accurate to 0.1" over a span of 6000 years. Laskar estimates the
       * precision to be 0.01" after 1000 years and a few seconds of arc
       * after 10000 years.
       */
      T /= 10.0;
      eps = ((((((((( 2.45e-10*T + 5.79e-9)*T + 2.787e-7)*T
		   + 7.12e-7)*T - 3.905e-5)*T - 2.4967e-3)*T
		- 5.138e-3)*T + 1.99925)*T - 0.0155)*T - 468.093)*T
	+ 84381.448;
      eps *= SwissEphemeris.DEGTORAD/3600;
    }
    return(eps);
  }

  static double swe_difdeg2n(double p1, double p2)
  {
    double dif;
    dif = swe_degnorm(p1 - p2);
    if (dif  >= 180.0) return (dif - 360.0);
    return (dif);
}

  static double swe_difrad2n(double p1, double p2)
  {
    double dif;
    dif = swe_radnorm(p1 - p2);
    if (dif  >= TWOPI / 2) return (dif - TWOPI);
    return (dif);
  }

}
