using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

using SystemNeo;
using SystemNeo.Collections;
using SystemNeo.IO;

namespace SystemNeo.Drawing.Imaging.Jfif
{
	/// <summary>
	/// 
	/// </summary>
	public sealed class JfifSegment
	{
		// public vpeB //

		/// <summary>
		/// 
		/// </summary>
		public long BeginOffset { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public SegmentContent Content { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public JfifSegmentMarker Marker { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public long SegmentSize { get; private set; }

		// internal RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="marker"></param>
		/// <param name="beginOffset"></param>
		internal JfifSegment(BinaryReaderNeo reader, JfifSegmentMarker marker, long beginOffset)
		{
			this.Marker = marker;
			this.BeginOffset = beginOffset;
			this.SegmentSize = reader.BaseStream.Length;
			this.Content = ReadContent(reader, marker);
		}

		// public static \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="input"></param>
		/// <returns></returns>
		public static JfifSegment[] ReadSegments(Stream input)
		{
			BinaryReaderNeo reader = PrepareSeekableReader(input);
			var segmentList = new List<JfifSegment>();
			while (! reader.EOF) {
				segmentList.Add(ReadSegment(reader));
			}
			return segmentList.ToArray();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="path"></param>
		/// <returns></returns>
		public static JfifSegment[] ReadSegments(string path)
		{
			using (var stream = new FileStream(path, FileMode.Open, FileAccess.Read)) {
				return ReadSegments(stream);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="input"></param>
		/// <returns></returns>
		public static IEnumerable<JfifSegment> ReadSegmentsSequentially(Stream input)
		{
			BinaryReaderNeo reader = PrepareSeekableReader(input);
			while (! reader.EOF) {
				yield return ReadSegment(reader);
			}
		}

		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			var sw = new StringWriter();
			sw.Write("{");
			if (this.Marker == JfifSegmentMarker.Data) {
				sw.Write("Data");
			} else {
				sw.Write("Marker=");
				sw.Write(EnumUtil.ToString(this.Marker));
			}
			sw.Write(", Offset=");
			sw.Write(this.BeginOffset.ToString("N0"));
			sw.Write(", Size=");
			sw.Write(this.SegmentSize.ToString("N0"));
			if (this.Content != null) {
				sw.Write(", ");
				sw.Write(this.Content);
			}
			sw.Write("}");
			return sw.ToString();
		}

		// private static \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="input"></param>
		/// <returns></returns>
		private static BinaryReaderNeo PrepareSeekableReader(Stream input)
		{
			if (! input.CanRead || ! input.CanSeek) {
				throw new ArgumentException("Xg[ǂݎEV[N\ł͂܂B");
			}
			input.Position = 0;
			return new BinaryReaderNeo(input, ByteOrder.BigEndian, Encoding.ASCII);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="marker"></param>
		/// <returns></returns>
		private static SegmentContent ReadContent(BinaryReaderNeo reader, JfifSegmentMarker marker)
		{
			switch (marker) {
			case JfifSegmentMarker.APP0:
				return new App0SegmentContent(reader);
			case JfifSegmentMarker.APP1:
				return new App1SegmentContent(reader);
			case JfifSegmentMarker.APP2:
				return new App2SegmentContent(reader);
			case JfifSegmentMarker.APP12:
				return new App12SegmentContent(reader);
			case JfifSegmentMarker.APP13:
				return new App13SegmentContent(reader);
			case JfifSegmentMarker.APP14:
				return new App14SegmentContent(reader);
			case JfifSegmentMarker.COM:
				return new CommentSegmentContent(reader);
			case JfifSegmentMarker.DHT:
				return new DhtSegmentContent(reader);
			case JfifSegmentMarker.DQT:
				return new DqtSegmentContent(reader);
			case JfifSegmentMarker.DRI:
				return new DriSegmentContent(reader);
			case JfifSegmentMarker.SOF0:
			case JfifSegmentMarker.SOF2:
				return new Sof0SegmentContent(reader);
			case JfifSegmentMarker.SOS:
				return new SosSegmentContent(reader);
			case JfifSegmentMarker.Data:
			case JfifSegmentMarker.EOI:
			case JfifSegmentMarker.SOI:
			case JfifSegmentMarker.RST0:
			case JfifSegmentMarker.RST1:
			case JfifSegmentMarker.RST2:
			case JfifSegmentMarker.RST3:
			case JfifSegmentMarker.RST4:
			case JfifSegmentMarker.RST5:
			case JfifSegmentMarker.RST6:
			case JfifSegmentMarker.RST7:
				return null;
			default:
				return new BinarySegmentContent(reader, false);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="beginOffset"></param>
		/// <returns></returns>
		private static JfifSegment ReadDataSegment(BinaryReaderNeo reader, long beginOffset)
		{
			try {
				for (;;) {
					byte b = reader.ReadByte();
					if (b == 0xFF) {
						b = reader.ReadByte();
						if (b != 0) {
							reader.Seek(-2);
							break;
						}
					}
				}
			} catch (EndOfStreamException) {
			}
			return ReadSegmentInternal(reader, JfifSegmentMarker.Data, beginOffset);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <returns></returns>
		private static JfifSegment ReadSegment(BinaryReaderNeo reader)
		{
			long beginOffset = reader.Position;
			JfifSegmentMarker marker;
			try {
				marker = (JfifSegmentMarker)reader.ReadUInt16();
			} catch (EndOfStreamException) {
				return ReadDataSegment(reader, beginOffset);
			}
			ushort segmentSize;
			switch (marker) {
			case JfifSegmentMarker.SOI:
			case JfifSegmentMarker.EOI:
				// ZOgɃ}[J[Ȃ
				segmentSize = sizeof(JfifSegmentMarker);
				break;
			default:
				// 0xFF00 ̃oCg̓}[J[ł͂Ȃ
				if ((ushort)marker <= (ushort)JfifSegmentMarker.Base) {
					return ReadDataSegment(reader, beginOffset);
				}
				segmentSize = reader.ReadUInt16();
				break;
			}
			beginOffset = reader.Position;
			reader.Position += (segmentSize - sizeof(JfifSegmentMarker));
			return ReadSegmentInternal(reader, marker, beginOffset);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="marker"></param>
		/// <param name="beginOffset"></param>
		/// <returns></returns>
		private static JfifSegment ReadSegmentInternal(
				BinaryReaderNeo reader, JfifSegmentMarker marker, long beginOffset)
		{
			using (reader.KeepPosition()) {
				var segStream = new PartialStream(reader.BaseStream, beginOffset, reader.Position);
				var segReader = new BinaryReaderNeo(segStream, ByteOrder.BigEndian, Encoding.ASCII);
				return new JfifSegment(segReader, marker, beginOffset);
			}
		}
	}
}
