using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing.Imaging;
using System.IO;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Text;

using SystemNeo;
using SystemNeo.IO;
using SystemNeo.Text;

namespace SystemNeo.Drawing.Imaging.Tiff
{
	/// <summary>
	/// Tiff 摜Ɋ܂܂ Image File Directory ̃Gg\܂B
	/// </summary>
	public sealed class IfdEntry
	{
		#region private static fields
		private const int dataLength = 4;
		private const int maxEntryCount = 256;
		private const int stringMaxLength = 1024 * 256;
		private static readonly Formatter formatter = new Formatter();
		private static readonly QuotedStringFormatter quotedFormatter
				= new QuotedStringFormatter(256);
		#endregion

		#region private fields
		private uint count;
		private byte[] valueBytes;
		#endregion

		// public vpeB //

		/// <summary>
		/// 
		/// </summary>
		public long BeginOffset { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public long EndOffset { get; private set; }

		/// <summary>
		/// f[^̌`擾܂B
		/// </summary>
		public IfdFieldType FieldType { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public IfdTag Tag { get; private set; }

		/// <summary>
		/// 
		/// </summary>
		public object Value { get; private set; }

		// static RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		static IfdEntry()
		{
			formatter.ByteArrayFormatter.Separator = " ";
			formatter.ByteArrayFormatter.MaxCount = 20;
			formatter.CollectionFormatter.MaxCount = 10;
		}

		// public RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="property"></param>
		public IfdEntry(PropertyItem property)
		{
			this.Tag = IfdTag.FromId(property.Id);
			this.FieldType = (IfdFieldType)property.Type;
			this.count = (uint)property.Len;
			this.valueBytes = property.Value;
			this.Value = this.ReadValue(ByteOrder.LittleEndian);
		}

		// internal RXgN^ //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="ifdList"></param>
		internal IfdEntry(BinaryReaderNeo reader, IList<Ifd> ifdList)
		{
			this.BeginOffset = reader.Position;
			ushort tagId = reader.ReadUInt16();
			this.Tag = IfdTag.FromId(tagId);
			this.FieldType = (IfdFieldType)reader.ReadUInt16();
			if (! EnumUtil.IsDefined(this.FieldType)) {
				string msg = string.Format("f[^`słB[{0}]", this.FieldType);
				throw new BadImageFormatException(msg);
			}
			this.count = reader.ReadUInt32();
			this.valueBytes = reader.ReadBytes(dataLength);
			using (reader.KeepPosition()) {
				this.Value = this.ReadValue(reader);
				this.EndOffset = reader.Position;
				this.Value = this.Tag.ConvertValue(reader, ifdList, this.Value);
			}
		}

		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			var sw = new StringWriter();
			sw.Write("{Tag=");
			sw.Write(this.Tag);
			sw.Write(", Offset=");
			sw.Write(this.BeginOffset.ToString("N0"));
			if (this.count != 1 && ! (this.Value is string)) {
				sw.Write(", Count=");
				sw.Write(this.count);
			}
			sw.Write(", Type=");
			sw.Write(this.FieldType);
			sw.Write(", Value=");
			this.WriteValue(sw);
			sw.Write('}');
			return sw.ToString();
		}

		// private static \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="readMethod"></param>
		/// <param name="count"></param>
		/// <returns></returns>
		private static object ReadArray(BinaryReaderNeo reader, MethodInfo readMethod, uint count)
		{
			Array array = Array.CreateInstance(readMethod.ReturnType, count);
			for (int i = 0; i < count; i++) {
				array.SetValue(readMethod.Invoke(reader, null), i);
			}
			return array;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="msReader"></param>
		/// <param name="readMethodName"></param>
		/// <param name="count"></param>
		/// <returns></returns>
		private static object ReadFractions(
				BinaryReaderNeo reader, BinaryReaderNeo msReader, string readMethodName, uint count)
		{
			Debug.Assert(count > 0);
			Array array = (Array)ReadValueInternal(reader, msReader, readMethodName, count * 2);
			return ReadFractionsInternal(array);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="array"></param>
		/// <returns></returns>
		private static object ReadFractionsInternal(Array array)
		{
			int fractionCount = array.Length / 2;
			var fArray = new Fraction[fractionCount];
			for (int i = 0; i < fractionCount; i++) {
				long numerator = Convert.ToInt64(array.GetValue(i * 2));
				long denominator = Convert.ToInt64(array.GetValue(i * 2 + 1));
				fArray[i] = new Fraction(numerator, denominator);
			}
			if (fractionCount == 1) {
				return fArray[0];
			} else {
				return fArray;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="msReader"></param>
		/// <param name="readMethodName"></param>
		/// <param name="count"></param>
		/// <returns></returns>
		private static object ReadValueInternal(
				BinaryReaderNeo reader, BinaryReaderNeo msReader, string readMethodName, uint count)
		{
			Debug.Assert(count > 0);
			MethodInfo readMethod = typeof(BinaryReaderNeo).GetMethod(readMethodName);
			if (count == 1) {
				return readMethod.Invoke(msReader, null);
			} else {
				int size = Marshal.SizeOf(readMethod.ReturnType);
				if (size * count <= sizeof(uint)) {
					return ReadArray(msReader, readMethod, count);
				} else {
					reader.Position = msReader.PeekUInt32();
					return ReadArray(reader, readMethod, count);
				}
			}
		}

		// private \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <param name="msReader"></param>
		/// <returns></returns>
		private byte[] ReadBytes(BinaryReaderNeo reader, BinaryReaderNeo msReader)
		{
			if (this.count <= sizeof(uint)) {
				return msReader.ReadBytes((int)this.count);
			} else {
				reader.Position = msReader.ReadUInt32();
				return reader.ReadBytes((int)this.count);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="msReader"></param>
		/// <param name="readMethodName"></param>
		/// <returns></returns>
		private object ReadFractions(BinaryReaderNeo msReader, string readMethodName)
		{
			Array array = (Array)this.ReadValueInternal(msReader, readMethodName);
			this.count = (uint)array.GetLength(0) / 2;
			return ReadFractionsInternal(array);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="byteOrder"></param>
		/// <returns></returns>
		private object ReadValue(ByteOrder byteOrder)
		{
			using (var ms = new MemoryStream(this.valueBytes, false)) {
				var msReader = new BinaryReaderNeo(ms, byteOrder, Encoding.ASCII);
				switch (this.FieldType) {
				case IfdFieldType.Byte:
				case IfdFieldType.SByte:
				case IfdFieldType.Int16:
				case IfdFieldType.UInt16:
				case IfdFieldType.Int32:
				case IfdFieldType.UInt32:
				case IfdFieldType.Single:
				case IfdFieldType.Double:
					return this.ReadValueInternal(msReader, "Read" + this.FieldType);
				case IfdFieldType.Rational:
					return this.ReadFractions(msReader, "ReadInt32");
				case IfdFieldType.URational:
					return this.ReadFractions(msReader, "ReadUInt32");
				case IfdFieldType.Ascii:
					return Encoding.ASCII.GetString(this.valueBytes).TrimEnd('\0');
				case IfdFieldType.Undefined:
					return this.valueBytes;
				}
				// ΉȂ̂łƂ肠 null
				return null;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="reader"></param>
		/// <returns></returns>
		private object ReadValue(BinaryReaderNeo reader)
		{
			using (var ms = new MemoryStream(this.valueBytes, false)) {
				var msReader = new BinaryReaderNeo(ms, reader.ByteOrder, Encoding.ASCII);
				switch (this.FieldType) {
				case IfdFieldType.Byte:
				case IfdFieldType.SByte:
				case IfdFieldType.Int16:
				case IfdFieldType.UInt16:
				case IfdFieldType.Int32:
				case IfdFieldType.UInt32:
				case IfdFieldType.Single:
				case IfdFieldType.Double:
					return ReadValueInternal(reader, msReader, "Read" + this.FieldType, this.count);
				case IfdFieldType.Rational:
					return ReadFractions(reader, msReader, "ReadInt32", this.count);
				case IfdFieldType.URational:
					return ReadFractions(reader, msReader, "ReadUInt32", this.count);
				case IfdFieldType.Ascii:
					return Encoding.ASCII.GetString(this.ReadBytes(reader, msReader)).TrimEnd('\0');
				case IfdFieldType.Undefined:
					return this.ReadBytes(reader, msReader);
				}
				// ΉȂ̂łƂ肠 null
				return null;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="msReader"></param>
		/// <param name="readMethodName"></param>
		/// <returns></returns>
		private object ReadValueInternal(BinaryReaderNeo msReader, string readMethodName)
		{
			MethodInfo readMethod = typeof(BinaryReaderNeo).GetMethod(readMethodName);
			this.count = (uint)(msReader.BaseStream.Length / Marshal.SizeOf(readMethod.ReturnType));
			if (this.count == 1) {
				return readMethod.Invoke(msReader, null);
			} else {
				return ReadArray(msReader, readMethod, this.count);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="writer"></param>
		private void WriteValue(TextWriter writer)
		{
			if (this.Value is string) {
				quotedFormatter.WriteTo(writer, this.Value);
			} else if (this.Value is Ifd) {
				writer.Write("<");
				writer.Write(((Ifd)this.Value).Name);
				writer.Write(">");
			} else {
				formatter.WriteTo(writer, this.Value);
			}
		}
	}
}
