﻿using System;
using System.Collections;
using System.ComponentModel;
using System.Linq;
using System.Xml;

using SystemNeo.Collections;
using SystemNeo.Net;

namespace SystemNeo.Xml.Xindice
{
	/// <summary>
	/// Xindice データベースにアクセスします。
	/// </summary>
	public class XindiceClient
	{
		// public 定数 //

		public const string DefaultUri = "http://localhost:8888/xindice/";

		#region private static fields
		private const string CollectionMemberName = "collection";
		private const string MessageMemberName = "message";
		private const string QueryMemberName = "query";
		private const string ResultMemberName = "result";
		private const string RunMethodName = "run";
		private const string TypeMemberName = "type";
		#endregion

		#region private fields
		private XmlRpc xmlrpc;
		#endregion

		// public プロパティ //

		/// <summary>
		/// 
		/// </summary>
		public string ApiVersion
		{
			get {
				var responseDic = (IDictionary)this.Call(MessageNames.GetApiVersion);
				return (string)responseDic[ResultMemberName];
			}
		}

		/// <summary>
		/// 
		/// </summary>
		public string ServerVersion
		{
			get {
				var responseDic = (IDictionary)this.Call(MessageNames.GetServerVersion);
				return (string)responseDic[ResultMemberName];
			}
		}

		/// <summary>
		/// 
		/// </summary>
		[DefaultValue(DefaultUri)]
		public string Uri
		{
			get {
				return this.xmlrpc.Uri;
			}
		}

		// public コンストラクタ //

		/// <summary>
		/// 
		/// </summary>
		public XindiceClient() : this(DefaultUri) {}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="uri"></param>
		public XindiceClient(string uri)
		{
			this.xmlrpc = new XmlRpc(uri);
		}

		// public メソッド //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="collectionPath"></param>
		/// <returns></returns>
		public bool GetCollectionConfiguration(string collectionPath)
		{
			var requestDic = new Hashtable();
			requestDic[CollectionMemberName] = collectionPath;
			var responseDic
					= (IDictionary)this.Call(MessageNames.GetCollectionConfiguration, requestDic);
			return (string)responseDic[ResultMemberName] == "yes";
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="instancePath"></param>
		/// <returns></returns>
		public string[] GetCollectionNames(string instancePath)
		{
			object[] names = GetCollectionNamesInternal(instancePath);
			return Array.ConvertAll(names, (name) => (string)name);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="collectionPath"></param>
		/// <returns></returns>
		public XindiceCollection GetCollection(string collectionPath)
		{
			return new XindiceCollection(this, collectionPath);
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="instancePath"></param>
		/// <returns></returns>
		public XindiceCollection[] GetCollections(string instancePath)
		{
			object[] names = GetCollectionNamesInternal(instancePath);
			return Array.ConvertAll(names,
					(name) => new XindiceCollection(this, instancePath + "/" + name));
		}

		// internal メソッド //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="collectionPath"></param>
		/// <returns></returns>
		internal XmlDocument GetCollectionMetadata(string collectionPath)
		{
			var requestDic = new Hashtable();
			requestDic[CollectionMemberName] = collectionPath;
			var responseDic = (IDictionary)this.Call(MessageNames.GetCollectionMeta, requestDic);
			string xml = (string)responseDic[ResultMemberName];
			var doc = new XmlDocument();
			doc.LoadXml(xml);
			return doc;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="collectionPath"></param>
		/// <returns></returns>
		internal string[] GetDocumentNames(string collectionPath)
		{
			var requestDic = new Hashtable();
			requestDic[CollectionMemberName] = collectionPath;
			var responseDic = (IDictionary)this.Call(MessageNames.ListDocuments, requestDic);
			return ((object[])responseDic[ResultMemberName]).Cast<string>().ToArray();
		}

		/// <summary>
		/// 指定された XPath 式に一致するノードのリストを取得します。
		/// </summary>
		/// <param name="collectionPath"></param>
		/// <param name="xpath"></param>
		/// <returns></returns>
		internal XmlDocument Query(string collectionPath, string xpath)
		{
			var requestDic = new Hashtable() {
				{CollectionMemberName, collectionPath},
				{QueryMemberName, xpath},
				{TypeMemberName, QueryTypes.XPath}};
			var responseDic = (IDictionary)this.Call(MessageNames.Query, requestDic);
			string xml = (string)responseDic[ResultMemberName];
			var doc = new XmlDocument();
			doc.LoadXml(xml);
			return doc;
		}

		internal object Update(string collectionPath, string xupdate)
		{
			var dic = new Hashtable() {
				{CollectionMemberName, collectionPath},
				{QueryMemberName, xupdate},
				{TypeMemberName, QueryTypes.XUpdate}};
			return this.Call(MessageNames.Query, dic);
		}

		// private メソッド //

		private object Call(MessageNames messageName)
		{
			return this.Call(messageName, new Hashtable());
		}

		private object Call(MessageNames messageName, IDictionary dic)
		{
			dic[MessageMemberName] = messageName.ToString();
			return this.xmlrpc.Call(RunMethodName, dic);
		}

		private object[] GetCollectionNamesInternal(string instancePath)
		{
			var requestDic = new Hashtable();
			requestDic[CollectionMemberName] = instancePath;
			var responseDic = (IDictionary)this.Call(MessageNames.ListCollections, requestDic);
			return (object[])responseDic[ResultMemberName];
		}

		// 型 //

		/// <summary>
		/// 
		/// </summary>
		private enum MessageNames
		{
			GetApiVersion,
			GetCollectionConfiguration,
			GetCollectionMeta,
			GetServerVersion,
			ListCollections,
			ListDocuments,
			Query
		}

		/// <summary>
		/// 
		/// </summary>
		private enum QueryTypes
		{
			XPath,
			XUpdate
		}
	}
}
