using System;
using System.Collections;
using System.ComponentModel;
using System.Reflection;
using System.Xml;

using SystemNeo;

namespace SystemNeo.Reflection
{
	/// <summary>
	///
	/// </summary>
	public class ObjectAnalyzer
	{
		#region private fields
		private int arrayLengthLimit = 128;
		private bool skipStaticFields = true;
		#endregion

		// public vpeB //

		/// <summary>
		/// 
		/// </summary>
		[DefaultValue(false)]
		public bool AnalyzePrimitiveValue { get; set; }

		/// <summary>
		/// 
		/// </summary>
		[DefaultValue(128)]
		public int ArrayLengthLimit
		{
			get {
				return this.arrayLengthLimit;
			}
			set {
				this.arrayLengthLimit = value;
			}
		}

		/// <summary>
		/// 
		/// </summary>
		[DefaultValue(true)]
		public bool SkipStaticFields
		{
			get {
				return this.skipStaticFields;
			}
			set {
				this.skipStaticFields = value;
			}
		}

		// public \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="target"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		public XmlElement Analyze(XmlDocument doc, object target, int recurseLimit)
		{
			XmlElement result = doc.CreateElement("root");
			result.AppendChild(this.GetValue(doc, target, recurseLimit));
			return result;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="target"></param>
		/// <param name="fieldName"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		public XmlElement Analyze(
				XmlDocument doc, object target, string fieldName, int recurseLimit)
		{
			Type type = target.GetType();
			do {
				FieldInfo field = type.GetField(fieldName, MemberUtil.AllBindingFlags);
				if (field != null) {
					return Analyze(doc, field.GetValue(target), recurseLimit);
				}
				type = type.BaseType;
			} while (type != null);

			throw new MissingFieldException(target.GetType().FullName, fieldName);
		}

		// private \bh //

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="target"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		private XmlElement GetFields(XmlDocument doc, object target, int recurseLimit)
		{
			XmlElement result = doc.CreateElement(ElementNames.Fields);
			if (--recurseLimit > 0) {
				BindingFlags flags = MemberUtil.AllBindingFlags;
				if (this.skipStaticFields) {
					flags &= ~ BindingFlags.Static;
				}
				Type t = target.GetType();
				do {
					foreach (FieldInfo field in t.GetFields(flags)) {
						if (field.DeclaringType == t) {
							result.AppendChild(this.GetField(doc, target, field, recurseLimit));
						}
					}
					t = t.BaseType;
				} while (t != null);
			}
			return result;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="target"></param>
		/// <param name="field"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		private XmlElement GetField(
				XmlDocument doc, object target, FieldInfo field, int recurseLimit)
		{
			XmlElement result = doc.CreateElement(ElementNames.Field);

			XmlAttribute nameAttr = doc.CreateAttribute(AttributeNames.Name);
			result.Attributes.Append(nameAttr);
			nameAttr.Value = field.Name;

			result.Attributes.Append(this.GetAccessibility(doc, (FieldInfoNeo)field));

			if (field.IsStatic) {
				XmlAttribute staticAttr = doc.CreateAttribute(AttributeNames.Static);
				result.Attributes.Append(staticAttr);
				staticAttr.Value = bool.TrueString;
			}

			if (field.IsLiteral) {
				XmlAttribute staticAttr = doc.CreateAttribute(AttributeNames.Const);
				result.Attributes.Append(staticAttr);
				staticAttr.Value = bool.TrueString;
			}

			if (field.IsInitOnly) {
				XmlAttribute staticAttr = doc.CreateAttribute(AttributeNames.ReadOnly);
				result.Attributes.Append(staticAttr);
				staticAttr.Value = bool.TrueString;
			}

			XmlElement typeElement = doc.CreateElement(ElementNames.ValueType);
			result.AppendChild(typeElement);
			typeElement.AppendChild(GetType(doc, field.FieldType));

			XmlElement decElement = doc.CreateElement(ElementNames.DeclaringType);
			result.AppendChild(decElement);
			decElement.AppendChild(GetType(doc, field.DeclaringType));

			result.AppendChild(this.GetValue(doc, field.GetValue(target), recurseLimit));

			return result;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="value"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		private XmlElement GetValue(XmlDocument doc, object value, int recurseLimit)
		{
			if (value == null) {
				return doc.CreateElement(ElementNames.NullValue);
			} else if (value.GetType().IsArray) {
				return this.GetArrayValue(doc, (Array)value, recurseLimit);
			} else {
				return this.GetObjectValue(doc, value, recurseLimit);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="array"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		private XmlElement GetArrayValue(XmlDocument doc, Array array, int recurseLimit)
		{
			XmlElement result = doc.CreateElement(ElementNames.ArrayValue);

			XmlAttribute rankAttr = doc.CreateAttribute(AttributeNames.ArrayRank);
			result.Attributes.Append(rankAttr);
			rankAttr.Value = array.Rank.ToString();

			XmlElement elementTypeElement = doc.CreateElement(ElementNames.ElementType);
			result.AppendChild(elementTypeElement);
			elementTypeElement.AppendChild(this.GetType(doc, array.GetType().GetElementType()));

			XmlElement elementsElement = doc.CreateElement(ElementNames.ArrayElements);
			result.AppendChild(elementsElement);
			XmlAttribute lengthAttr = doc.CreateAttribute(AttributeNames.ArrayLength);
			elementsElement.Attributes.Append(lengthAttr);
			lengthAttr.Value = array.Length.ToString();

			int count = 0;
			foreach (object element in array) {
				if (++count > this.arrayLengthLimit) {
					break;
				}
				elementsElement.AppendChild(this.GetValue(doc, element, recurseLimit));
			}

			return result;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="target"></param>
		/// <param name="recurseLimit"></param>
		/// <returns></returns>
		private XmlElement GetObjectValue(XmlDocument doc, object target, int recurseLimit)
		{
			Type type = target.GetType();
			string strValue = target.ToString();
			XmlElement result = doc.CreateElement(ElementNames.Value);
			if (strValue != type.FullName) {
				XmlAttribute strAttr = doc.CreateAttribute(AttributeNames.ValueAsString);
				result.Attributes.Append(strAttr);
				strAttr.Value = strValue;
				XmlAttribute unicodeAttr
						= doc.CreateAttribute(AttributeNames.ValueAsUnicodeSequence);
				result.Attributes.Append(unicodeAttr);
				unicodeAttr.Value = StringUtil.CreateUnicodeSequence(strValue, "{0:X2}", " ");
			}
			result.AppendChild(this.GetType(doc, type));
			if (! type.IsPrimitive || this.AnalyzePrimitiveValue) {
				result.AppendChild(this.GetFields(doc, target, recurseLimit));
			}
			return result;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="type"></param>
		/// <returns></returns>
		private XmlElement GetType(XmlDocument doc, Type type)
		{
			XmlElement result = doc.CreateElement(ElementNames.Type);

			XmlAttribute nameAttr = doc.CreateAttribute(AttributeNames.Name);
			result.Attributes.Append(nameAttr);
			nameAttr.Value = type.FullName;

			XmlAttribute assmAttr = doc.CreateAttribute(AttributeNames.AssemblyName);
			result.Attributes.Append(assmAttr);
			assmAttr.Value = type.Assembly.GetName().Name;

			if (type.IsEnum) {
				XmlAttribute enumAttr = doc.CreateAttribute(AttributeNames.Enum);
				result.Attributes.Append(enumAttr);
				enumAttr.Value = bool.TrueString;
			}

			if (type.IsValueType) {
				XmlAttribute structAttr = doc.CreateAttribute(AttributeNames.Struct);
				result.Attributes.Append(structAttr);
				structAttr.Value = bool.TrueString;
			}

			if (type.IsPrimitive) {
				XmlAttribute primAttr = doc.CreateAttribute(AttributeNames.Primitive);
				result.Attributes.Append(primAttr);
				primAttr.Value = bool.TrueString;
			}

			return result;
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="doc"></param>
		/// <param name="member"></param>
		/// <returns></returns>
		private XmlAttribute GetAccessibility(XmlDocument doc, MemberInfoNeo member)
		{
			XmlAttribute result = doc.CreateAttribute(AttributeNames.Accessibility);
			result.Value = member.Accessibility.ToString();
			return result;
		}

		// ^ //

		/// <summary>
		/// 
		/// </summary>
		public static class AttributeNames
		{
			public const string Accessibility = "accessibility";
			public const string ArrayLength = "array-length";
			public const string ArrayRank = "array-rank";
			public const string AssemblyName = "assembly-name";
			public const string Const = "const";
			public const string Enum = "enum";
			public const string Name = "name";
			public const string Primitive = "primitive";
			public const string ReadOnly = "readonly";
			public const string Static = "static";
			public const string Struct = "struct";
			public const string ValueAsString = "value-as-string";
			public const string ValueAsUnicodeSequence = "value-as-unicode-sequence";
		}

		/// <summary>
		/// 
		/// </summary>
		public static class ElementNames
		{
			public const string ArrayElements = "array-elements";
			public const string ArrayValue = "array-value";
			public const string DeclaringType = "declaring-type";
			public const string ElementType = "element-type";
			public const string Field = "field";
			public const string Fields = "fields";
			public const string NullValue = "null-value";
			public const string Type = "type";
			public const string Value = "value";
			public const string ValueType = "value-type";
		}
	}
}
