using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Windows.Forms;
using SystemNeo;
using SystemNeo.Collections;
using NUnit.Framework;

namespace SystemNeoTest.Collections.CollectionUtilTest
{
	/// <summary>
	/// CastSelect \bh̃eXg
	/// </summary>
	[TestFixture]
	public class CastSelect
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable source = new string[] { "5", "10", "03" };
			Func<string, int> selector = (s) => int.Parse(s);
				
			// s
			IEnumerable<int> result = CollectionUtil.CastSelect(source, selector);

			// 
			Assert.That(result, Is.EquivalentTo(new int[] { 5, 10, 3 }));
		}
	}

	[TestFixture]
	public class Concat_IEnumerableTArray
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable<int> source1 = new int[] { 4, 1, 2 };
			IEnumerable<int> source2 = new int[] { 3, 0 };
			IEnumerable<int> source3 = new int[] { 1, 5 };

			// s
			IEnumerable<int> result = CollectionUtil.Concat<int>(source1, source2, source3);

			// 
			Assert.That(result, Is.EquivalentTo(new int[] { 4, 1, 2, 3, 0, 1, 5 }));
		}
	}

	/// <summary>
	/// Copy \bh̃eXg
	/// </summary>
	[TestFixture]
	public class Copy_IEnumerable_IList
	{
		[Test]
		public void destination̕()
		{
			// 
			int[] source = { 1, 3, 5 };
			int[] dest = { 2, 4, 6, 8 };

			// s
			int result = CollectionUtil.Copy(source, dest);

			// 
			Assert.That(result, Is.EqualTo(3));
			Assert.That(dest, Is.EqualTo(new int[] { 1, 3, 5, 8 }));
		}

		[Test]
		public void source̕()
		{
			// 
			int[] source = { 1, 3, 5, 7 };
			int[] dest = { 2, 4 };

			// s
			int result = CollectionUtil.Copy(source, dest);

			// 
			Assert.That(result, Is.EqualTo(2));
			Assert.That(dest, Is.EqualTo(new int[] { 1, 3 }));
		}
	}

	/// <summary>
	/// CreateArray \bh̃eXg
	/// </summary>
	[TestFixture]
	public class CreateArray
	{
		[Test]
		public void 0()
		{
			// s
			string[] result = CollectionUtil.CreateArray(0, (i) => (i * i).ToString());

			// 
			Assert.AreEqual(new string[] {}, result);
		}

		[Test]
		public void 2ȏ()
		{
			// s
			string[] result = CollectionUtil.CreateArray(4, (i) => (i * i).ToString());

			// 
			Assert.AreEqual(new string[] {"0", "1", "4", "9"}, result);
		}
	}

	/// <summary>
	/// CreateTree \bh̃eXg
	/// </summary>
	[TestFixture]
	public class CreateTree
	{
		private bool HasAsParent(string child, string parent)
		{
			return child.StartsWith(parent);
		}

		private TreeNode CreateNode(string value)
		{
			return new TreeNode(value);
		}

		private void AddChild(TreeNode parent, TreeNode child)
		{
			parent.Nodes.Add(child);
		}

		private string NodeValueGetter(TreeNode node)
		{
			return node.Text;
		}

		[Test][ExpectedException(typeof(ArgumentNullException))]
		public void @null()
		{
			// 
			string[] nodeData = null;

			// s
			TreeNode[] result = CollectionUtil.CreateTree<string, TreeNode>(
					nodeData, HasAsParent, CreateNode, AddChild, NodeValueGetter);
		}

		[Test]
		public void ̔z()
		{
			// 
			string[] nodeData = {};

			// s
			TreeNode[] result = CollectionUtil.CreateTree<string, TreeNode>(
					nodeData, HasAsParent, CreateNode, AddChild, NodeValueGetter);

			// 
			Assert.AreEqual(0, result.Length);
		}

		[Test]
		public void P̔z()
		{
			// 
			string[] nodeData = {"A"};

			// s
			TreeNode[] result = CollectionUtil.CreateTree<string, TreeNode>(
					nodeData, HasAsParent, CreateNode, AddChild, NodeValueGetter);

			// 
			Assert.AreEqual(1, result.Length);
			Assert.AreEqual("A", result[0].Text);
		}

		[Test]
		public void eq֌ŴȂz()
		{
			// 
			string[] nodeData = {"A", "B"};

			// s
			TreeNode[] result = CollectionUtil.CreateTree<string, TreeNode>(
					nodeData, HasAsParent, CreateNode, AddChild, NodeValueGetter);

			// 
			Assert.AreEqual(2, result.Length);
			Assert.AreEqual("A", result[0].Text);
			Assert.AreEqual("B", result[1].Text);
			Assert.Null(result[0].Parent);
			Assert.Null(result[1].Parent);
			Assert.AreEqual(0, result[0].Nodes.Count);
			Assert.AreEqual(0, result[1].Nodes.Count);
		}

		[Test]
		public void eq֌Ŵz()
		{
			// 
			string[] nodeData = {"A", "B", "Bookmark", "Apple", "Book", "Empty", "Art"};

			// s
			TreeNode[] result = CollectionUtil.CreateTree<string, TreeNode>(
					nodeData, HasAsParent, CreateNode, AddChild, NodeValueGetter);

			// 
			Assert.AreEqual(3, result.Length);
			TreeNode nodeA = result[0];
			TreeNode nodeB = result[1];
			TreeNode nodeEmpty = result[2];
			Assert.AreEqual("A", nodeA.Text);
			Assert.AreEqual("B", nodeB.Text);
			Assert.AreEqual("Empty", nodeEmpty.Text);
			Assert.Null(nodeA.Parent);
			Assert.Null(nodeB.Parent);
			Assert.Null(nodeEmpty.Parent);
			Assert.AreEqual(2, nodeA.Nodes.Count);
			Assert.AreEqual(1, nodeB.Nodes.Count);
			Assert.AreEqual(0, nodeEmpty.Nodes.Count);
			TreeNode nodeApple = nodeA.Nodes[0];
			TreeNode nodeArt = nodeA.Nodes[1];
			TreeNode nodeBook = nodeB.Nodes[0];
			Assert.AreEqual("Apple", nodeApple.Text);
			Assert.AreEqual("Art", nodeArt.Text);
			Assert.AreEqual("Book", nodeBook.Text);
			Assert.AreEqual(0, nodeApple.Nodes.Count);
			Assert.AreEqual(0, nodeArt.Nodes.Count);
			Assert.AreEqual(1, nodeBook.Nodes.Count);
			TreeNode nodeBookmark = nodeBook.Nodes[0];
			Assert.AreEqual("Bookmark", nodeBookmark.Text);
			Assert.AreEqual(0, nodeBookmark.Nodes.Count);
		}
	}

	/// <summary>
	/// FindSame \bh̃eXg
	/// </summary>
	[TestFixture]
	public class FindSame_IEnumerable
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new string[] { "A", "B", "a", "B", "C", "A", "B" };

			// s
			IEnumerable result = CollectionUtil.FindSame(e);

			// 
			IEnumerator enumerator = result.GetEnumerator();
			Assert.True(enumerator.MoveNext());
			IEnumerable group1 = (IEnumerable)enumerator.Current;
			Assert.That(group1, Is.EquivalentTo(new object[] { "A", "A" }));
			Assert.True(enumerator.MoveNext());
			IEnumerable group2 = (IEnumerable)enumerator.Current;
			Assert.That(group2, Is.EquivalentTo(new object[] { "B", "B", "B" }));
			Assert.False(enumerator.MoveNext());
		}
	}

	/// <summary>
	/// FindSame \bh̃eXg
	/// </summary>
	[TestFixture]
	public class FindSame_IEnumerableT
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable<string> e = new string[] { "A", "B", "a", "B", "C", "A", "B" };

			// s
			IEnumerable<IEnumerable<string>> result = CollectionUtil.FindSame(e);

			// 
			IEnumerator<IEnumerable<string>> enumerator = result.GetEnumerator();
			Assert.True(enumerator.MoveNext());
			Assert.That(enumerator.Current, Is.EquivalentTo(new object[] { "A", "A" }));
			Assert.True(enumerator.MoveNext());
			Assert.That(enumerator.Current, Is.EquivalentTo(new object[] { "B", "B", "B" }));
			Assert.False(enumerator.MoveNext());
		}
	}

	/// <summary>
	/// FindSame \bh̃eXg
	/// </summary>
	[TestFixture]
	public class FindSame_IEnumerableT_IEqualityComparerT
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable<string> e = new string[] { "A", "B", "a", "B", "C", "A", "B" };
			IEqualityComparer<string> comparer = StringComparer.InvariantCultureIgnoreCase;

			// s
			IEnumerable<IEnumerable<string>> result = CollectionUtil.FindSame(e, comparer);

			// 
			IEnumerator<IEnumerable<string>> enumerator = result.GetEnumerator();
			Assert.True(enumerator.MoveNext());
			Assert.That(enumerator.Current, Is.EquivalentTo(new object[] { "A", "a", "A" }));
			Assert.True(enumerator.MoveNext());
			Assert.That(enumerator.Current, Is.EquivalentTo(new object[] { "B", "B", "B" }));
			Assert.False(enumerator.MoveNext());
		}
	}

	[TestFixture]
	public class Flatten
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new object[] {
				"1",
				new object[] {
					"2.1",
					new string[] { "2.2.1", "2.2.2" },
					"2.3"
				},
				"3"
			};

			// s
			IEnumerable result = CollectionUtil.Flatten(e);

			// 
			IEnumerable expected = new string[] { "1", "2.1", "2.2.1", "2.2.2", "2.3", "3" };
			Assert.That(result, Is.EquivalentTo(expected));
		}
	}

	[TestFixture]
	public class ForEach
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new int[] { 3, 0, 2, 1 };
			IList<int> list = new List<int>();
			Action<int> action = list.Add;

			// s
			CollectionUtil.ForEach(e, action);

			// 
			Assert.That(list, Is.EquivalentTo(new int[] { 3, 0, 2, 1 }));
		}
	}

	[TestFixture]
	public class GetNextItem_Struct
	{
		[Test]
		public void Nullable()
		{
			// 
			IEnumerable e = new int[] { 5, 2, 0, 1, 3, 0, 4 };
			int? current = 0;

			// s
			int? result = CollectionUtil.GetNextItem(e, current);

			// 
			Assert.That(result, Is.EqualTo(1));
		}

		[Test]
		public void Nullable_()
		{
			// 
			IEnumerable e = new int[] { 5, 2, 0, 1, 3, 0, 4 };
			int? current = 4;

			// s
			int? result = CollectionUtil.GetNextItem(e, current);

			// 
			Assert.Null(result);
		}
	}

	[TestFixture]
	public class GetPreviousItem_Struct
	{
		[Test]
		public void Nullable()
		{
			// 
			IEnumerable e = new int[] { 5, 2, 0, 1, 3, 0, 4 };
			int? current = 0;

			// s
			int? result = CollectionUtil.GetPreviousItem(e, current);

			// 
			Assert.That(result, Is.EqualTo(2));
		}

		[Test]
		public void Nullable_擪()
		{
			// 
			IEnumerable e = new int[] { 5, 2, 0, 1, 3, 0, 4 };
			int? current = 5;

			// s
			int? result = CollectionUtil.GetPreviousItem(e, current);

			// 
			Assert.Null(result);
		}
	}

	[TestFixture]
	public class Max_IEnumerable_Func
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new int[] { 15, 100, 1 };
			Func<int, string> valueGetter = (x) => x.ToString();

			// s
			int result = CollectionUtil.Max(e, valueGetter);

			// 
			Assert.AreEqual(15, result);
		}
	}

	[TestFixture]
	public class Max_IEnumerable_Func_Comparison
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new int[] { 100, 15, -100, -15 };
			Func<int, string> valueGetter = (x) => x.ToString();
			Comparison<string> comparison = (x, y) => x.Length - y.Length;

			// s
			int result = CollectionUtil.Max(e, valueGetter, comparison);

			// 
			Assert.AreEqual(-100, result);
		}
	}

	[TestFixture]
	public class Min_IEnumerable_Func
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new int[] { 150, 10, 5 };
			Func<int, string> valueGetter = (x) => x.ToString();
			
			// s
			int result = CollectionUtil.Min(e, valueGetter);

			// 
			Assert.AreEqual(10, result);
		}
	}

	[TestFixture]
	public class Min_IEnumerable_Func_Comparison
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new int[] { 100, 15, -100, -15 };
			Func<int, string> valueGetter = (x) => x.ToString();
			Comparison<string> comparison = (x, y) => x.Length - y.Length;

			// s
			int result = CollectionUtil.Min(e, valueGetter, comparison);

			// 
			Assert.AreEqual(15, result);
		}
	}

	[TestFixture]
	public class PickUp_IList_Object
	{
		[Test]
		public void Normal()
		{
			// 
			IList list = new ArrayList() { 1, 3, 0, 2, 3, 1 };

			// s
			IList result = CollectionUtil.PickUp(list, 3);

			// 
			Assert.That(result, Is.EquivalentTo(new int[] { 3, 3 }));
			Assert.That(list, Is.EquivalentTo(new int[] { 1, 0, 2, 1 }));
		}
	}

	[TestFixture]
	public class PickUp_IListT_T
	{
		[Test]
		public void Normal()
		{
			// 
			IList<int> list = new List<int>() { 1, 3, 0, 2, 3, 1 };

			// s
			IList<int> result = CollectionUtil.PickUp<int>(list, 3);

			// 
			Assert.That(result, Is.EquivalentTo(new int[] { 3, 3 }));
			Assert.That(list, Is.EquivalentTo(new int[] { 1, 0, 2, 1 }));
		}
	}

	[TestFixture]
	public class Select
	{
		[Test]
		public void Normal()
		{
			// 
			IEnumerable e = new string[] { "", "3", "2", "13 characters" };
			// ZIԃ\bh
			Selector<string> selector = (x, y) => (x.Length < y.Length ? x : y);

			// s
			string result = CollectionUtil.Select(e, selector);

			// 
			Assert.AreEqual("2", result);
		}
	}
}
