/*******************************************************
*
* Author: Hirokazu Kato
*
*         kato@sys.im.hiroshima-cu.ac.jp
*
* Revision: 2.11
* Date: 00/05/06
*
*******************************************************/

#include <stdio.h>
#include <math.h>
#include <AR/ar.h>
#include <AR/matrix.h>

extern void DbgDisplayOut( void* image, int ave );

#define   DEBUG        0
#define   EVEC_MAX     10

#define REDUCED_IMAGE_W   8
#define REDUCED_IMAGE_H   8


static int pattern_num = -1;
static int patf[AR_PATT_NUM_MAX] = { 0 };
static int pat[AR_PATT_NUM_MAX][4][AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3];
static int patpow[AR_PATT_NUM_MAX][4];
static int patBW[AR_PATT_NUM_MAX][4][AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3];
static int patpowBW[AR_PATT_NUM_MAX][4];

static ARVAL evec[EVEC_MAX][AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3];
static ARVAL epat[AR_PATT_NUM_MAX][4][EVEC_MAX];
static int    evec_dim;
static int    evecf = 0;
//static double evecBW[EVEC_MAX][AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3];
//static double epatBW[AR_PATT_NUM_MAX][4][EVEC_MAX];
//static int    evec_dimBW;
static int    evecBWf = 0;

int isCodePatt = 0;

static void   get_cpara( ARVAL world[4][2], ARVAL vertex[4][2],
												ARVAL para[3][3] );
static int    pattern_match( ARUint8 *data, int dataave, int *code, int *dir, ARVAL *cf );
static void   put_zero( ARUint8 *p, int size );
static void   gen_evec(void);
static long long getTagCode( unsigned char img[ REDUCED_IMAGE_W*REDUCED_IMAGE_H ], int* pdir );
static void reduceImage( unsigned char img[ REDUCED_IMAGE_W*REDUCED_IMAGE_H ], unsigned short src[ AR_PATT_SIZE_Y ][ AR_PATT_SIZE_X ], int ave );


void arSetPattMode( int mode )
{
	isCodePatt = (mode == 1);
}

int arLoadPatt( const char *filename )
{
	FILE    *fp;
	int     patno;
	int     h, i, j, l, m;
	int     i1, i2, i3;

	if(pattern_num == -1 ) {
		for( i = 0; i < AR_PATT_NUM_MAX; i++ ) patf[i] = 0;
		pattern_num = 0;
	}

	for( i = 0; i < AR_PATT_NUM_MAX; i++ ) {
		if(patf[i] == 0) break;
	}
	if( i == AR_PATT_NUM_MAX ) return -1;
	patno = i;

	if( (fp=arfopen(filename, "r")) == NULL ) {
		arDbgTrace("\"%s\" not found!!\n", filename);
		return(-1);
	}

	for( h=0; h<4; h++ ) {
		l = 0;
		for( i3 = 0; i3 < 3; i3++ ) {
			for( i2 = 0; i2 < AR_PATT_SIZE_Y; i2++ ) {
				for( i1 = 0; i1 < AR_PATT_SIZE_X; i1++ ) {
					if( arfscanf(fp, "%d", &j) != 1 ) {
						arDbgTrace("Pattern Data read error!!\n");
						return -1;
					}
					j = 255-j;
					pat[patno][h][(i2*AR_PATT_SIZE_X+i1)*3+i3] = j;
					if( i3 == 0 ) patBW[patno][h][i2*AR_PATT_SIZE_X+i1]  = j;
					else          patBW[patno][h][i2*AR_PATT_SIZE_X+i1] += j;
					if( i3 == 2 ) patBW[patno][h][i2*AR_PATT_SIZE_X+i1] /= 3;
					l += j;
				}
			}
		}
		l /= (AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3);

		m = 0;
		for( i = 0; i < AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; i++ ) {
			pat[patno][h][i] -= l;
			m += (pat[patno][h][i]*pat[patno][h][i]);
		}
		patpow[patno][h] = SqrtInt( m );
		if( patpow[patno][h] == 0 ) patpow[patno][h] = 1;

		m = 0;
		for( i = 0; i < AR_PATT_SIZE_Y*AR_PATT_SIZE_X; i++ ) {
			patBW[patno][h][i] -= l;
			m += (patBW[patno][h][i]*patBW[patno][h][i]);
		}
		patpowBW[patno][h] = SqrtInt( m );
		if( patpowBW[patno][h] == 0 ) patpowBW[patno][h] = 1;
	}
	arfclose(fp);

	patf[patno] = 1;
	pattern_num++;

	/*
	gen_evec();
	*/

	return( patno );
}

int arLoadPattFromImage( const char *string, int len )
{
	char    scanbuf[ 16 ];
	int     rp = 0;
	int     patno;
	int     h, i, j, l, m;
	int     i1, i2, i3;

	if(pattern_num == -1 ) {
		for( i = 0; i < AR_PATT_NUM_MAX; i++ ) patf[i] = 0;
		pattern_num = 0;
	}

	for( i = 0; i < AR_PATT_NUM_MAX; i++ ) {
		if(patf[i] == 0) break;
	}
	if( i == AR_PATT_NUM_MAX ) return -1;
	patno = i;

	for( h=0; h<4; h++ ) {
		l = 0;
		for( i3 = 0; i3 < 3; i3++ ) {
			for( i2 = 0; i2 < AR_PATT_SIZE_Y; i2++ ) {
				for( i1 = 0; i1 < AR_PATT_SIZE_X; i1++ ) {

					int n;
					for ( ; rp<len; rp++ )
					{
						if ( string[ rp ]>='0' && string[ rp ]<='9' )
							break;
					}
					if ( rp >= len )
					{
						arDbgTrace("Pattern Data read error!!\n");
						return -1;
					}
					for ( n=0;  n<sizeof(scanbuf)-1 && rp<len;  n++,rp++ )
					{
						if ( string[ rp ]<'0' || string[ rp ]>'9' )
							break;
						scanbuf[ n ] = string[ rp ];
					}
					scanbuf[ n ] = 0;
					if ( sscanf(scanbuf, "%d", &j) != 1 ) 
					{
						arDbgTrace("Pattern Data read error!!\n");
						return -1;
					}
					
					j = 255-j;
					pat[patno][h][(i2*AR_PATT_SIZE_X+i1)*3+i3] = j;
					if( i3 == 0 ) patBW[patno][h][i2*AR_PATT_SIZE_X+i1]  = j;
					else          patBW[patno][h][i2*AR_PATT_SIZE_X+i1] += j;
					if( i3 == 2 ) patBW[patno][h][i2*AR_PATT_SIZE_X+i1] /= 3;
					l += j;
				}
			}
		}
		l /= (AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3);

		m = 0;
		for( i = 0; i < AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; i++ ) {
			pat[patno][h][i] -= l;
			m += (pat[patno][h][i]*pat[patno][h][i]);
		}
		patpow[patno][h] = SqrtInt( m );
		if( patpow[patno][h] == 0 ) patpow[patno][h] = 1;

		m = 0;
		for( i = 0; i < AR_PATT_SIZE_Y*AR_PATT_SIZE_X; i++ ) {
			patBW[patno][h][i] -= l;
			m += (patBW[patno][h][i]*patBW[patno][h][i]);
		}
		patpowBW[patno][h] = SqrtInt( m );
		if( patpowBW[patno][h] == 0 ) patpowBW[patno][h] = 1;
	}

	patf[patno] = 1;
	pattern_num++;

	/*
	gen_evec();
	*/

	return( patno );
}

int arFreePatt( int patno )
{
	if( patf[patno] == 0 ) return -1;

	patf[patno] = 0;
	pattern_num--;

	gen_evec();

	return 1;
}

int arActivatePatt( int patno )
{
	if( patf[patno] == 0 ) return -1;

	patf[patno] = 1;

	return 1;
}

int arDeactivatePatt( int patno )
{
	if( patf[patno] == 0 ) return -1;

	patf[patno] = 2;

	return 1;
}

int arGetCode( ARUint8 *image, int *x_coord, int *y_coord, int *vertex,
							long long *code, int *dir, ARVAL *cf )
{
	//#if DEBUG
	//static int count = 0;
	//static double a1 = 0.0;
	//static double a2 = 0.0;
	//double b1, b2, b3;
	//#endif
	int pattave;
	int pattid;

#if ( arTemplateMatchingMode == AR_TEMPLATE_MATCHING_COLOR )
	ARUint8 ext_pat[AR_PATT_SIZE_Y][AR_PATT_SIZE_X][3];
#else
	ARUint16 ext_pat[AR_PATT_SIZE_Y][AR_PATT_SIZE_X];
#endif

	//#if DEBUG
	//b1 = arUtilTimer();
	//#endif
	if ( arGetPatt( image, x_coord, y_coord, vertex, ext_pat, &pattave ) < 0 )
		return -1;
	//#if DEBUG
	//b2 = arUtilTimer();
	//#endif

//DbgDisplayOut( ext_pat, pattave );

	if ( isCodePatt )
	{
		unsigned char rimg[ REDUCED_IMAGE_W*REDUCED_IMAGE_H ];
		reduceImage( rimg, ext_pat, pattave );
		*code = getTagCode( rimg, dir );
		if ( *code < 0 )
		{
			*code = -1LL;
			*cf = 0;
			*dir = 0;
			//return -1;
		}
		else
		{
			*cf = DoubleToARVAL( 1.0 );
		}
	}
	else
	{
		if ( pattern_match( (ARUint8 *)ext_pat, pattave, &pattid, dir, cf ) < 0 )
		{
			//return -1;
		}
		*code = pattid;
	}
	//#if DEBUG
	//b3 = arUtilTimer();
	//#endif

	//#if DEBUG
	//a1 += (b2 - b1);
	//a2 += (b3 - b2);
	//count++;
	//if( count == 60 ) {
	//    printf("%10.5f[msec], %10.5f[msec]\n", a1*1000.0/60.0, a2*1000.0/60.0);
	//    count = 0;
	//    a1 = a2 = 0.0;
	//}
	//#endif

	return 0;
}

static int matNorm( int a, int b )
{
	long long r = a*(long long)a + b*(long long)b;
	r >>= 2;
	if ( r > 0x7fffffff )
		r = 0x7fffffff;
	return (int)r;
}

#if (AR_PATT_SIZE_X == 16)
#define AR_PATT_SIZE_X_SHIFT 4
#endif
#if (AR_PATT_SIZE_Y == 16)
#define AR_PATT_SIZE_Y_SHIFT 4
#endif
#if (AR_PATT_SAMPLE_NUM == 64)
#define AR_PATT_SAMPLE_NUM_SHIFT 6
#endif

#if ( arTemplateMatchingMode == AR_TEMPLATE_MATCHING_COLOR )
int arGetPatt( ARUint8 *image, int *x_coord, int *y_coord, int *vertex, ARUint8 ext_pat[AR_PATT_SIZE_Y][AR_PATT_SIZE_X][3], int *ext_pat_ave )
{
	ARUint32  ext_pat2[AR_PATT_SIZE_Y][AR_PATT_SIZE_X][3];
#else
int arGetPatt( ARUint8 *image, int *x_coord, int *y_coord, int *vertex, ARUint16 ext_pat[AR_PATT_SIZE_Y][AR_PATT_SIZE_X], int *ext_pat_ave )
{
	ARUint32  ext_pat2[AR_PATT_SIZE_Y][AR_PATT_SIZE_X];
#endif
	ARVAL    world[4][2];
	ARVAL    local[4][2];
	ARVAL    para[3][3];
	ARVAL    d, dc, xw, yw, xwc, ywc;
	int       xc, yc;
	int       xdivshift, ydivshift;
	int       xdiv2shift, ydiv2shift;
	int       lx1, lx2, ly1, ly2;
	int       ext_pat2_x_index, ext_pat2_y_index;
	int       image_index;
	int       i, j;
	int       ave;

	world[0][0] = IntToARVAL( 100 );
	world[0][1] = IntToARVAL( 100 );
	world[1][0] = IntToARVAL( 100 + 10 );
	world[1][1] = IntToARVAL( 100 );
	world[2][0] = IntToARVAL( 100 + 10 );
	world[2][1] = IntToARVAL( 100 + 10 );
	world[3][0] = IntToARVAL( 100 );
	world[3][1] = IntToARVAL( 100 + 10 );
	for( i = 0; i < 4; i++ ) {
		local[i][0] = IntToARVAL( x_coord[vertex[i]] );
		local[i][1] = IntToARVAL( y_coord[vertex[i]] );
	}
	get_cpara( world, local, para );

	lx1 = matNorm( x_coord[vertex[0]] - x_coord[vertex[1]],  y_coord[vertex[0]] - y_coord[vertex[1]] );
	lx2 = matNorm( x_coord[vertex[2]] - x_coord[vertex[3]],  y_coord[vertex[2]] - y_coord[vertex[3]] );
	ly1 = matNorm( x_coord[vertex[1]] - x_coord[vertex[2]],  y_coord[vertex[1]] - y_coord[vertex[2]] );
	ly2 = matNorm( x_coord[vertex[3]] - x_coord[vertex[0]],  y_coord[vertex[3]] - y_coord[vertex[0]] );
	if( lx2 > lx1 ) lx1 = lx2;
	if( ly2 > ly1 ) ly1 = ly2;
	xdiv2shift = AR_PATT_SIZE_X_SHIFT;
	ydiv2shift = AR_PATT_SIZE_Y_SHIFT;
	if( arImageProcMode == AR_IMAGE_PROC_IN_FULL ) {
		while( 1<<(xdiv2shift*2) < lx1  &&  xdiv2shift < AR_PATT_SAMPLE_NUM_SHIFT ){ xdiv2shift++; }
		while( 1<<(ydiv2shift*2) < ly1  &&  ydiv2shift < AR_PATT_SAMPLE_NUM_SHIFT ){ ydiv2shift++; }
	}
	else {
		while( 1<<(xdiv2shift*2+2) < lx1  &&  xdiv2shift < AR_PATT_SAMPLE_NUM_SHIFT ){ xdiv2shift++; }
		while( 1<<(ydiv2shift*2+2) < ly1  &&  ydiv2shift < AR_PATT_SAMPLE_NUM_SHIFT ){ ydiv2shift++; }
	}
	if( xdiv2shift > AR_PATT_SAMPLE_NUM_SHIFT ){ xdiv2shift = AR_PATT_SAMPLE_NUM_SHIFT; }
	if( ydiv2shift > AR_PATT_SAMPLE_NUM_SHIFT ){ ydiv2shift = AR_PATT_SAMPLE_NUM_SHIFT; }

	xdivshift = xdiv2shift - AR_PATT_SIZE_X_SHIFT;
	ydivshift = ydiv2shift - AR_PATT_SIZE_Y_SHIFT;
	/*
	printf("%3d(%f), %3d(%f)\n", xdiv2, sqrt(lx1), ydiv2, sqrt(ly1));
	*/

	put_zero( (ARUint8 *)ext_pat2, sizeof(ext_pat2) );
	yw =  DoubleToARVAL( 102.5 ) + (IntToARVAL( 5 ) >> (ydiv2shift+1)) - (IntToARVAL( 5 ) >> ydiv2shift); //102.5 + 5.0 * (j+0.5) * ydiv2_reciprocal


	for( j=0; j < 1<<ydiv2shift; j++ ) 
	{
		yw += IntToARVAL( 5 ) >> ydiv2shift;
		xwc = MulARVAL( para[0][1], yw ) + para[0][2];
		ywc = MulARVAL( para[1][1], yw ) + para[1][2];
		dc  = MulARVAL( para[2][1], yw ) + para[2][2];
		
		xw = DoubleToARVAL( 102.5 ) + (IntToARVAL( 5 ) >> (xdiv2shift+1)) - (IntToARVAL( 5 ) >> xdiv2shift); //102.5 + 5.0 * (i+0.5) * xdiv2_reciproc
		
		for( i=0; i < 1<<xdiv2shift; i++ ) 
		{
			xw += IntToARVAL( 5 ) >> xdiv2shift;
			
			d = MulARVAL( para[2][0], xw ) + dc;
			if ( d == 0 ) 
				return -1;

			xc = ARVALtoInt( DivARVAL( MulARVAL( para[0][0], xw ) + xwc, d ) );
			yc = ARVALtoInt( DivARVAL( MulARVAL( para[1][0], xw ) + ywc, d ) );

			if( arImageProcMode == AR_IMAGE_PROC_IN_HALF ) 
			{
				// ŉʃrbg؂グ
				xc = (xc+1) & ~1;
				yc = (yc+1) & ~1;
			}
			if( xc >= 0 && xc < arImXsize && yc >= 0 && yc < arImYsize ) 
			{
				ext_pat2_y_index = j >> ydivshift;
				ext_pat2_x_index = i >> xdivshift;
				image_index = (yc*arImXsize+xc)*AR_PIX_SIZE_DEFAULT;
#if ( arTemplateMatchingMode == AR_TEMPLATE_MATCHING_COLOR )
#	if (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_ARGB)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+3];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+2];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+1];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_ABGR)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+2];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+3];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_BGRA)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+0];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+2];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_BGR)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+0];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+2];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_RGBA)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+2];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+0];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_RGB)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+2];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+0];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_MONO)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_2vuy)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+1];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+1];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_yuvs)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += image[image_index+0];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += image[image_index+0];
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += image[image_index+0];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_RGB565)
				{
					int r,g,b;
					unsigned short pixel = *(unsigned short*)&image[image_index];
					getRGBfromRGB565( r, g, b, pixel );
					ext_pat2[ext_pat2_y_index][ext_pat2_x_index][0] += b;
					ext_pat2[ext_pat2_y_index][ext_pat2_x_index][1] += g;
					ext_pat2[ext_pat2_y_index][ext_pat2_x_index][2] += r;
				}
#	else
#	  error Unknown default pixel format defined in config.h
#	endif
#else
#	if (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_ARGB) || (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_ABGR)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index] += image[image_index+1] + image[image_index+2] + image[image_index+3];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_BGRA) || (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_BGR) || (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_RGB) || (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_RGBA)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index] += image[image_index+0] + image[image_index+1] + image[image_index+2];
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_MONO)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index] += image[image_index]*3;
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_2vuy)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index] += image[image_index+1]*3;
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_yuvs)
				ext_pat2[ext_pat2_y_index][ext_pat2_x_index] += image[image_index+0]*3;
#	elif (AR_DEFAULT_PIXEL_FORMAT == AR_PIXEL_FORMAT_RGB565)
				{
					int r,g,b;
					unsigned short pixel = *(unsigned short*)&image[image_index];
					getRGBfromRGB565( r, g, b, pixel );
					ext_pat2[ext_pat2_y_index][ext_pat2_x_index] += b+g+r;
				}
#	else
#	  error Unknown default pixel format defined in config.h
#	endif
#endif
			}
		}
	}

	xdivshift += ydivshift;
	ave = 0;
	for( j = 0; j < AR_PATT_SIZE_Y; j++ ) {
		for( i = 0; i < AR_PATT_SIZE_X; i++ ) {				// PRL 2006-06-08.
#if ( arTemplateMatchingMode == AR_TEMPLATE_MATCHING_COLOR )
			ext_pat[j][i][0] = 255 - (ARUint8)(ext_pat2[j][i][0] >> xdivshift);
			ext_pat[j][i][1] = 255 - (ARUint8)(ext_pat2[j][i][1] >> xdivshift);
			ext_pat[j][i][2] = 255 - (ARUint8)(ext_pat2[j][i][2] >> xdivshift);
			ave += ext_pat[j][i][0] + ext_pat[j][i][1] + ext_pat[j][i][2];
#else
			ext_pat[j][i] = 255*3 - (ARUint16)(ext_pat2[j][i] >> xdivshift);
			ave += ext_pat[j][i];
#endif
		}
	}
	ave >>= AR_PATT_SIZE_X_SHIFT + AR_PATT_SIZE_Y_SHIFT;
	*ext_pat_ave = ave;

	return(0);
}

static void get_cpara( ARVAL world[4][2], ARVAL vertex[4][2],
											ARVAL para[3][3] )
{
	ARMat   *a, *b, *c;
	int     i, k, sft;

	a = arMatrixAlloc( 8, 8 );
	b = arMatrixAlloc( 8, 1 );
	c = arMatrixAlloc( 8, 1 );
	sft = 0;
	for( i = 0; i < 4; i++ ) {
		a->ml[i*16+0]  = world[i][0];
		a->ml[i*16+1]  = world[i][1];
		a->ml[i*16+2]  = IntToARVAL( 1 );
		a->ml[i*16+3]  = 0;
		a->ml[i*16+4]  = 0;
		a->ml[i*16+5]  = 0;
		a->ml[i*16+6]  = MulARVAL( -world[i][0], vertex[i][0] );
		a->ml[i*16+7]  = MulARVAL( -world[i][1], vertex[i][0] );
		a->ml[i*16+8]  = 0;
		a->ml[i*16+9]  = 0;
		a->ml[i*16+10] = 0;
		a->ml[i*16+11] = world[i][0];
		a->ml[i*16+12] = world[i][1];
		a->ml[i*16+13] = IntToARVAL( 1 );
		a->ml[i*16+14] = MulARVAL( -world[i][0], vertex[i][1] );
		a->ml[i*16+15] = MulARVAL( -world[i][1], vertex[i][1] );
		b->ml[i*2+0] = vertex[i][0];
		b->ml[i*2+1] = vertex[i][1];
		for ( k=0; k<16; k++ )
			a->ml[ i*16+k ] >>= sft;
	}
	arMatrixSelfInv( a );
	arMatrixMulShift( c, &sft, a, b, -sft );
	//if ( sft > 0 )
	//	DebugBreak();
	for( i = 0; i < 2; i++ ) {
		para[i][0] = c->ml[i*3+0] << sft;
		para[i][1] = c->ml[i*3+1] << sft;
		para[i][2] = c->ml[i*3+2] << sft;
	}
	para[2][0] = c->ml[2*3+0] << sft;
	para[2][1] = c->ml[2*3+1] << sft;
	para[2][2] = IntToARVAL( 1 );
	arMatrixFree( a );
	arMatrixFree( b );
	arMatrixFree( c );
}

static int pattern_match( ARUint8 *data, int dataave, int *code, int *dir, ARVAL *cf )
{
	int    i, j, l, k;
	int    sum, res, res2, datapow;
	ARDBL  max, sum2;
	
#if ( arTemplateMatchingMode == AR_TEMPLATE_MATCHING_COLOR )
	int input[AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3];
	dataave /= 3;
	for ( i=0,sum=0; i<AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; i++ ) 
	{
		input[i] = data[i] - dataave;
		sum += input[i]*input[i];
	}
#else
	ARUint16* d16 = (ARUint16*)data;
	int input[AR_PATT_SIZE_Y*AR_PATT_SIZE_X];
	for ( i=0,sum=0; i<AR_PATT_SIZE_Y*AR_PATT_SIZE_X; i++ )
	{
		input[i] = d16[i] - dataave;
		sum += input[i]*input[i];
	}
#endif
	if( sum == 0 ) {
		*code = 0;
		*dir  = 0;
		*cf   = 0;
		return -1;
	}
	datapow = SqrtInt( sum );

	res = res2 = -1;
#if ( arTemplateMatchingMode == AR_TEMPLATE_MATCHING_COLOR )
	if( arMatchingPCAMode == AR_MATCHING_WITH_PCA && evecf ) {

		ARDBL  min;
		ARVAL invec[EVEC_MAX];
		for( i = 0; i < evec_dim; i++ ) {
			invec[i] = 0;
			for( j = 0; j < AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; j++ ) {
				invec[i] += MulARVAL( evec[i][j], IntToARVAL( input[j] ) );
			}
			invec[i] /= datapow;
		}

		min = IntToARDBL( 10000 );
		k = -1;
		for( l = 0; l < pattern_num; l++ ) {
			k++;
			while( patf[k] == 0 ) k++;
			if( patf[k] == 2 ) continue;
			//#if DEBUG
			//printf("%3d: ", k);
			//#endif
			for( j = 0; j < 4; j++ ) {
				sum2 = 0;
				for(i = 0; i < evec_dim; i++ ) {
					ARDBL v = ARVALtoARDBL(invec[i] - epat[k][j][i]);
					sum2 += MulARDBL( v, v );
				}
				//#if DEBUG
				//printf("%10.7f ", sum2);
				//#endif
				if ( sum2 < min ) 
				{
					min = sum2; 
					res = j;
					res2 = k; 
				}
			}
			//#if DEBUG
			//printf("\n");
			//#endif
		}
		for ( i=0,sum=0; i<AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; i++ )
			sum += input[i] * pat[res2][res][i];
		max = IntToARDBL( sum ) / (patpow[res2][res] * datapow);
	}
	else 
	{
		for( l=0,k=-1,max=0; l < pattern_num; l++ ) 
		{
			k++;
			while( patf[k] == 0 ) k++;
			if( patf[k] == 2 ) continue;
			for( j = 0; j < 4; j++ ) {
				sum = 0;
				for(i=0;i<AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3;i++)
					sum += input[i] * pat[k][j][i];
				sum2 = IntToARDBL( sum ) / (patpow[k][j] * datapow);
				if ( sum2 > max ) 
				{
					max = sum2;
					res = j; 
					res2 = k; 
				}
			}
		}
	}
#else
	for( l=0,k=-1,max=0; l < pattern_num; l++ ) 
	{
		k++;
		while ( patf[ k ] == 0 )
			k++;
		if ( patf[ k ] == 2 )
			continue;
			
		for ( j=0; j<4; j++ ) 
		{
			int* pbw = patBW[ k ][ j ];
			for( i=0,sum=0; i<AR_PATT_SIZE_Y*AR_PATT_SIZE_X; i++ ) 
				sum += input[ i ] * pbw[ i ];
				
			sum2 = IntToARDBL( sum ) / (patpowBW[ k ][ j ] * datapow);
			if( sum2 > max )
			{
				max = sum2;
				res = j;
				res2 = k;
			}
		}
	}
#endif

	*code = res2;
	*dir  = res;
	*cf   = ARDBLtoARVAL( max );

	//#if DEBUG
	//    printf("%d %d %f\n", res2, res, max);
	//#endif

	return 0;
}

static void   put_zero( ARUint8 *p, int size )
{
	while( (size--) > 0 ) *(p++) = 0;
}

static void gen_evec(void)
{
	int    i, j, k, ii, jj;
	ARMat  *input, *wevec;
	ARVec  *wev;
	ARVAL sum;
	ARDBL sum2;
	int    dim;
	ARVAL sumthres = DoubleToARVAL( 0.9 );

	if( pattern_num < 4 ) {
		evecf   = 0;
		evecBWf = 0;
		return;
	}

//#if DEBUG
//	printf("------------------------------------------\n");
//#endif

	dim = (pattern_num*4 < AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3)? pattern_num*4: AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3;
	input  = arMatrixAlloc( pattern_num*4, AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3 );
	wevec   = arMatrixAlloc( dim, AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3 );
	wev     = arVecAlloc( dim );

	for( j = jj = 0; jj < AR_PATT_NUM_MAX; jj++ ) {
		if( patf[jj] == 0 ) continue;
		for( k = 0; k < 4; k++ ) {
			for( i = 0; i < AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; i++ ) {
				input->ml[(j*4+k)*AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3+i] = ARDBLtoARVAL( IntToARDBL( pat[j][k][i] ) / patpow[j][k] );
			}
		}
		j++;
	}

	if( arMatrixPCA2(input, wevec, wev) < 0 ) {
		arMatrixFree( input );
		arMatrixFree( wevec );
		arVecFree( wev );
		evecf   = 0;
		evecBWf = 0;
		return;
	}

	sum = 0;
	for( i = 0; i < dim; i++ ) {
		sum += wev->vl[i];
		//#if DEBUG
		//        printf("%2d(%10.7f): \n", i+1, sum);
		//#endif
		if( sum > sumthres ) break;
		if( i == EVEC_MAX-1 ) break;
	}
	evec_dim = i+1;

	for( j = 0; j < evec_dim; j++ ) {
		for( i = 0; i < AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3; i++ ) {
			evec[j][i] = wevec->ml[j*AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3+i];
		}
	}

	for( i = 0; i < AR_PATT_NUM_MAX; i++ ) {
		if(patf[i] == 0) continue;
		for( j = 0; j < 4; j++ ) {
			//#if DEBUG
			//            printf("%2d[%d]: ", i+1, j+1);
			//#endif
			sum2 = 0;
			for( k = 0; k < evec_dim; k++ ) {
				//ARDBL dbl;
				sum = 0;
				for(ii=0;ii<AR_PATT_SIZE_Y*AR_PATT_SIZE_X*3;ii++) {
					sum += MulARVAL( evec[k][ii], IntToARVAL( pat[i][j][ii] ) ) / patpow[i][j];
				}
				//#if DEBUG
				//                printf("%10.7f ", sum);
				//#endif
				epat[i][j][k] = sum;
				//dbl = ARVALtoARDBL( sum );
				//sum2 += MulARDBL( dbl, dbl );
			}
			//#if DEBUG
			//            printf(":: %10.7f\n", sqrt(sum2));
			//#endif
		}
	}

	arMatrixFree( input );
	arMatrixFree( wevec );
	arVecFree( wev );

	evecf   = 1;
	evecBWf = 0;

	return;
}


static int crc8( unsigned char* data, int begin, int length )
{
	int n;
  int crc = 0xff;
	for ( n=0; n<length; n++ )
  {
		unsigned char d = data[ n + begin ];
    int i;
    for ( i=0; i<8; i++ )
    {
	    int b = crc & 1;
	    crc = (crc>>1) | ((d & 1)<<7);
	    d >>= 1;
	    if ( b != 0 )
	    {
		    crc ^= 0xb1;
	    }
    }
  }
  return crc;
}
static int crc12( unsigned char* data, int begin, int length )
{
  int n;
  int crc = 0xfff;
  for ( n = 0; n < length; n++ )
  {
    unsigned char d = data[ n + begin ];
    int i;
    for ( i = 0; i < 8; i++ )
    {
      int b = crc & 1;
      crc = (crc >> 1) | ((d & 1)<<11);
      d >>= 1;
      if ( b != 0 )
      {
        crc ^= 0xf01;
      }
    }
  }
  return crc;
}
static int crc16( unsigned char* data, int begin, int length )
{
	int n;
  int crc = 0xffff;
	for ( n=0; n<length; n++ )
  {
		unsigned char d = data[ n + begin ];
    int i;
    for ( i=0; i<8; i++ )
    {
	    int b = crc & 1;
	    crc = (crc>>1) | ((d & 1)<<15);
	    d >>= 1;
	    if ( b != 0 )
	    {
		    crc ^= 0x8408;
	    }
    }
  }
  return crc;
  // 1 0001 0000 0010 0001  ->  1 0000 1000 0001 0001
}

static void reduceImage( unsigned char img[ REDUCED_IMAGE_W*REDUCED_IMAGE_H ], unsigned short src[ AR_PATT_SIZE_Y ][ AR_PATT_SIZE_X ], int ave )
{
	int x,y;
	for ( y=0; y<AR_PATT_SIZE_Y; y+=AR_PATT_SIZE_Y/REDUCED_IMAGE_H )
	{
		for ( x=0; x<AR_PATT_SIZE_X; x+=AR_PATT_SIZE_X/REDUCED_IMAGE_W )
		{
#if (AR_PATT_SIZE_X > REDUCED_IMAGE_W)
			int v0 = src[ y   ][ x   ];
			int v1 = src[ y   ][ x+1 ];
			int v2 = src[ y+1 ][ x   ];
			int v3 = src[ y+1 ][ x+1 ];
			int v = (((v0 + v1 + v2 + v3)>>2) - ave);
			v = (v < 0)? 0 : ((v > 255)? 255 : v);
			if ( v < 16*3 ) v = 0;	else v = 255;
			img[ (x + y*REDUCED_IMAGE_W)>>1 ] = v;
#else
			img[ x + y*rw ] = src[ x ][ y ] - ave;
#endif
		}
	}
}

static int getBox( unsigned char img[ REDUCED_IMAGE_W*REDUCED_IMAGE_H ], int x, int y, int dir )
{
	switch ( dir )
	{
	case 0:
		return img[ x + y*REDUCED_IMAGE_W ];
	case 1:
		return img[ y + REDUCED_IMAGE_W*(REDUCED_IMAGE_W-1 - x) ];
	case 2:
		return img[ (REDUCED_IMAGE_W-1 - x) + REDUCED_IMAGE_W*(REDUCED_IMAGE_H-1 - y) ];
	case 3:
		return img[ (REDUCED_IMAGE_H-1 - y) + x*REDUCED_IMAGE_W ];
	}
	return 0;
}

static int nextPermutaion( int* p, int n )
{
	int tmp, next;
	if ( n <= 1 )
		return FALSE;
	for ( next=n-1; ; )
	{
		int np,nl,nx;
		nx = next--;
		if ( p[ next ] < p[ nx ] )
		{
			int m;
			for ( m=n-1;  ; m-- )
			{
				if ( p[ next ] < p[ m ] )
				{
					tmp = p[ m ];
					p[ m ] = p[ next ];
					p[ next ] = tmp;
					break;
				}
			}

			for ( np=nx,nl=n-1;  np<nl;  np++,nl-- )
			{
				tmp = p[ np ];
				p[ np ] = p[ nl ];
				p[ nl ] = tmp;
			}
			return TRUE;
		}

		if ( next == 0 )
		{
			for ( np=0,nl=n-1;  np<nl;  np++,nl-- )
			{
				tmp = p[ np ];
				p[ np ] = p[ nl ];
				p[ nl ] = tmp;
			}
			return FALSE;
		}
	}
}

static long long getTagCode( unsigned char img[ REDUCED_IMAGE_W*REDUCED_IMAGE_H ], int* pdir )
{
	int i,x,y,dir,id,crc;
	int isZero;
	long long ldat;
#define MAX_PARERR  4	// < 8
	unsigned char datasrc[ 7 ];
	int candidate[ MAX_PARERR*MAX_PARERR ][ 2 ];
	int npeIdx[ MAX_PARERR ];
	int nParErr;
	int nEr;
	int isEnd;
	
	for ( dir=0; dir<4; dir++ )
	{
		// 𔻒
		if ( dir == 0  &&  img[ 0 ] == 0 )
			continue;
		if ( dir == 1  &&  img[ (REDUCED_IMAGE_H-1)*REDUCED_IMAGE_W ] == 0 )
			continue;
		if ( dir == 2  &&  img[ REDUCED_IMAGE_W-1 + (REDUCED_IMAGE_H-1)*REDUCED_IMAGE_W ] == 0 )
			continue;
		if ( dir == 3  &&  img[ REDUCED_IMAGE_W-1 ] == 0 )
			continue;
		*pdir = dir;

		id = (getBox( img, 1, 0, dir ) != 0)? 1:0;
		if ( id == 1 )
		{
			// dpeB+CRC-6
			unsigned char data[ 7 ];
			int oddpA = 0;
			int oddpB = 0;
			int parityA = 0;
			int parityB = 0;
			
			for ( y=6; y>=0; y-- )
			{
				data[ y ] = 0;
				for ( x=7; x>=1; x-- )
				{
					data[ y ] <<= 1;
					if ( getBox( img, x, y, dir ) )
					{
						data[ y ] |= 1;
						oddpA ^= 1 << y;
						oddpB ^= 1 << (x-1);
					}
				}
			}

			// 0`6sbit0̓peBA
			for ( y=6; y>=0; y-- )
			{
				parityA <<= 1;
				if ( getBox( img, 0, y, dir ) )
					parityA |= 1;
			}
			// 7sbit1`7̓peBB
			for ( x=7; x>=1; x-- )
			{
				parityB <<= 1;
				if ( getBox( img, x, 7, dir ) )
					parityB |= 1;
			}

			//
			// peBG[oƒ
			//
			// rbg_񋓂
			nParErr = 0;
			for ( y=0; y<=6; y++ )
			{
				if ( (oddpA & (1<<y)) != (parityA & (1<<y)) )
				{
					// xŃpeBG[
					// ỹG[T
					for ( x=0; x<=6; x++ )
					{
						if ( (oddpB & (1<<x)) != (parityB & (1<<x)) )
						{
							// x,yŃG[Ȃ炻̃rbgƂ݂Ȃ
							if ( ((oddpB & (1<<x))^(1<<x))  ==  (parityB & (1<<x)) )
							{
								if ( nParErr < MAX_PARERR*MAX_PARERR )
								{
									candidate[ nParErr ][ 0 ] = x;
									candidate[ nParErr ][ 1 ] = y;
								}
								nParErr++;
							}
						}
					}
					if ( x <= 6 )
					{
						// ỹpeBG[Ȃ΃peB
						oddpA ^= 1<<y;
					}
				}
			}

			if ( nParErr > MAX_PARERR*MAX_PARERR )
				break;

			// rbg_ʂɒĂ݂
			for ( i=MAX_PARERR; i>=0; i-- )
			{
				if ( nParErr >= i*i )
				{
					nEr = i;
					break;
				}
			}
			for ( i=0; i<nEr; i++ )
			{
				npeIdx[ i ] = i;
			}
			for ( isZero=TRUE,i=0; i<sizeof(data); i++ )
			{
				datasrc[ i ] = data[ i ];
				if ( data[ i ] )
					isZero = FALSE;
			}
			if ( isZero )
				break;
				
			for ( isEnd=FALSE; !isEnd;  )
			{
				if ( nEr > 1 )
				{
					for ( i=0; i<sizeof(data); i++ )
						data[ i ] = datasrc[ i ];
				}
				for ( i=0; i<nEr; i++ )
				{
					int idx = i + npeIdx[ i ]*nEr;
					data[ candidate[ idx ][ 1 ] ] ^= 1 << candidate[ idx ][ 0 ];
				}
				isEnd = !nextPermutaion( npeIdx, nEr );

				id = data[ 0 ] & 1;
				if ( id == 1 )
				{
					int p;
					ldat = 0;
					for ( i=6; i>=1; i-- )
					{
						for ( p=6; p>=0; p-- )
						{
							ldat <<= 1;
							ldat |= (data[ i ] & (1<<p))? 1:0;
						}
					}
					for ( p=4; p>=1; p-- )
					{
						ldat <<= 1;
						ldat |= (data[ 0 ] & (1<<p))? 1:0;
					}

					crc = (int)((ldat >> 34) & 0xfff);
					ldat &= (1LL<<34) - 1;

					if ( ldat != 0  &&  ldat != (1LL<<34)-1  &&  crc == crc12( (unsigned char*)&ldat, 0, 5 ) )
					{
						// fC^[[u
						long long c = 0;
						long long m = 1;
						int k;

						if ( ldat & (1LL<<8) )
							c |= 1LL<<33;
						ldat = ((ldat>>1) & ~0xffULL) | (ldat & 0xff);
						if ( ldat & (1LL<<16) )
							c |= 1LL<<32;
						ldat = ((ldat>>1) & ~0xffffULL) | (ldat & 0xffff);
						for ( i = 0; i < 8; i++ )
						{
							for ( k = i; k < 32; k += 8, m<<=1 )
							{
								if ( ldat & m )
								{
									c |= 1LL << k;
								}
							}
						}
						return c;
					}
				}

			}
		}

		//
		// ID == 0
		//
		// peB{CRC-12
		ldat = 0;
		crc = 0;
		for ( x=7; x>=1; x-- )
		{
			ldat <<= 1;
			if ( getBox( img, x, 7, dir ) )
				ldat |= 1;
		}
		for ( y=6; y>=1; y-- )
		{
			for ( x=7; x>=0; x-- )
			{
				ldat <<= 1;
				if ( getBox( img, x, y, dir ) )
					ldat |= 1;
			}
		}
		for ( x=6; x>=2; x-- )
		{
			ldat <<= 1;
			if ( getBox( img, x, 0, dir ) )
				ldat |= 1;
		}

		crc = (int)((ldat >> 44) & 0xffff);
		ldat &= (1LL<<44) - 1;

		if ( ldat != 0  &&  ldat != (1LL<<44)-1  &&  crc == crc16( (unsigned char*)&ldat, 0, 6 ) )
		{
			// fC^[[u
			long long c = 0;
			long long m = 1;
			int k;
			for ( i = 0; i < 11; i++ )
			{
				for ( k = i; k < 44; k += 11, m<<=1 )
				{
					if ( ldat & m )
					{
						c |= 1LL << k;
					}
				}
			}
			return c;
		}
	} // for ( dir=0; dir<4; dir++ )
	return -1LL;
}
