/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Evaluators/EvaluatorFamily.cs#6 $
 * $DateTime: 2007/10/25 12:56:12 $
 * 
 * IEvaluator̎BBinaryOperator݂ł₱
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;

using Bellagio.Script;
using Bellagio.Values;

namespace Bellagio.Evaluators {

    //Evaluator̋@\ɐ[悤Ȋ֐
    //zmap, count, anyAVtgnAletAȂ
    public interface IBSystemFunction {
        //̕]A^`FbNق
        BSystemFunctionArg CreateArgEvaluators(EvaluatorBuilder builder, InvokeExpression expr);
        //{
        ExecResult Execute(EvalContext context, Evaluator[] evaluators, BV result);
    }
    public class BSystemFunctionArg {
        public Evaluator[] arg_evaluators;
        public BT retType;
        public BSystemFunctionArg(BT retT, Evaluator[] args) {
            retType = retT;
            arg_evaluators = args;
        }
    }

    //Expressionn̊e^ɑΉEvaluator
    //ꂼꂪʂێBVnoɎBEvalƂɉߏnewHvB
    public abstract class Evaluator : IEvaluator {
        protected BT _retType;
        public Evaluator(BT type) {
            Debug.Assert(type!=null);
            _retType = type;
        }

        public BT BT {
            get {
                return _retType;
            }
        }
        public abstract BV Eval(EvalContext context);

    }

    //Öق̌^ϊ
    public class TypeConversionEvaluator<ST,DT> : Evaluator where ST:BV where DT:BV {
        public delegate void Conv(ST val, DT dest);
        protected int _resultIndex;
        protected Conv _conv;
        protected Evaluator _content;
        public TypeConversionEvaluator(EvaluatorBuildContext ctx, BT dest, Evaluator content, Conv conv)
            : base(dest) {
            _conv = conv;
            _content = content;
            _resultIndex = ctx.DefineLocal("", dest);
        }
        public override BV Eval(EvalContext context) {
            BV src = _content.Eval(context);
            if(src.IsNil)
                return _retType.Nil;
            else {
                DT result = (DT)context.Stack.GetAt(_resultIndex);
                _conv((ST)src, result);
                return result;
            }
        }
    }


    public class LiteralEvaluator : Evaluator {
        private BV _value;

        public LiteralEvaluator(BV value) : base(value.BT)  {
            _value = value;
            //NOTE fobOpɒlύXs̃bNق悢
        }
        public BV Content {
            get {
                return _value;
            }
        }

        public override BV Eval(EvalContext context) {
            return _value;
        }

        public static LiteralEvaluator CreateNumber(string value) {
            if(value.IndexOf('.')!=-1 || value.Length>=10) //10ȏDoubleŏ
                return new LiteralEvaluator(new BDouble(Double.Parse(value)));
            else
                return new LiteralEvaluator(new BInt(Int32.Parse(value)));
        }
        public static LiteralEvaluator CreateBool(bool value) {
            return new LiteralEvaluator(new BBoolean(value));
        }
        public static LiteralEvaluator CreateTime(string value) {
            return new LiteralEvaluator(new BTime(Int32.Parse(value)));
        }
        public static LiteralEvaluator CreateDate(string value) {
            return new LiteralEvaluator(new BDate(Int32.Parse(value)));
        }
        public static LiteralEvaluator CreateString(string value) {
            return new LiteralEvaluator(new BString(value));
        }
    }

    public class ParameterEvaluator : Evaluator {
        private INamedValue _value;
        public ParameterEvaluator(INamedValue value) : base(value.BT) {
            _value = value;
        }
        public override BV Eval(EvalContext context) {
            return _value.GetValue(context);
        }
        public INamedValue Content {
            get {
                return _value;
            }
        }
    }


    public class ConditionalEvaluator : Evaluator {
        //private Expression _condExpr;
        private Evaluator _condition;
        private Evaluator _trueCase;
        private Evaluator _falseCase;

        public ConditionalEvaluator(Evaluator condition, Evaluator trueCase, Evaluator falseCase) : base(trueCase.BT) {
            //_condExpr = expr;
            _condition = condition;
            _trueCase = trueCase;
            _falseCase = falseCase;
        }

        public override BV Eval(EvalContext context) {
            BV c = _condition.Eval(context);
            if(c.IsNil) return _retType.Nil;

            if(((BBoolean)c).Value)
                return _trueCase.Eval(context);
            else
                return _falseCase.Eval(context);
        }

    }

    //SymbolEvaluator: contextɊ֘Ǎ݂^疼Oă^ṽo𓾂delegatelăLbVAƂȊ

    //BinaryOpx[X
    public abstract class BinaryOpEvaluator : Evaluator {
        protected Evaluator _left;
        protected Evaluator _right;
        protected int _resultLocalIndex;
        protected BinaryOpEvaluator(EvaluatorBuildContext ctx, BT retType, Evaluator left, Evaluator right) : base(retType) {
            _left = left;
            _right = right;
            _resultLocalIndex = ctx.DefineLocal("", retType);
        }
        protected BinaryOpEvaluator(EvaluatorBuildContext ctx, BT retType, Evaluator left, Evaluator right, bool local_index_required)
            : base(retType) {
            _left = left;
            _right = right;
            _resultLocalIndex = local_index_required? ctx.DefineLocal("", retType) : -1;
        }
        public Evaluator LeftEvaluator {
            get {
                return _left;
            }
        }
        public Evaluator RightEvaluator {
            get {
                return _right;
            }
        }
    }


    //BinaryOp̂P : BDoublepEvaluator
    public class BinaryOpDoubleArithEvaluator : BinaryOpEvaluator {
        public delegate double NumOp(double left, double right);
        private NumOp _operator;

        public BinaryOpDoubleArithEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Double, left, right) {
            _operator = ToOperatorDelegate(op);
            Debug.Assert(left.BT==BT.Double);
            Debug.Assert(right.BT==BT.Double);
        }

        public override BV Eval(EvalContext context) {
            BV left = _left.Eval(context);
            if(left.IsNil) return _retType.Nil;

            BV right = _right.Eval(context);
            if(right.IsNil) return _retType.Nil;

            double v = _operator(((BDouble)left).Value, ((BDouble)right).Value);
            if(Double.IsNaN(v))
                return BT.Double.Nil;
            else {
                BDouble result = (BDouble)context.Stack.GetAt(_resultLocalIndex);
                result.Value = v;
                return result;
            }
        }

        //Z
        public static double AddOp(double left, double right) {
            return left + right;
        }
        public static double SubOp(double left, double right) {
            return left - right;
        }
        public static double MulOp(double left, double right) {
            return left * right;
        }
        public static double DivOp(double left, double right) {
            return right==0? Double.NaN : (left / right);
        }
        public static NumOp ToOperatorDelegate(BinaryOperator op) {
            switch(op) {
                case BinaryOperator.Plus:
                    return new NumOp(AddOp);
                case BinaryOperator.Minus:
                    return new NumOp(SubOp);
                case BinaryOperator.Multiply:
                    return new NumOp(MulOp);
                case BinaryOperator.Divide:
                    return new NumOp(DivOp);
                default:
                    throw new ScriptException("lɂ͎lẐ݂ł");
            }
        }
    }

    //BinaryOp̂Q : BIntpEvaluator
    public class BinaryOpIntArithEvaluator : BinaryOpEvaluator {
        public delegate int NumOp(int left, int right);
        private BinaryOperator _enumOp;
        private NumOp _operator;

        public BinaryOpIntArithEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Int, left, right) {
            _enumOp = op;
            _operator = ToOperatorDelegate(op);
        }
        public override BV Eval(EvalContext context) {
            BV left = _left.Eval(context);
            if(left.IsNil) return _retType.Nil;

            BV right = _right.Eval(context);
            if(right.IsNil) return _retType.Nil;

            int raw_left  = ((BInt)left).Value;
            int raw_right = ((BInt)right).Value;
            if(raw_right==0 && (_enumOp==BinaryOperator.Divide || _enumOp==BinaryOperator.Mod))
                return _retType.Nil;
            else {
                BInt result = (BInt)context.Stack.GetAt(_resultLocalIndex);
                result.Value = _operator(raw_left, raw_right);
                return result;
            }
        }

        //Z
        public static int AddOp(int left, int right) {
            return left + right;
        }
        public static int SubOp(int left, int right) {
            return left - right;
        }
        public static int MulOp(int left, int right) {
            return left * right;
        }
        public static int DivOp(int left, int right) {
            return left / right;
        }
        public static int ModOp(int left, int right) {
            return left % right;
        }
        public static NumOp ToOperatorDelegate(BinaryOperator op) {
            switch(op) {
                case BinaryOperator.Plus:
                    return new NumOp(AddOp);
                case BinaryOperator.Minus:
                    return new NumOp(SubOp);
                case BinaryOperator.Multiply:
                    return new NumOp(MulOp);
                case BinaryOperator.Divide:
                    return new NumOp(DivOp);
                case BinaryOperator.Mod:
                    return new NumOp(ModOp);
                default:
                    throw new ScriptException("lɂ͎lẐ݂ł");
            }
        }
    }

    //BinaryOp̂R : BBooleanpEvaluator
    public class BinaryOpBoolEvaluator : BinaryOpEvaluator {
        //boolZ̓V[gJbĝŃfQ[gȂ
        private BinaryOperator _operator;
        public BinaryOpBoolEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Bool, left, right, false) {
            _operator = op;
        }

        //BBooleanւ̎QƂŏɎĂ܂AƂ̂͂ǂ낤BEvalƂɕԂIuWFNgςĂ͂ȂȂƂ͂A_operatorĂԂƂ̃LXgsvɂȂ
        public override BV Eval(EvalContext context) {
            BV left = _left.Eval(context);
            if(left.IsNil) return _retType.Nil;
            bool x = ((BBoolean)left).Value;

            switch(_operator) {
                case BinaryOperator.And:
                    if(!x) return left; //ӂfalse
                    break;
                case BinaryOperator.Or:
                    if(x) return left; //true
                    break;
                case BinaryOperator.Predication:
                    if(!x) return new BBoolean(true); //[ᔽPred͋HA
                    break;
            }

            //L`FbNzright̒lŜ̒l
            return _right.Eval(context);
        }

    }

    //BinaryOp̂S : BDoubleprEvaluator
    public class BinaryOpDoubleCompEvaluator : BinaryOpEvaluator {
        public delegate bool CompOp(double left, double right);
        private CompOp _operator;

        public BinaryOpDoubleCompEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Bool, left, right) {
            _operator = ToOperatorDelegate(op);
        }

        public override BV Eval(EvalContext context) {
            BV left = _left.Eval(context);
            if(left.IsNil) return _retType.Nil;

            BV right = _right.Eval(context);
            if(right.IsNil) return _retType.Nil;

            BBoolean result = (BBoolean)context.Stack.GetAt(_resultLocalIndex);
            result.Value = _operator(((BDouble)left).Value, ((BDouble)right).Value);
            return result;
        }

        //Z
        public static bool GreaterOp(double left, double right) {
            return left > right;
        }
        public static bool GreaterEqOp(double left, double right) {
            return left >= right;
        }
        public static bool SmallerOp(double left, double right) {
            return left < right;
        }
        public static bool SmallerEqOp(double left, double right) {
            return left <= right; 
        }
        public static bool EqOp(double left, double right) {
            return left == right; 
        }
        public static bool NotEqOp(double left, double right) {
            return left != right;
        }
        public static CompOp ToOperatorDelegate(BinaryOperator op) {
            switch(op) {
                case BinaryOperator.Greater:
                    return new CompOp(GreaterOp);
                case BinaryOperator.GreaterEq:
                    return new CompOp(GreaterEqOp);
                case BinaryOperator.Smaller:
                    return new CompOp(SmallerOp);
                case BinaryOperator.SmallerEq:
                    return new CompOp(SmallerEqOp);
                case BinaryOperator.Eq:
                    return new CompOp(EqOp);
                case BinaryOperator.NotEq:
                    return new CompOp(NotEqOp);
                default:
                    throw new ScriptException("rZqG[");
            }
        }
    }
    //BinaryOp̂T : BIntprEvaluator
    public class BinaryOpIntCompEvaluator : BinaryOpEvaluator {
        public delegate bool CompOp(int left, int right);
        private CompOp _operator;

        public BinaryOpIntCompEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Bool, left, right) {
            _operator = ToOperatorDelegate(op);
        }

        public override BV Eval(EvalContext context) {
            BV left = _left.Eval(context);
            if(left.IsNil) return _retType.Nil;

            BV right = _right.Eval(context);
            if(right.IsNil) return _retType.Nil;

            BBoolean v = (BBoolean)context.Stack.GetAt(_resultLocalIndex);
            v.Value = _operator(((BInt)left).Value, ((BInt)right).Value);
            return v;
        }

        //Z
        public static bool GreaterOp(int left, int right) {
            return left > right;
        }
        public static bool GreaterEqOp(int left, int right) {
            return left >= right;
        }
        public static bool SmallerOp(int left, int right) {
            return left < right;
        }
        public static bool SmallerEqOp(int left, int right) {
            return left <= right;
        }
        public static bool EqOp(int left, int right) {
            return left == right;
        }
        public static bool NotEqOp(int left, int right) {
            return left != right;
        }
        public static CompOp ToOperatorDelegate(BinaryOperator op) {
            switch(op) {
                case BinaryOperator.Greater:
                    return new CompOp(GreaterOp);
                case BinaryOperator.GreaterEq:
                    return new CompOp(GreaterEqOp);
                case BinaryOperator.Smaller:
                    return new CompOp(SmallerOp);
                case BinaryOperator.SmallerEq:
                    return new CompOp(SmallerEqOp);
                case BinaryOperator.Eq:
                    return new CompOp(EqOp);
                case BinaryOperator.NotEq:
                    return new CompOp(NotEqOp);
                default:
                    throw new ScriptException("rZqG[");
            }
        }
    }
    //BinaryOp̂UAV@int[]܂double[]̔z񉉎ZBZEẐ
    public abstract class BinaryOpNumberArrayArithEvaluator : BinaryOpEvaluator {
        protected BArrayT _resultType;
        public BinaryOpNumberArrayArithEvaluator(EvaluatorBuildContext ctx, BArrayT resultType, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, resultType, left, right) {
            _resultType = resultType;
        }
        private void AdjustResult(BRegularArray result, int len) {
            if(result.Count!=len)
                result.Extend(len);
        }
        public override BV Eval(EvalContext context) {
            BArray la = (BArray)_left.Eval(context);
            BArray ra = (BArray)_right.Eval(context);
            if(la.IsNil || ra.IsNil) {
                return _resultType.Nil;
            }


            int len = la.Count;
            if(len!=ra.Count)
                throw new ScriptException(String.Format("array length mismatch: {0} / {1}", la.Count, ra.Count));
            BRegularArray result = (BRegularArray)context.Stack.GetAt(_resultLocalIndex);
            AdjustResult(result, len);
            Debug.Assert(la.BArrayT.ElementType==_resultType.ElementType);
            for(int i=0; i<len; i++) {
                ArithBody(i, la, ra, result);
            }
            return result;
        }
        protected abstract void ArithBody(int index, BArray left, BArray right, BRegularArray result);
    }
    public class BinaryOpIntArrayArithEvaluator : BinaryOpNumberArrayArithEvaluator {
        private BinaryOpIntArithEvaluator.NumOp _numOp;
        public BinaryOpIntArrayArithEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Int.ArrayType(), left, right, op) {
            _numOp = BinaryOpIntArithEvaluator.ToOperatorDelegate(op);
        }
        protected override void ArithBody(int index, BArray left, BArray right, BRegularArray result) {
            BInt e = (BInt)result[index];
            e.Value = _numOp(((BInt)left[index]).Value, ((BInt)right[index]).Value);
        }
    }
    public class BinaryOpDoubleArrayArithEvaluator : BinaryOpNumberArrayArithEvaluator {
        private BinaryOpDoubleArithEvaluator.NumOp _numOp;
        public BinaryOpDoubleArrayArithEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Double.ArrayType(), left, right, op) {
            _numOp = BinaryOpDoubleArithEvaluator.ToOperatorDelegate(op);
            BArrayT lat = left.BT.AsArray();
            BArrayT rat = right.BT.AsArray();
            Debug.Assert(lat.ElementType==BT.Double);
            Debug.Assert(rat.ElementType==BT.Double);
        }
        protected override void ArithBody(int index, BArray left, BArray right, BRegularArray result) {
            BDouble e = (BDouble)result[index];
            e.Value = _numOp(((BDouble)left[index]).Value, ((BDouble)right[index]).Value);
        }
    }
    //BinaryOp̂W : 
    public class StringCompareEvaluator : BinaryOpEvaluator {
        public delegate bool CompOp(string left, string right);

        private CompOp _operator;
        public StringCompareEvaluator(EvaluatorBuildContext ctx, Evaluator left, Evaluator right, BinaryOperator op)
            : base(ctx, BT.Bool, left, right, true) {
            _operator = ToOperatorDelegate(op);
        }

        public override BV Eval(EvalContext context) {
            BV left = _left.Eval(context);
            if(left.IsNil) return _retType.Nil;

            BV right = _right.Eval(context);
            if(right.IsNil) return _retType.Nil;

            BBoolean result = (BBoolean)context.Stack.GetAt(_resultLocalIndex);
            result.Value = _operator(((BString)left).Value, ((BString)right).Value);
            return result;
        }

        //Z
        public static bool EqOp(string left, string right) {
            return left == right;
        }
        public static bool NotEqOp(string left, string right) {
            return left != right;
        }
        public static CompOp ToOperatorDelegate(BinaryOperator op) {
            switch(op) {
                case BinaryOperator.Eq:
                    return new CompOp(EqOp);
                case BinaryOperator.NotEq:
                    return new CompOp(NotEqOp);
                default:
                    throw new ScriptException("rZqG[");
            }
        }

    }

    public abstract class UnaryOpEvaluator : Evaluator {
        protected Evaluator _content;
        protected int _resultIndex;

        public UnaryOpEvaluator(EvaluatorBuildContext ctx, Evaluator content, BT type)
            : base(type) {
            _resultIndex = ctx.DefineLocal("", type);
            _content = content;
            
        }
    }

    public class InverseEvaluator : UnaryOpEvaluator {
        public InverseEvaluator(EvaluatorBuildContext ctx, Evaluator content, BT type)
            : base(ctx, content, type) {
        }
        public override BV Eval(EvalContext context) {
            BV t = _content.Eval(context);
            BV result = context.Stack.GetAt(_resultIndex);
            if(t.IsNil)
                return t;
            else if(t is BDouble)
                ((BDouble)result).Value = - ((BDouble)t).Value;
            else
                ((BInt)result).Value = -((BInt)t).Value;

            return result;
        }
    }
    public class LogicalNotEvaluator : UnaryOpEvaluator {
        public LogicalNotEvaluator(EvaluatorBuildContext ctx, Evaluator content, BT type)
            : base(ctx, content, type) {
        }
        public override BV Eval(EvalContext context) {
            BV t = _content.Eval(context);
            BV result = context.Stack.GetAt(_resultIndex);
            if(t.IsNil)
                return t;
            else
                ((BBoolean)result).Value = !((BBoolean)t).Value;

            return result;
        }
    }

    public class ReferenceEvaluator : Evaluator {
        private Evaluator _left;
        private Evaluator _right;

        public ReferenceEvaluator(Evaluator left, Evaluator right) : base(right.BT) {
            _left = left;
            _right = right;
        }

        public override BV Eval(EvalContext context) {
            BV l = _left.Eval(context);
            if(l.IsNil) return _retType.Nil;

            BV t = context.ContextValue; //left̃ReLXĝƂright]
            context.ContextValue = l;
            BV r = _right.Eval(context);
            context.ContextValue = t;
            return r;
        }
    }

    public class InvokeEvaluator : Evaluator {
        private Evaluator _function;
        private Evaluator[] _argEvaluators;
        private int _argumentMemoryIndex;
        private int _resultIndex;

        public InvokeEvaluator(EvaluatorBuildContext ctx, Evaluator function, Evaluator[] args) : base(ToReturnType(function)) {
            _function = function;
            _argEvaluators = args;
            _argumentMemoryIndex = (_argEvaluators==null ||_argEvaluators.Length==0)? -1 : ctx.AllocateInvokeArgMemory(_argEvaluators.Length);
            _resultIndex = ctx.DefineLocal("", _retType);
        }
        private static BT ToReturnType(Evaluator e) {
            BFunctionT ft = e.BT as BFunctionT;
            Debug.Assert(ft!=null);
            return ft.ReturnType;
        }
        public override BV Eval(EvalContext context) {
            BV target = context.ContextValue;
            if(target!=null && target.IsNil) return _retType.Nil;

            BV[] args = _argumentMemoryIndex==-1? null : context.Stack.GetInvokeArgumentBuffer(_argumentMemoryIndex);
            if(args!=null) {
                for(int i=0; i<args.Length; i++) {
                    BV t = _argEvaluators[i].Eval(context);
                    if(t.IsNil) return _retType.Nil;
                    args[i] = t;
                }
            }

            BFunction body = _function.Eval(context) as BFunction;
            Debug.Assert(body!=null); //RXgN^Ō^ׂ͒Ă
            BV result = context.Stack.GetAt(_resultIndex);
            ExecResult r = body.Execute(target, context, args, result);
            if(r==ExecResult.Nil)
                return _retType.Nil;
            else
                return result;
        }
    }


    public class ArrayEvaluator : Evaluator {
        private Evaluator _array;
        private Evaluator _index;

        public ArrayEvaluator(Evaluator array, Evaluator index) : base(array.BT.AsArray().ElementType) {
            _array = array;
            _index = index;
        }

        public override BV Eval(EvalContext context) {
            BV arr = _array.Eval(context);
            if(arr.IsNil) return _retType.Nil;

            BV index = _index.Eval(context);
            if(index.IsNil) return _retType.Nil;

            return arr.AsArray()[((BInt)index).Value];
        }
    }

    public class LetEvaluator : Evaluator {
        private int[] _indices;
        private Evaluator[] _localEvals;
        private Evaluator _body;

        public LetEvaluator(EvaluatorBuildContext ctx, int[] indices, Evaluator[] evals, Evaluator body) : base(body.BT) {
            _indices = indices;
            _localEvals = evals;
            _body = body;
        }

        public override BV Eval(EvalContext context) {
            //[Jϐ]
            for(int i=0; i<_indices.Length; i++) {
                BV v = _localEvals[i].Eval(context);
                if(v.IsNil)
                    return _body.BT.Nil;
                BV l = context.Stack.GetAt(_indices[i]);
                l.Let(v);
            }
            //{
            return _body.Eval(context);
        }

    }

    public class LambdaEvaluator : Evaluator {
        private BMethod _method;
        private BMethodValue _value;

        //{́u֐̌^vCX^VG[gׂA蔲B
        //OϐȂ֐̏ꍇiĂ͂jAB̃CX^X_valuegB
        //łȂꍇEval̂ǊOReLXgŕ]lLet

        public LambdaEvaluator(EvaluatorBuildContext ctx, BMethod m)
            : base(m.FunctionT) {
            _method = m;
            _value = new BMethodValue(m);
        }
        public override BV Eval(EvalContext context) {
            BMethod.ExternalBoundValueTag[] external_bounds = _method.ExternalContextValues;
            if(external_bounds!=null && external_bounds.Length>0) {
                BMethodValue b = new BMethodValue(_method);
                for(int i=0; i<external_bounds.Length; i++) {
                    //OReLXgŕ]lݒ肷
                    BV q = external_bounds[i].externalVar.GetValue(context);
                    b.ExternalBoundValues[i].Let(q);
                }
                return b;
            }
            else
                return _value;
        }
    }

    public class SystemFunctionEvaluator : Evaluator {
        private IBSystemFunction _function;
        private Evaluator _targetEvaluator;
        private Evaluator[] _argEvaluators;
        private int _resultIndex;
        public SystemFunctionEvaluator(EvaluatorBuildContext ctx, IBSystemFunction function, Evaluator target_evaluator, BSystemFunctionArg args)
            : base(args.retType) {
            _function = function;
            _targetEvaluator = target_evaluator;
            _argEvaluators = args.arg_evaluators;
            _resultIndex = ctx.DefineLocal("", args.retType);
        }
        public override BV Eval(EvalContext context) {
            BV target = context.ContextValue;
            if(target!=null && target.IsNil) return _retType.Nil;

            if(_targetEvaluator!=null) {
                BV t = _targetEvaluator.Eval(context);
                if(t.IsNil) return _retType.Nil;
                context.ContextValue = t;
            }

            BV result = context.Stack.GetAt(_resultIndex);
            ExecResult r = _function.Execute(context, _argEvaluators, result);
            context.ContextValue = target;
            if(r==ExecResult.Nil)
                return _retType.Nil;
            else
                return result;
        }
    }

    public class NamedValueRefEvaluator : Evaluator {
        private INamedValue _value;
        public NamedValueRefEvaluator(INamedValue v) : base(v.BT) {
            _value = v;
        }
        public override BV Eval(EvalContext context) {
            return _value.GetValue(context);
        }
    }
    //NamedValueAňɌ肷^CvEvaluator
    public class MultipleNamedValueRefEvaluator : Evaluator {
        private INamedValue[] _values;
        private int _collectIndex;
        public MultipleNamedValueRefEvaluator(INamedValue[] values)
            : base(values[0].BT) {
            _values = values;
            _collectIndex = -1;
        }
        public int Count {
            get {
                return _values.Length;
            }
        }
        public INamedValue GetAt(int index) {
            return _values[index];
        }
        public int CollectIndex {
            get {
                return _collectIndex;
            }
            set {
                _collectIndex = value;
                _retType = _values[value].BT;
            }
        }
        public override BV Eval(EvalContext context) {
            return _values[_collectIndex].GetValue(context);
        }
    }

    //'this'QƁB{INaedValue.GetValueEvalContextׂȂ
    public class ThisEvaluator : Evaluator {
        public ThisEvaluator(BT type)
            : base(type) {
        }
        public override BV Eval(EvalContext context) {
            return context.ContextValue;
        }
    }



    internal static class EvaluatorUtil {
        //悭
        public static Evaluator CreateI2D(EvaluatorBuildContext ctx, Evaluator content) {
            Debug.Assert(content.BT==BT.Int);
            //int -> doubleϊ
            return new TypeConversionEvaluator<BInt, BDouble>(ctx, BT.Double, content,
                delegate(BInt val, BDouble dest) { dest.Value = val.Value; });
        }
    }

    //letŒ`邩֐̉̃[Jϐ
    public class LocalVariable : INamedValue {
        private string _name;
        private BT _type;
        private int _index;

        public LocalVariable(string name, BT type, int index) {
            Debug.Assert(type!=null);
            _name = name;
            _type = type;
            _index = index;
        }
        
        public BT BT {
            get {
                return _type;
            }
        }

        public string Name {
            get {
                return _name;
            }
        }

        public int Index {
            get {
                return _index;
            }
        }

        public BV GetValue(EvalContext ctx) {
            return ctx.Stack.GetAt(_index);
        }

        private static LocalVariable[] _emptyArray = new LocalVariable[0];
        public static LocalVariable[] EmptyArray {
            get {
                return _emptyArray;
            }
        }
    }

}
