/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Chart/ChartControl.cs#14 $
 * $DateTime: 2008/04/17 18:21:41 $
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Drawing;
using System.Diagnostics;

using Travis;
using Travis.LocationOperation;
using Travis.Forms;
using Poderosa;

using Bellagio.Drawing;
using Bellagio.Data;
using Bellagio.Forms;
using Bellagio.Environment;
using Bellagio.Values;

namespace Bellagio.Chart {
    //ȂǁAŗL\NX
    public class StockInfoViewerControl : Control {
        protected BellagioDocumentBase _documentBase;

        protected static WaitingMark _waitingMark = new WaitingMark(); //Ŝŋ

        public StockInfoViewerControl(BellagioDocumentBase doc) {
            _documentBase = doc;
            base.SetStyle(ControlStyles.Selectable, false);
        }

        //hbOhbv
        protected int _draggingPrice; //hbOĂƂ̎wl 
        protected Point _dragStartPoint; //hbOJn_W
        public delegate void DragPriceHandler(int price);
        public event DragPriceHandler StartDragPrice;
        protected void FireStartDragPrice(int price) {
            if(this.StartDragPrice!=null) this.StartDragPrice(price);
        }        

        public StockInfoViewerControl() {
            _draggingPrice = -1;
        }

        //̂ƂAу[ĥƂnh nhłtrueԂ
        protected bool PaintIncompleteCases(PaintEventArgs args) {
            Graphics g = args.Graphics;

            switch(_documentBase.DocumentStatus) {
                case BellagioDocumentStatus.Complete:
                    return false;
                case BellagioDocumentStatus.Loading: {
                        int s = _waitingMark.Radius * 2;
                        if(args.ClipRectangle.IntersectsWith(DrawingUtil.CenterPointToRect(DrawingUtil.RectCenterPoint(this.ClientRectangle), s, s))) {
                            using(HighQualityDrawingSupport dr = new HighQualityDrawingSupport(g)) {
                                _waitingMark.Draw(g, DrawingUtil.RectCenterPoint(this.ClientRectangle), BellagioPlugin.Instance.SansaPlatform.getRootTimer().getTick());
                            }
                        }
                        return true;
                    }
                default:
                    DrawStringUtil.MiddleCenter(g, _documentBase.DocumentStatusMessage, this.Font, SystemBrushes.Window, this.ClientSize.Width/2, this.ClientSize.Height/2);
                    return true;
            }
        }

        public virtual void SetDisconnectedStatus() {
            _documentBase.DocumentStatusMessage = "f[^擾ł܂";
            Invalidate();
        }

        public virtual void PrepareResume() {
        }
    }

    //ƗRg[ł̃`[g
    public class ChartControl : StockInfoViewerControl {

        public interface IChartControlCustomize {
            void AdjustRectangleOperator(RectangleOperatorBuilder bld);
            string FormatHeaderText(Stock stock);
            void CustomPaint(PaintEventArgs args);
        }

        private IChartControlCustomize _customize;
        private ChartDocumentBase _document;
        private ChartDrawing _chartDrawing;
        private ChartDrawingSettings _chartDrawingSettings;
        private TravisScrollBar _scrollBar;
        private RectOp _rectangleOperator;
        private int _lastTimeAndSalesCount;


        public ChartControl(ChartDocumentBase document, ChartDrawingSettings settings, IChartControlCustomize customize) : base(document) {
            this.SetStyle(ControlStyles.AllPaintingInWmPaint|ControlStyles.OptimizedDoubleBuffer, true);

            _customize = customize;
            //this.BorderStyle = BorderStyle.Fixed3D;
            this.BackColor = Color.Black;
            this.ForeColor = Color.White;

            _document = document;
            _document.NotifyDelegate = document is StaticChartDocument? new DataThreadToMainThread(StaticDataHandler) : new DataThreadToMainThread(RealTimeDataHandler);

            _lastTimeAndSalesCount = 0;
            _chartDrawingSettings = settings;

            _chartDrawing = new ChartDrawing(this, BellagioPlugin.Instance.SansaPlatform, _chartDrawingSettings);
            BellagioPlugin.Instance.SansaPlatform.AssureStart();
            _chartDrawing.SetChartDataSource(document);

            _scrollBar = new TravisHScrollBar();
            _scrollBar.ScrollValueUIChanged += new EventHandler(OnScrollValueUIChanged);

            this.Controls.Add(_scrollBar.AsControl);

            //ʒu
            RectangleOperatorBuilder bld = new RectangleOperatorBuilder();
            bld.Cut(SplitDir.Bottom, _scrollBar.AsControl.Height);
            if(_customize!=null) _customize.AdjustRectangleOperator(bld);
            _rectangleOperator = bld.Append(_chartDrawing.RectOp).Finish();
        }

        public ChartDocumentBase Document {
            get {
                return _document;
            }
        }
        public ChartDrawing ChartDrawing {
            get {
                return _chartDrawing;
            }
        }
        public ChartDrawingSettings ChartDrawingSettings {
            get {
                return _chartDrawingSettings;
            }
        }

        protected override void Dispose(bool disposing) {
            base.Dispose(disposing);
            if(disposing) {
                _chartDrawing.Dispose();
                _document.SetCurrentStock(null);
            }
        }
        protected override void OnHandleCreated(EventArgs e) {
            base.OnHandleCreated(e);
            _document.NotifyDelegate(); //ڌďo
        }

        private void OnScrollValueUIChanged(object sender, EventArgs args) {
            int new_latest_index = _scrollBar.ScrollPosition + _scrollBar.ViewSize - 1;
            _chartDrawing.DisplayPosition = _document.SectionInfo.GetChartPosition(new_latest_index);
            if(this.IsHandleCreated)
                _chartDrawing.DataUpdateDirect();

        }

        public void SetCurrentStock(Stock stock) {
            if(stock==_document.CurrentStock) return;

            _document.SetCurrentStock(stock);
            _lastTimeAndSalesCount = 0;
            _scrollBar.SetValueToMaximum();
            if(stock!=null)
                _chartDrawing.DataUpdateDirect();
            else
                this.Invalidate();
        }
        public override void SetDisconnectedStatus() {
            _document.SetCurrentStock(null);
            base.SetDisconnectedStatus();
            _scrollBar.AsControl.Enabled = false;
        }
        public override void PrepareResume() {
            _scrollBar.AsControl.Enabled = false;
            RealTimeChartDocument rtd = _document as RealTimeChartDocument;
            if(rtd!=null && rtd.DocumentStatus==BellagioDocumentStatus.Complete) {
                rtd.SuspendStock();
                BellagioRoot.DataSubscriberManager.PrepareResume(rtd);
            }
            base.PrepareResume();
            this.Invalidate();
        }

        public void ReplaceDocument(ChartDocumentBase document) {
            ChartDocumentBase prev = _document;
            _document = document;
            _documentBase = document;

            prev.NotifyDelegate = null;
            Stock stock = prev.Stock;

            _document.NotifyDelegate = _document is StaticChartDocument? new DataThreadToMainThread(StaticDataHandler) : new DataThreadToMainThread(RealTimeDataHandler);
            _chartDrawing.SetChartDataSource(_document);

            this.SetCurrentStock(stock);
            prev.SetCurrentStock(null);
        }

        private void StaticDataHandler() {
            _scrollBar.SetRange(_chartDrawing.GetMaxCandleCount(), _document.SectionInfo.GetDisplayLength());
            _scrollBar.SetValueToMaximum();
            Invalidate();
        }
        //A^CpB AlertDetectedFormڌĂł邪
        public void RealTimeDataHandler() {
            try {
                if(!this.IsHandleCreated || !this.Visible) return;
                if(_document.DocumentStatus!=BellagioDocumentStatus.Complete) {
                    this.Invalidate();
                    _scrollBar.SetRange(1,0);
                    return;
                }

                //TODO ̏RealTimeChartViewContentƂقړ
                IntraDayTradeBasedChartDocument qd = (IntraDayTradeBasedChartDocument)_document;
                TimeAndSales ts = qd.IntraDayTrade.TimeAndSales;
                if(ts.Count >_lastTimeAndSalesCount) {
                    _chartDrawing.BeginDataUpdate();
                    ChartScaleInfo.PriceMoveEffectT me = _chartDrawing.ChartScaleInfo.PriceMoveEffect;
                    if(BellagioRoot.FixedPreferences.Chart.EnableChartAnimation) {
                        me.available = true;
                        me.lastPrice = ts.Count>=1? ts.LastTick.Price : 0;
                        me.semiLastPrice = ts.Count>=2? ts.TickAt(ts.Count-2).Price : me.lastPrice; //PȂΓ̒lg
                        me.isUptick = ts.PreDifferentPrice < ts.LastPrice;
                    }
                    else
                        me.available = false;

                    _lastTimeAndSalesCount = ts.Count;

                    _scrollBar.SetValueToMaximum(); //{͂ꂾƋEXN[ɂȂĂ܂
                    _chartDrawing.EndDataUpdate();
                }
                else if(ts.Count==0) { //DynamicResetƂɑΉ
                    qd.ReloadQuote();
                    _scrollBar.SetValueToMaximum();
                    _chartDrawing.DataUpdateDirect();
                }

                _lastTimeAndSalesCount = ts.Count;
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }

        protected override void OnMouseDown(MouseEventArgs e) {
            base.OnMouseDown(e);
            if(e.Button==MouseButtons.Left) {
                _draggingPrice = _chartDrawing.GetCurrentMouseTrackingPrice();
                if(_draggingPrice!=-1)
                    _dragStartPoint = e.Location;
            }
        }
        protected override void OnMouseUp(MouseEventArgs e) {
            base.OnMouseUp(e);
            if(e.Button==MouseButtons.Left)
                _draggingPrice = -1; //Zbg
        }

        protected override void OnMouseMove(MouseEventArgs e) {
            base.OnMouseMove(e);
            if(_document.CurrentStock!=null) {
                _chartDrawing.OnMouseMoveForTrackingLine(e.X, e.Y);
            }
            if(e.Button==MouseButtons.Left && _draggingPrice!=-1 &&
                Math.Abs(e.X-_dragStartPoint.X)+Math.Abs(e.Y-_dragStartPoint.Y) < 6)
                FireStartDragPrice(_draggingPrice);
        }
        protected override void OnMouseLeave(EventArgs e) {
            base.OnMouseLeave(e);
            if(_document.CurrentStock!=null) {
                _chartDrawing.OnMouseMoveForTrackingLine(-1, -1);
            }
        }

        protected override void OnResize(EventArgs e) {
            base.OnResize(e);
            Rectangle rc = this.ClientRectangle;
            _rectangleOperator.Rect(rc);
            if(_document.CurrentStock!=null && _document.DocumentStatus==BellagioDocumentStatus.Complete) {
                _scrollBar.SetRange(_chartDrawing.GetMaxCandleCount(), _document.SectionInfo.GetDisplayLength());
                if(!_chartDrawing.DisplayPosition.IsEmpty)
                    _scrollBar.SetScrollPositionByRight(_chartDrawing.DisplayPosition.ToDisplayIndex());
                _chartDrawing.DisplayPosition = _document.SectionInfo.GetChartPosition(_scrollBar.ScrollEndPosition);
                            
                _chartDrawing.DataUpdateDirect();
            }
            else
                this.Invalidate();
        }

        protected override void OnPaint(PaintEventArgs e) {
            base.OnPaint(e);
            try {
                Graphics g = e.Graphics;
                if(base.PaintIncompleteCases(e))
                    return;
                else {
                    PaintHeader(g);
                    _chartDrawing.Paint(g, e.ClipRectangle);
                    if(_customize!=null) _customize.CustomPaint(e);
                }
            }
            catch(Exception ex) {
                RuntimeUtil.ReportException(ex);
            }
        }
        private void PaintHeader(Graphics g) {
            if(_customize!=null) {
                Brush br = new SolidBrush(this.ForeColor);
                Stock st = _document.CurrentStock;
                g.DrawString(_customize.FormatHeaderText(st), this.Font, br, 10, 2);
                br.Dispose();
            }
        }

    }

}
