/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Bookmark/BookmarkItems.cs#3 $
 * $DateTime: 2008/05/14 13:05:12 $
 * 
 * ubN}[N@ڂ͖ł͂ȂAXN[jOʂȂǂ낢Ă\
 */

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;

using Poderosa;
using Poderosa.Preferences;

using Bellagio.Common;
using Bellagio.Data;
using Bellagio.Environment;

namespace Bellagio.Bookmark {
    public interface IBookmarkItem : IAdaptable {
        string FormatName();
        IAdaptable Value { get; }
        string Key { get; }
    }

    public class BookmarkFolder : IBookmarkItem {
        private List<BookmarkFolder> _folders;
        private List<IBookmarkItem> _items;
        private string _name;
        private bool _bound; //xoChłǂ
        private string _lateBindStocks; //Preferenceǂ񂾂ł͂܂GlobalStockCollectionm肵ĂȂAx邽߂܂Ƃ߂ĕŎĂƂŃZbg

        public BookmarkFolder( string name) {
            _name = name;
            _folders = new List<BookmarkFolder>();
            _items = new List<IBookmarkItem>();
        }


        public string Name {
            get {
                return _name;
            }
            set {
                _name = value;
            }
        }
        public IAdaptable Value {
            get {
                return this;
            }
        }
        public string Key {
            get {
                return _name;
            }
        }

        public string FormatName() {
            return _name;
        }

        public IList<BookmarkFolder> Folders {
            get {
                return _folders;
            }
        }
        public IList<IBookmarkItem> Items {
            get {
                return _items;
            }
        }
        public bool IsEmpty {
            get {
                return _folders.Count==0 && _items.Count==0;
            }
        }

        public string GetNewFolderCandidateName() {
            int count = 0;
            do {
                StringBuilder bld = new StringBuilder();
                bld.Append("VtH_");
                if(count > 0) bld.Append(" - ").Append(count+1);
                string name = bld.ToString();
                if(FindChildFolderByName(name)==null) return name;
                count++;
            } while(true);
        }
        public BookmarkFolder FindChildFolderByName(String name) {
            foreach(BookmarkFolder f in _folders)
                if(f.Name==name) return f;
            return null;
        }

        public void AddItem(IBookmarkItem item) {
            if(item is BookmarkFolder)
                _folders.Add((BookmarkFolder)item);
            else
                _items.Add(item);
        }
        public void RemoveItem(IBookmarkItem item) {
            if(item is BookmarkFolder)
                _folders.Remove((BookmarkFolder)item);
            else
                _items.Remove(item);
        }
        public void InsertItem(IBookmarkItem item, IBookmarkItem destination) {
            if(item is BookmarkFolder) {
                int i = destination is BookmarkFolder? _folders.IndexOf((BookmarkFolder)destination)  : -1;
                if(i!=-1)
                    _folders.Insert(i, (BookmarkFolder)item);
                else
                    _folders.Add((BookmarkFolder)item);
            }
            else {
                int i = _items.IndexOf(destination);
                if(i!=-1)
                    _items.Insert(i, item);
                else
                    _items.Add(item);
            }
        }
        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }

        //LateBind֌W
        public void SetLateBindStocks(string v) {
            if(!String.IsNullOrEmpty(v))
                _lateBindStocks = v;
            else
                _lateBindStocks = null;
            _bound = false;
        }

        public void LateBind() {
            foreach(BookmarkFolder f in _folders) f.LateBind();

            if(!_bound) {
                if(!String.IsNullOrEmpty(_lateBindStocks)) {
                    string[] codes = _lateBindStocks.Split(';');
                    foreach(string code in codes) {
                        AbstractStockProfile prof = BellagioRoot.GlobalStockCollection.FindExact(code); //Ȃ͖
                        if(prof!=null) _items.Add(new StockBookmark(prof.Primary));
                    }
                }
                _bound = true;
            }
        }

        public void SaveContentTo(StructuredText parent) {
            string stocks;
            if(_bound) {
                foreach(BookmarkFolder f in _folders) {
                    StructuredText n = new StructuredText(f.Name);
                    f.SaveContentTo(n);
                    parent.AddChild(n);
                }
                stocks = FormatStockBookmarks();
            }
            else
                stocks = _lateBindStocks;

            if(!String.IsNullOrEmpty(stocks))
                parent.Set("stocks", stocks);
                
        }
        private string FormatStockBookmarks() {
            StringBuilder b = new StringBuilder();
            foreach(IBookmarkItem item in _items) {
                Debug.Assert(item is StockBookmark); //͑ubN}[N邱Ƃ͂Ȃ
                if(b.Length>0) b.Append(';');
                b.Append(item.Key);
            }
            return b.ToString();
        }

        public static BookmarkFolder LoadFrom(StructuredText t) {
            BookmarkFolder f = new BookmarkFolder(t.Name);
            foreach(object ch in t.Children) {
                StructuredText.Entry e = ch as StructuredText.Entry;
                if(e!=null && e.name=="stocks") { //entry
                    f.SetLateBindStocks(e.value);
                }
                else { //child node
                    f.AddItem(LoadFrom((StructuredText)ch));
                }
            }
            return f;
        }
    }

    public class StockBookmark : IBookmarkItem {
        private Stock _stock;

        public StockBookmark(Stock s) {
            _stock = s;
        }

        public string FormatName() {
            return String.Format("{0} {1}", _stock.Profile.Code, _stock.Profile.Name);
        }

        public IAdaptable Value {
            get { return _stock; }
        }
        public Stock Stock {
            get {
                return _stock;
            }
        }
        public string Key {
            get {
                return _stock.Profile.Code;
            }
        }


        public IAdaptable GetAdapter(Type adapter) {
            return BUtil.DefaultGetAdapter(this, adapter);
        }
    }

    public interface IBookmarkEventListener {
        void RefreshFolder(BookmarkFolder folder);
    }

    public class BookmarkFeature : PreferenceSupplierImpl, IPreferenceLooseNodeContent {
        private BookmarkFolder _rootFolder;
        private bool _bound;
        private Listeners _listeners;

        public BookmarkFeature() : base("tactico.bookmark") {
            _rootFolder = new BookmarkFolder("ubN}[N");

            _listeners = new Listeners();
        }

        public BookmarkFolder RootFolder {
            get {
                return _rootFolder;
            }
        }
        public Listeners FireEvent {
            get {
                return _listeners;
            }
        }

        public override void InitializePreference(IPreferenceBuilder builder, IPreferenceFolder folder) {
            builder.DefineLooseNode(folder, this, "content");
        }

        public void AddEventListener(IBookmarkEventListener l) {
            _listeners.AddEventListener(l);
        }
        public void RemoveEventListener(IBookmarkEventListener l) {
            _listeners.RemoveEventListener(l);
        }


        public IPreferenceLooseNodeContent Clone() {
            return null;
        }

        public void Reset() {
        }


        public void LoadFrom(StructuredText node) {
            foreach(object ch in node.Children) {
                StructuredText.Entry e = ch as StructuredText.Entry;
                if(e!=null && e.name=="stocks") { //entry
                    _rootFolder.SetLateBindStocks(e.value); 
                }
                else { //child node
                    _rootFolder.AddItem(BookmarkFolder.LoadFrom((StructuredText)ch));
                }
            }
        }

        public void SaveTo(StructuredText node) {
            _rootFolder.SaveContentTo(node);
        }

        public void AssureLateBind() {
            if(!_bound) {
                _rootFolder.LateBind();
                _bound = true;
            }
        }

        public class Listeners : IBookmarkEventListener {
            private List<IBookmarkEventListener> _listeners;
            public Listeners() {
                _listeners = new List<IBookmarkEventListener>();
            }

            public void AddEventListener(IBookmarkEventListener l) {
                _listeners.Add(l);
            }
            public void RemoveEventListener(IBookmarkEventListener l) {
                _listeners.Remove(l);
            }

            public void RefreshFolder(BookmarkFolder folder) {
                foreach(IBookmarkEventListener l in _listeners)
                    l.RefreshFolder(folder);
            }
        }

    }

    public static class BookmarkUtil {
        public static TreeNode CreateTreeNode(BookmarkFolder folder, bool folder_only) {
            TreeNode node = new TreeNode(folder.Name);
            node.ImageIndex = ImageListFormIconIndex.FOLDER;
            node.SelectedImageIndex = node.ImageIndex;
            node.Tag = folder;
            CreateTreeNodeAndAppend(node, folder, folder_only);
            return node;
        }
        //folder̎qɑΉ̂쐬parentɂ
        public static void CreateTreeNodeAndAppend(TreeNode parent, BookmarkFolder folder, bool folder_only) {
            foreach(BookmarkFolder f in folder.Folders)
                parent.Nodes.Add(CreateTreeNode(f, folder_only));
            if(!folder_only) {
                foreach(IBookmarkItem i in folder.Items) {
                    TreeNode tn = new TreeNode(i.FormatName());
                    tn.ImageIndex = ImageListFormIconIndex.DOCUMENT;
                    tn.SelectedImageIndex = tn.ImageIndex;
                    tn.Tag = i;
                    parent.Nodes.Add(tn);
                }
            }
        }

        //l[̃Cxgnh
        public static void OnAfterLabelEdit(object sender, NodeLabelEditEventArgs args) {
            TreeNode node = args.Node;
            if(node.Parent==null) return;
            string text = args.Label==null? node.Text : args.Label;

            if(String.IsNullOrEmpty(text)) {
                BUtil.ShowWarningMessageBox("Oɂ͂ł܂");
                args.CancelEdit = true;
                node.BeginEdit();
                return;
            }

            foreach(TreeNode ch in node.Parent.Nodes) {
                if(ch!=node && ch.Text==text) {
                    BUtil.ShowWarningMessageBox("ɓ̎qvf܂");
                    args.CancelEdit = true;
                    node.BeginEdit();
                    return;
                }
            }

            //tH_l[ȂO񂪂A
            BookmarkFolder f = node.Tag as BookmarkFolder;
            if(f!=null) f.Name = text;

        }

        //xy̑cłƂtrue
        public static bool IsAncestorOf(TreeNode x, TreeNode y) {
            while(y!=null) {
                if(x==y) return true;
                y = y.Parent;
            }
            return false;
        }

        public static bool ContainsStockBookmark(BookmarkFolder folder, Stock stock) {
            foreach(IBookmarkItem item in folder.Items) {
                StockBookmark bm = item as StockBookmark;
                if(bm!=null && bm.Value==stock) return true;
            }
            return false;
        }

    }
}
