/*
 * Trading Platform "Bellagio"
 * Copyright (c) 2006, 2007  Lagarto Technology, Inc.
 * 
 * $Id: //depot/Bellagio/Demeter/Evaluators/EvaluatorBuilder.cs#7 $
 * $DateTime: 2008/01/28 15:25:25 $
 * 
 * Expressioñc[Evaluator̃c[B
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.Diagnostics;
using Bellagio.Script;
using Bellagio.Values;


namespace Bellagio.Evaluators {

    //TESTINFO EvalTests.cs
    public class EvaluatorBuilder : ExpressionVisitor<Evaluator> {
        private EvaluatorBuildContext _info;

        public EvaluatorBuilder(EvaluatorBuildContext info) {
            _info = info;
        }
        
        public Evaluator Build(Expression expr) {
            return expr.Apply<Evaluator>(this);
        }
        public EvaluatorBuildContext Context {
            get {
                return _info;
            }
        }

        public override Evaluator LiteralNumber(LiteralNumberExpression expr) {
            return LiteralEvaluator.CreateNumber(expr.Value);
        }
        public override Evaluator LiteralBoolean(LiteralBooleanExpression expr) {
            return LiteralEvaluator.CreateBool(expr.Value);
        }
        public override Evaluator LiteralTime(LiteralTimeExpression expr) {
            return LiteralEvaluator.CreateTime(expr.Value);
        }
        public override Evaluator LiteralDate(LiteralDateExpression expr) {
            return LiteralEvaluator.CreateDate(expr.Value);
        }
        public override Evaluator LiteralString(LiteralStringExpression expr) {
            return LiteralEvaluator.CreateString(expr.Value);
        }
        public override Evaluator Symbol(SymbolExpression expr) {
            //this͍ŗD
            if(expr.Symbol=="this") {
                if(_info.ContextType!=null)
                    return new ThisEvaluator(_info.ContextType);
                else
                    throw new ScriptException("ReLXg^CvȂԂthisQƂ͂ł܂");
            }

            //ʂ̃V{B[Jϐ֐̎Q
            NamedValueCollection v = _info.ResolveValue(expr.Symbol);
            Debug.Assert(v!=null); //邩null(G[ResolveValueŏo)

            if(v.single!=null)
                return new NamedValueRefEvaluator(v.single);
            else
                return new MultipleNamedValueRefEvaluator(v.multiple);
        }
        private class SystemFunctionCandidate {
            public string name;
            public Expression target_expr;
            public SystemFunctionCandidate(string n, Expression e) {
                name = n;
                target_expr = e;
            }
        }

        public override Evaluator Invoke(InvokeExpression expr) {
            BT targetType = _info.ContextType;

            //֐Kpł邩𒲂ׂ ֐targetƗEvaluatorɂ͂Ȃ肦Ȃ̂łŌ^邵Ȃ
            SystemFunctionCandidate candidate = CheckSystemFunctionInvocation(expr);
            if(candidate!=null) {
                IBSystemFunction system_function = _info.ResolveSystemFunction(candidate.name);
                if(system_function!=null) { //ArrayIteration̓P[XɃqbg̏ꍇ...
                    Evaluator target_evaluator = candidate.target_expr==null? null : candidate.target_expr.Apply<Evaluator>(this);
                    return new SystemFunctionEvaluator(_info, system_function, target_evaluator, system_function.CreateArgEvaluators(this, expr));
                }
            }

            //ʏ̃P[X
            // Evaluatorƌ^𒲂ׂ
            List<Evaluator> es = new List<Evaluator>();
            List<BT> ts = new List<BT>();
            foreach(Expression e in expr) {
                Evaluator v = e.Apply<Evaluator>(this);
                es.Add(v);
                ts.Add(v.BT);
            }
            
            Evaluator f = expr.Target.Apply<Evaluator>(this);
            //target]ʕ̌ʂAɑ΂Ĉ\ЂƂɍîȂ炻
            if(f is MultipleNamedValueRefEvaluator) {
                MultipleNamedValueRefEvaluator mf = (MultipleNamedValueRefEvaluator)f;
                for(int i=0; i<mf.Count; i++) {
                    BFunctionT ft = mf.GetAt(i).BT as BFunctionT;
                    if(ft!=null && ft.IsMatchArgs(_info.ContextType, ts.ToArray())) {
                        mf.CollectIndex = i;
                        f = new NamedValueRefEvaluator(mf.GetAt(i));
                        break;
                    }
                }
                if(mf.CollectIndex==-1)
                    throw new ScriptException(String.Format(" {0} ɑ΂Ăяoł܂", FormatExpression(expr.Target)));
            }
            else {
                BFunctionT ft = f.BT as BFunctionT;
                if(ft==null) 
                    throw new ScriptException(String.Format(" {0} ͊֐ł͂܂", FormatExpression(expr.Target)));
                if(!ft.IsMatchArgs(_info.ContextType, ts.ToArray()))
                    throw new ScriptException(String.Format(" {0} ɑ΂Ăяoň̌^قȂ܂\nExpected:{1} Actual:{2}", FormatExpression(expr.Target),ft.Name,  BT.ListNames(ts.ToArray())));
            }

            Evaluator result = new InvokeEvaluator(_info, f, es.ToArray());
            return result;
        }
        //VXe֐Ăяoł邱Ƃ`FbN@Oƌ^̃T[`Ɋ܂߂悤ɕύXĂ悢
        private SystemFunctionCandidate CheckSystemFunctionInvocation(InvokeExpression expr) {
            SymbolExpression se = expr.Target as SymbolExpression;
            if(se!=null) {
                return new SystemFunctionCandidate(se.Symbol, null);
            }

            ReferenceExpression re = expr.Target as ReferenceExpression;
            if(re!=null) {
                se = re.Right as SymbolExpression;
                if(se!=null) {
                    return new SystemFunctionCandidate(se.Symbol, re.Left);
                }
            }

            return null;
        }

        public override Evaluator Parameter(ParameterExpression expr) {
            INamedValue value = _info.GetParameter(expr.Symbol);
            if(value==null) throw new ScriptException(String.Format("p[^ {0} ܂", expr.Symbol));
            return new ParameterEvaluator(value);
        }

        public override Evaluator UnaryOp(UnaryOpExpression expr) {
            Evaluator content = expr.Content.Apply(this);
            UnaryOperator op = expr.Operator;
            BT type = content.BT;
            if(op==UnaryOperator.Inverse && IsNumberType(type))
                return new InverseEvaluator(_info, content, type);
            if(op==UnaryOperator.LogicalNot && type==BT.Bool)
                return new LogicalNotEvaluator(_info, content, type);

            throw new ScriptException(String.Format("{0}  '{1}' ŉZ邱Ƃ͂ł܂B", FormatExpression(expr.Content), expr.OperatorAsString));
        }
        public override Evaluator BinaryOp(BinaryOpExpression expr) {
            Evaluator left = expr.Left.Apply<Evaluator>(this);
            Evaluator right = expr.Right.Apply<Evaluator>(this);
            BinaryOperator op = expr.Operator;
            
            //̌^opׂĐVEvaluator\z
            BT lt = left.BT;
            BT rt = right.BT;

            if(lt==BT.Int && rt==BT.Int) {
                if(IsArithmeticOp(op))
                    return new BinaryOpIntArithEvaluator(_info, left, right, op);
                else if(IsCompareOp(op))
                    return new BinaryOpIntCompEvaluator(_info, left, right, op);
            }
            else if(IsNumberType(lt) && IsNumberType(rt)) { //int->doubleϊɒ
                if(lt==BT.Int) left = EvaluatorUtil.CreateI2D(_info, left);
                if(rt==BT.Int) right = EvaluatorUtil.CreateI2D(_info, right);
                if(IsArithmeticOp(op))
                    return new BinaryOpDoubleArithEvaluator(_info, left, right, op);
                else if(IsCompareOp(op))
                    return new BinaryOpDoubleCompEvaluator(_info, left, right, op);
            }
            else if(IsNumberArrayType(lt) && IsNumberArrayType(rt) && (op==BinaryOperator.Plus || op==BinaryOperator.Minus|| op==BinaryOperator.Multiply)) {
                if(lt.AsArray().ElementType==BT.Int && rt.AsArray().ElementType==BT.Int)
                    return new BinaryOpIntArrayArithEvaluator(_info, left, right, op);
                else
                    return new BinaryOpDoubleArrayArithEvaluator(_info, left, right, op);
            }
            else if(lt==BT.Bool && rt==BT.Bool && IsLogicalOp(op))
                return new BinaryOpBoolEvaluator(_info, left, right, op);
            else if(lt==BT.String && rt==BT.String && IsCompareOp(op))
                return new StringCompareEvaluator(_info, left, right, op);
          
            throw new ScriptException(String.Format("{0}  {1}  '{2}' ŉZ邱Ƃ͂ł܂B", FormatExpression(expr.Left), FormatExpression(expr.Right), expr.OperatorAsString));
        }
        public override Evaluator Condition(ConditionalExpression expr) {
            Evaluator cond = expr.Condition.Apply<Evaluator>(this);
            if(cond.BT!=BT.Bool)
                throw new ScriptException(String.Format("{0} bool ^łȂ΂Ȃ܂", FormatExpression(expr.Condition)));
            Evaluator tc = expr.TrueCase.Apply<Evaluator>(this);
            Evaluator fc = expr.FalseCase.Apply<Evaluator>(this);
            if(tc.BT==BT.Int && fc.BT==BT.Int) {
                return new ConditionalEvaluator(expr, cond, tc, fc);
            }
            else if(IsNumberType(tc.BT) && IsNumberType(fc.BT)) {
                if(tc.BT==BT.Int) tc = EvaluatorUtil.CreateI2D(_info, tc);
                if(fc.BT==BT.Int) fc = EvaluatorUtil.CreateI2D(_info, fc);
                return new ConditionalEvaluator(expr, cond, tc, fc);
            }
            else {
                if(!tc.BT.IsConvertibleTo(fc.BT))
                    throw new ScriptException(String.Format("{0}{1}Ɍ^̌݊܂", FormatExpression(expr.TrueCase), FormatExpression(expr.FalseCase)));
                return new ConditionalEvaluator(expr, cond, tc, fc);
            }
        }

        public override Evaluator Reference(ReferenceExpression expr) {
            Evaluator left = expr.Left.Apply<Evaluator>(this);
            BT t = _info.ContextType;
            
            _info.ContextType = left.BT;
            Evaluator right = expr.Right.Apply<Evaluator>(this);
            _info.ContextType = t;

            return new ReferenceEvaluator(left, right);
        }
        public override Evaluator Array(ArrayExpression expr) {
            Evaluator array = expr.Target.Apply<Evaluator>(this);
            if(array.BT.AsArray()==null)
                throw new ScriptException(String.Format("{0}͔zł͂܂", FormatExpression(expr.Target)));
            Evaluator index = expr.Index.Apply<Evaluator>(this);
            if(index.BT!=BT.Int)
                throw new ScriptException(String.Format("{0}͐ł͂܂", FormatExpression(expr.Index)));

            return new ArrayEvaluator(array, index);
        }
        public override Evaluator Lambda(LambdaExpression expr) {
            EvaluatorBuildContext ctx = new EvaluatorBuildContext(_info.ParameterProvider);
            ctx.Initialize(null, expr.Args);
            ctx.SetExternalValueHolders(_info); //_infoOoChƂăZbg

            Evaluator body = expr.Body.Apply<Evaluator>(new EvaluatorBuilder(ctx));
            BMethod m = new BMethod(ctx.Arguments, body);
            m.Fix(ctx);

            return new LambdaEvaluator(ctx, m);

        }
        public BMethod BuildMethod(Expression body_expr) {
            Evaluator body = body_expr.Apply<Evaluator>(this);

            return new BMethod(_info.Arguments, body);
        }

        public override Evaluator Let(LetExpression expr) {
            List<int> locals = new List<int>();
            List<Evaluator> evaluators = new List<Evaluator>();
            for(int i=0; i<expr.LocalVarCount; i++) {
                Evaluator v = expr.LocalExprAt(i).Apply<Evaluator>(this);
                evaluators.Add(v);
                int index = _info.DefineLocal(expr.LocalNameAt(i), v.BT);
                locals.Add(index);
            }

            Evaluator body = expr.Body.Apply<Evaluator>(this);

            return new LetEvaluator(_info, locals.ToArray(), evaluators.ToArray(), body);
        }


        //util
        private static bool IsArithmeticOp(BinaryOperator op) {
            return op==BinaryOperator.Plus || op==BinaryOperator.Minus || op==BinaryOperator.Multiply || op==BinaryOperator.Divide || op==BinaryOperator.Mod;
        }
        private static bool IsLogicalOp(BinaryOperator op) {
            return op==BinaryOperator.And || op==BinaryOperator.Or || op==BinaryOperator.Predication;
        }
        private static bool IsCompareOp(BinaryOperator op) {
            return op==BinaryOperator.Greater || op==BinaryOperator.GreaterEq || op==BinaryOperator.Smaller || op==BinaryOperator.SmallerEq || op==BinaryOperator.Eq || op==BinaryOperator.NotEq;
        }
        private static bool IsNumberType(BT vt) {
            return vt==BT.Int || vt==BT.Double;
        }
        private static bool IsNumberArrayType(BT vt) {
            BArrayT arr = vt.AsArray();
            return arr!=null && IsNumberType(arr.ElementType);
        }

        //G[o
        private static string FormatExpression(Expression expr) {
            ExpressionFormatter f = new ExpressionFormatter();
            return f.Format(expr);
        }
    }

}
